/*  _________________________________________________________________________
 *
 *  Coliny: A Library of COLIN optimizers
 *  Copyright (c) 2003, Sandia National Laboratories.
 *  This software is distributed under the GNU Lesser General Public License.
 *  For more information, see the README.html file in the top Coliny directory.
 *  _________________________________________________________________________
 */

/**
 * \file SolisWets.h
 *
 * Defines the coliny::SolisWets class.
 */

#ifndef coliny_SolisWets_h
#define coliny_SolisWets_h

#include <acro_config.h>
#include <utilib/default_rng.h>
#include <utilib/Normal.h>
#include <utilib/NumArray.h>
#include <utilib/Uniform.h>
#include <colin/StdOptSolver.h>
#include <colin/BatchEvaluator.h>

namespace coliny {

using colin::real;
using utilib::BasicArray;
using utilib::NumArray;

/** An implemention of the
stochastic direct search algorithms described by Solis and
Wets '81.  Algorithm 1 uses normal deviations and Algorithm 2 uses 
uniform deviations.
*/
class SolisWets :
	public colin::StdOptSolver<BasicArray<double>,
				   colin::AppResponse_Utilib>, 
	protected colin::BatchEvaluator<colin::OptProblem<BasicArray<double>,
                                 		   colin::AppResponse_Utilib>,
               			 BasicArray<double> >
{
public:

  typedef colin::BatchEvaluator<colin::OptProblem<BasicArray<double>,response_t>,
                         BasicArray<double> > batch_evaluator_t;

  ///
  SolisWets();

  ///
  void reset();

  ///
  void minimize();

  ///
  void write(std::ostream& os) const;

protected:

  ///
  void initialize_best_point()
	{
	if ((best().point.size() > 0) &&
	    (best().point.size() != problem.num_real_params()))
	   EXCEPTION_MNGR(std::runtime_error, "initialize_best_point - user-provided best point has length " << best().point.size() << " but the problem size is " << problem.num_real_params() << std::endl);
        best().point.resize(problem.num_real_params());
	}

  /// The number of consecutive successful iterations before \Ref{Delta} is expanded.
  int max_success;

  /// The number of consecutive failed iterations before \Ref{Delta} is contracted.
  int max_failure;

  /// Expansion factor for \Ref{Delta}
  double ex_factor;

  /// Contraction factor for \Ref{Delta}
  double ct_factor;

  /// Lower bound on the value of \Ref{Delta}
  double Delta_thresh;

  /// Initial value of \Ref{Delta} (set after reset is called)
  double Delta_init;

  ///
  std::string update_type;

  /// The flag that controls the types of updates for Delta
  int update_id;

  /// Flag which indicates whether the bias is used.
  bool bias_flag;

  ///
  std::string neighborhood_type;

  ///
  int neighborhood_id;

  ///
  NumArray<double> Sigma;

protected:

  ///
  void virt_debug_io(std::ostream& os, const bool finishing, const int output_level);

  ///
  void UpdateDelta(bool flag);

  ///
  virtual void gen_new_point(NumArray<double>& new_pt, 
			NumArray<double>& mean_vec, double offset,
			bool& bound_feasible);

  /// Rescale the search in each dimension
  bool auto_rescale_flag;

  /// Current step length
  double Delta;

  /// The minimum value of Delta seen so far (used for constraint penalties
  double Delta_min;

  ///
  int n_failure;

  ///
  int n_success;

  /// True if there has been one or more expansions
  bool expand_flag;

  /// A status for the current iterations
  int iteration_status;

  ///
  NumArray<double> vec1, vec2, vec3;

  ///
  NumArray<double> bias;

  ///
  utilib::Uniform unif_dev;

  ///
  utilib::Normal normal_dev;

#if 0
  ///
  void perform_evaluation(BasicArray<double>& point, SolisWets_response_t& response,
			real& val, real& cval, BasicArray<real>& cvals,
                        bool& constraint_feasible, bool& bound_feasible);
#endif

  ///
  response_t tmp_response;

};

} // namespace coliny

#endif
