/*  _________________________________________________________________________
 *
 *  Coliny: A Library of COLIN optimizers
 *  Copyright (c) 2003, Sandia National Laboratories.
 *  This software is distributed under the GNU Lesser General Public License.
 *  For more information, see the README.html file in the top Coliny directory.
 *  _________________________________________________________________________
 */

/**
 * \file RandomMOO.h
 *
 * Defines the coliny::RandomMOO class.
 */

#ifndef coliny_RandomMOO_h
#define coliny_RandomMOO_h

#include <list>
#include <acro_config.h>
#include <utilib/default_rng.h>
#include <utilib/Normal.h>
#include <colin/StdOptSolver.h>
#include <colin/BatchEvaluator.h>

namespace coliny {

using colin::real;
using utilib::BasicArray;
using utilib::Normal;
using utilib::NumArray;


/** A comparison function for assessing whether one point dominates another. */
class BasicArrayDomination
{
public:

  /// Returns true if x dominates y
  bool operator()(const BasicArray<colin::real>& x, const BasicArray<colin::real>& y)
	{
        bool eq_flag=true;
	for (unsigned int i=0; i<x.size(); i++) {
	  if (y[i] < x[i])
	     return false;
	  if (y[i] > x[i]) eq_flag=false;
          }
	if (eq_flag==true)
	   return false;
	return true;
	}
};


/** A cache for pareto-optimal solutions.
 *
 *  This will probably get moved to a more commonly useful location...
 */
template <class DomainT, class ValueT, class DominationFuncT>
class ParetoCache
{
public:

  ///
  typedef std::pair<DomainT,ValueT> point;

  ///
  ParetoCache() {}

  ///
  unsigned int size() const {return points.size();}

  ///
  void clear()
	{ points.clear(); }

  ///
  bool insert(const DomainT& x, const ValueT& values)
	{
	typename std::list<point>::iterator curr = points.begin();
	typename std::list<point>::iterator end  = points.end();
	while (curr != end) {
	  //
	  // Remove points in the cache that this point dominates
	  //
	  while ((curr != end) && (compare(values,curr->second)==true))
	    curr = points.erase(curr);
	  //
	  // If the cache point dominates this point, then return
	  //
	  //ucout << "COMPARE: " << curr->second << " / " << values << " " << compare(curr->second,values) << std::endl;
	  if (compare(curr->second,values) == true)
	     return false;
	  curr++;
	  }
	points.push_back( std::pair<DomainT,ValueT>(x,values) );
	return true;
	}

  ///
  DomainT& get_random(utilib::Uniform& urnd)
	{
	if (points.size() == 0)
	   EXCEPTION_MNGR(std::runtime_error, "ERROR - cannot call get_random with no points in the cache.");
	unsigned int i= utilib::Discretize<unsigned int>(urnd(),0,points.size()-1);
        unsigned int j=0;
	typename std::list<point>::iterator curr = points.begin();
	while (j<i) {
	  curr++;
	  j++;
	  }
	return curr->first;
	}

  ///
  void write(std::ostream& os) const
	{
	typename std::list<point>::const_iterator curr = points.begin();
	typename std::list<point>::const_iterator end  = points.end();
	while (curr != end) {
	  os << curr->first << " / " << curr->second << std::endl;
	  curr++;
	  }
	}

protected:

  ///
  DominationFuncT compare;

  ///
  std::list<point> points;

};


/** An implemention of a randomized multi-objective search method.
*/
class RandomMOO :
	public colin::StdOptSolver<BasicArray<double>,
				   colin::AppResponse_Utilib>, 
	protected colin::BatchEvaluator<colin::OptProblem<BasicArray<double>,
                                 		   colin::AppResponse_Utilib>,
               			 BasicArray<double> >
{
public:

  typedef colin::BatchEvaluator<colin::OptProblem<BasicArray<double>,response_t>, BasicArray<double> > batch_evaluator_t;

  ///
  RandomMOO();

  ///
  void reset();

  ///
  void minimize();

protected:

  ///
  bool check_convergence() {return false;}

  ///
  utilib::Normal nrnd;

  ///
  utilib::Uniform urnd;

  ///
  ParetoCache<utilib::BasicArray<double>,utilib::BasicArray<colin::real>,BasicArrayDomination> cache;

};

} // namespace coliny

template <class DomainT, class ValueT, class DominationFuncT>
std::ostream& operator<<(std::ostream& os, 
	const coliny::ParetoCache<DomainT,ValueT,DominationFuncT> & obj)
{ obj.write(os); return os; }

#endif
