/*  _________________________________________________________________________
 *
 *  COLIN: A Common Optimization Library INterface
 *  Copyright (c) 2003, Sandia National Laboratories.
 *  This software is distributed under the GNU Lesser General Public License.
 *  For more information, see the README.html file in the top COLIN directory.
 *  _________________________________________________________________________
 */

/**
 * \file Factory.h
 *
 * Macros for working with a factory for Coliny solvers.
 */

#ifndef coliny_Factory_h
#define coliny_Factory_h

#include <utilib/Factory.h>
#include <utilib/BasicArray.h>
#include <utilib/MixedIntVars.h>
#include <colin/OptSolver.h>
#include <colin/OptSolverHandle.h>
#include <colin/ColinUtilib.h>
#include <colin/OptSolverWrapperDerived.h>

///
/// Add references here to solvers that are managed with
/// the COLIN factory.
///
#ifdef ACRO_USING_COBYLA
FACTORY_REFERENCE(cobyla)
#endif
#ifdef ACRO_USING_APPSPACK
FACTORY_REFERENCE(apps)
#endif
FACTORY_REFERENCE(direct)
FACTORY_REFERENCE(DockingPS)
FACTORY_REFERENCE(ps)
#ifdef ACRO_USING_PLGO
FACTORY_REFERENCE(plgo)
#endif
FACTORY_REFERENCE(plipshitzian)
FACTORY_REFERENCE(rmoo)
FACTORY_REFERENCE(sw)
FACTORY_REFERENCE(ms)


namespace coliny {

namespace {
typedef colin::OptSolver<utilib::BasicArray<double>,colin::AppResponse_Utilib> base_t;
typedef colin::OptSolver<utilib::MixedIntVars,colin::AppResponse_Utilib> mi_base_t;
typedef colin::OptSolver<std::vector<double> > std_base_t;
}

///
colin::OptSolverBase* factory_create_base(const char* solver);

///
void get_solver_info(std::vector<std::string>& names, std::vector<std::string>& descriptions);

///
void print_factory_solvers(std::ostream& os);

template <class DomainT, class ResponseT>
colin::OptSolverHandle<DomainT,ResponseT>* solver_factory(const char* solver)
{
{
base_t* tmp = 0;
try {
tmp = utilib::factory_create<base_t*>(solver);
}
catch (std::exception&) {}
if (tmp != 0)
   return new colin::OptSolverHandle<DomainT,ResponseT>( 
		new colin::OptSolverWrapperDerived<DomainT,ResponseT,utilib::BasicArray<double>,colin::AppResponse_Utilib>(tmp,true), true
		);
}
{
mi_base_t* tmp = 0;
try {
tmp = utilib::factory_create<mi_base_t*>(solver);
}
catch (std::exception&) {}
if (tmp != 0)
   return new colin::OptSolverHandle<DomainT,ResponseT>(
		new colin::OptSolverWrapperDerived<DomainT,ResponseT,utilib::MixedIntVars,colin::AppResponse_Utilib>(tmp,true), true
		);
}
{
std_base_t* tmp = 0;
try {
tmp = utilib::factory_create<std_base_t*>(solver);
}
catch (std::exception&){}
if (tmp != 0)
   return new colin::OptSolverHandle<DomainT,ResponseT>( 
		new colin::OptSolverWrapperDerived<DomainT,ResponseT,std::vector<double>,colin::AppResponse<> >(tmp,true), true
		);
}
return 0;
}

}

#endif
