/*  _________________________________________________________________________
 *
 *  COLIN: A Common Optimization Library INterface
 *  Copyright (c) 2003, Sandia National Laboratories.
 *  This software is distributed under the GNU Lesser General Public License.
 *  For more information, see the README.html file in the top COLIN directory.
 *  _________________________________________________________________________
 */

/**
 * \file SysCallAnalysisCode.h
 *
 * Defines the colin::SysCallAnalysisCode class.
 */

#ifndef colin_SysCallAnalysisCode_h
#define colin_SysCallAnalysisCode_h

#include <acro_config.h>
#include <colin/AnalysisCode.h>
#include <colin/CommandShell.h>

namespace colin {


/** \class SysCallAnalysisCode
  *
  * Derived class in the AnalysisCode class hierarchy which spawns
  * simulations using system calls.
  *
  * \c SysCallAnalysisCode creates separate simulation processes using
  * the C system() command.  It utilizes \c CommandShell to manage shell
  * syntax and asynchronous invocations.
  */
template <class DomainT, class ResponseT, class StringT=std::string>
class SysCallAnalysisCode: public AnalysisCode<DomainT,ResponseT,StringT>
{
public:

  /// Constructor
  SysCallAnalysisCode();
  /// Destructor
  ~SysCallAnalysisCode() {}

  /** Spawn a complete function evaluation.
    * Put the SysCallAnalysisCode to the shell using either the default syntax
    * or specified commandUsage syntax.  This function is used when all portions
    * of the function evaluation (i.e., all analysis drivers) are executed on
    * the local processor.
    */
  void spawn_evaluation(bool block_flag);

  /** Spawn the input filter portion of a function evaluation.
    * Put the input filter to the shell.  This function is used when multiple
    * analysis drivers are spread between processors.  No need to check for a
    * Null input filter, as this is checked externally.  Use of nonblocking
    * shells is supported in this fn, although its use is currently prevented
    * externally.
    */
  void spawn_input_filter(bool block_flag);

  /** Spawn a single analysis as part of a function evaluation.
    * Put a single analysis to the shell using the default syntax (no
    * commandUsage support for analyses).  This function is used when
    * multiple analysis drivers are spread between processors.  Use of
    * nonblocking shells is supported in this fn, although its use is
    * currently prevented externally.
    */
  void spawn_analysis(const int& analysis_id, bool block_flag);

  /** Spawn the output filter portion of a function evaluation.
    * Put the output filter to the shell.  This function is used when multiple
    * analysis drivers are spread between processors.  No need to check for a
    * Null output filter, as this is checked externally.  Use of nonblocking
    * shells is supported in this fn, although its use is currently prevented
    * externally.
    */
  void spawn_output_filter(bool block_flag);

  /// return commandUsage
  const StringT& command_usage() const
		{return commandUsage;}
  /// return commandUsage
  StringT& command_usage()
		{return commandUsage;}

  /// optional command usage string for supporting nonstandard
  /// command syntax (supported only by SysCall analysis codes)
  StringT commandUsage;
};



//============================================================================
//
//
template <class DomainT, class ResponseT, class StringT>
SysCallAnalysisCode<DomainT,ResponseT,StringT>
	::SysCallAnalysisCode() 
{ }



//============================================================================
//
//
template <class DomainT, class ResponseT, class StringT>
void SysCallAnalysisCode<DomainT,ResponseT,StringT>
	::spawn_evaluation(bool block_flag)
{
// MSE, 11/17/99: system call file passing changed to pass both files to all
// 3 executables since: (1) in many cases, the OFilter will need to know the
// asv, parameters, etc., and (2) in asynch usage, each of the 3 pieces must
// be able to manage tagged files and/or working subdirectories.

CommandShell shell;

// Input filter portion
shell << "(";  
if (this->iFilterName.size() > 0)
   shell << this->iFilterName << " " << this->modifiedParamsFileName << " "
         << this->modifiedResultsFileName << "; ";
  
// Analysis code portion (function evaluation may be asynchronous, but
// analyses must be sequential within each function evaluation)
if (commandUsage.size() == 0) {
   std::list<std::string>::const_iterator curr = this->programNames.begin();
   std::list<std::string>::const_iterator last = this->programNames.end();
   unsigned int i=0;
   while (curr != last) {
     i++;
     shell << *curr << " " << this->modifiedParamsFileName << " "
           << this->modifiedResultsFileName;
     if (this->numPrograms > 1) { // append program cntr to modifiedResultsFileName
        char prog_num[16];
        sprintf(prog_num, ".%d", i+1);
        shell << prog_num;
        }
     if (i != this->numPrograms-1)
        shell << "; ";
     curr++;
     }
  }
else                      // if numPrograms>1, then commandUsage must
   shell << commandUsage; // include the syntax for all programs.

// Output filter portion
if (this->oFilterName.size() > 0)
   shell << "; " << this->oFilterName << " " << this->modifiedParamsFileName << " "
         << this->modifiedResultsFileName;
shell << ")";

// Process definition complete; now set the shell's asynchFlag/quietFlag from
// the incoming block_flag & the program's quietFlag and spawn the process.
shell.asynch_flag(!block_flag);
shell.quiet_flag(this->quietFlag);
shell << flush;
}


//============================================================================
//
//
template <class DomainT, class ResponseT, class StringT>
void SysCallAnalysisCode<DomainT,ResponseT,StringT>
	::spawn_input_filter(bool block_flag)
{
  CommandShell shell;
  shell << this->iFilterName << " " << this->modifiedParamsFileName << " "
        << this->modifiedResultsFileName;
  shell.asynch_flag(!block_flag);
  shell.quiet_flag(this->quietFlag);
  shell << flush;
}


//============================================================================
//
//
template <class DomainT, class ResponseT, class StringT>
void SysCallAnalysisCode<DomainT,ResponseT,StringT>::
	spawn_analysis(const int& analysis_id, bool block_flag)
{
  CommandShell shell;
  shell << this->programNames[analysis_id-1] << " " << this->modifiedParamsFileName << " "
        << this->modifiedResultsFileName;
  if (this->numPrograms > 1) { // append program cntr to modifiedResultsFileName
    char prog_num[16];
    sprintf(prog_num, ".%d", analysis_id);
    shell << prog_num;
  }
  shell.asynch_flag(!block_flag);
  shell.quiet_flag(this->quietFlag);
  shell << flush;
}


//============================================================================
//
//
template <class DomainT, class ResponseT, class StringT>
void SysCallAnalysisCode<DomainT,ResponseT,StringT>::
	spawn_output_filter(bool block_flag)
{
  CommandShell shell;
  shell << this->oFilterName << " " << this->modifiedParamsFileName << " "
        << this->modifiedResultsFileName;
  shell.asynch_flag(!block_flag);
  shell.quiet_flag(this->quietFlag);
  shell << flush;
}

}

#endif
