/*  _________________________________________________________________________
 *
 *  COLIN: A Common Optimization Library INterface
 *  Copyright (c) 2003, Sandia National Laboratories.
 *  This software is distributed under the GNU Lesser General Public License.
 *  For more information, see the README.html file in the top COLIN directory.
 *  _________________________________________________________________________
 */

/**
 * \file OptSolverWrapper.h
 *
 * Defines the colin::OptSolverWrapper class.
 */

#ifndef colin_OptSolverWrapper_h
#define colin_OptSolverWrapper_h

#include <acro_config.h>
#include <colin/OptProblem.h>
#include <colin/OptResponse.h>

namespace colin {

class OptSolverBase;


/**
 * This class is used within OptSolver to point to a generic colin::OptSolver
 * object.
 */
template <class DomainT, class ResponseT>
class OptSolverWrapper
{
public:

  /// Constructor.
  OptSolverWrapper() : rep(0) {}

  /// Destructor.
  virtual ~OptSolverWrapper()
		{if (rep) delete rep;}

  /// Set the wrapper data in the current object.
  void operator=(OptSolverWrapper<DomainT,ResponseT>* wrapper)
		{
		if (!wrapper) return;
		if (rep) delete rep;
		rep = wrapper;
		}
		
  /// Returns true if the wrapper data has been set.
  operator bool() const
		{return (rep != 0);}

  /// Set the initial point for the optimizer
  virtual void set_initial_point(DomainT& point_)
		{
		if (!rep)
		   EXCEPTION_MNGR(std::runtime_error,"OptSolverWrapper::set_initial_point - undefined rep");
		rep->set_initial_point(point_);
		}

  ///
  virtual OptProblem<DomainT,ResponseT> & get_problem()
        	{
		if (!rep)
		   EXCEPTION_MNGR(std::runtime_error,"OptSolverWrapper::get_problem - undefined rep");
		return rep->get_problem();
		}

  ///
  virtual const OptProblem<DomainT,ResponseT> & get_problem() const
        	{
		if (!rep)
		   EXCEPTION_MNGR(std::runtime_error,"OptSolverWrapper::get_problem - undefined rep");
		return rep->get_problem();
		}

  /// Set the problem in the wrapper.
  virtual void set_problem(OptProblem<DomainT,ResponseT>& prob)
		{
		if (!rep)
		   EXCEPTION_MNGR(std::runtime_error, "OptSolverWrapper::set_problem - undefined rep");
		rep->set_problem(prob);
		}

  /// Return the response of the best point seen so far
  virtual OptResponse<DomainT,ResponseT>& opt_response()
		{
		if (!rep)
		   EXCEPTION_MNGR(std::runtime_error, "OptSolverWrapper::opt_response - undefined rep");
		return rep->opt_response();
		}

  /// Return the response of the best point seen so far
  virtual void opt_response(OptResponse<DomainT,ResponseT>& response_)
		{
		if (!rep)
		   EXCEPTION_MNGR(std::runtime_error, "OptSolverWrapper::opt_response - undefined rep");
		return rep->opt_response(response_);
		}

  /// Return a pointer to the base \c OptSolver information
  virtual OptSolverBase& base()
		{
		if (!rep)
		   EXCEPTION_MNGR(std::runtime_error, "OptSolverWrapper::base - undefined rep");
		return rep->base();
		}

  /// Return a pointer to the base \c OptSolver information
  virtual OptSolverBase* operator->()
		{
		if (!rep)
		   EXCEPTION_MNGR(std::runtime_error, "OptSolverWrapper::operator-> - undefined rep");
		return (*rep).operator->();
		}

  /// Return a pointer to the base \c OptSolver information
  virtual const OptSolverBase* operator->() const
		{
		if (!rep)
		   EXCEPTION_MNGR(std::runtime_error, "OptSolverWrapper::operator-> - undefined rep");
		return (*rep).operator->();
		}

  /// Return a reference to the base \c OptSolver information
  virtual OptSolverBase& operator*()
		{
		if (!rep)
		   EXCEPTION_MNGR(std::runtime_error, "OptSolverWrapper::operator-> - undefined rep");
		return rep->base();
		}

  /// Return a pointer to the base \c OptSolver information
  virtual const OptSolverBase& operator*() const
		{
		if (!rep)
		   EXCEPTION_MNGR(std::runtime_error, "OptSolverWrapper::operator-> - undefined rep");
		return rep->base();
		}

protected:

  /// A wrapper for a COLIN solver.
  OptSolverWrapper<DomainT,ResponseT>* rep;

};

}


//============================================================================
//
//
/// Write a colin::OptSolverWrapper object.
template <class DomainT, class ResponseT>
std::ostream& operator<<(std::ostream& os, const colin::OptSolverWrapper<DomainT,ResponseT> & obj)
{ obj.write(os); return os; }

#endif
