/*  _________________________________________________________________________
 *
 *  COLIN: A Common Optimization Library INterface
 *  Copyright (c) 2003, Sandia National Laboratories.
 *  This software is distributed under the GNU Lesser General Public License.
 *  For more information, see the README.html file in the top COLIN directory.
 *  _________________________________________________________________________
 */

/**
 * \file OptResponse.h
 *
 * Defines the colin::OptResponse class.
 */

#ifndef __colin_OptResponse_h
#define __colin_OptResponse_h

#include <acro_config.h>
#include <utilib/enum_def.h>
#include <utilib/PackObject.h>
#include <colin/StatusCodes.h>

namespace colin {

using utilib::PackBuffer;
using utilib::UnPackBuffer;

/// A response object for optimization.
template <class DomainT, class AppResponseT=AppResponse<> >
class OptResponse : public utilib::PackObject
{
public:

  /// Constructor.
  OptResponse()
	{ values.resize(1); reset(); }

  /// Destructor.
  virtual ~OptResponse()
	{}

  /// Reset the data in this response.
  void reset()
	{
	value() = real::positive_infinity;
	constraint_violation = real::positive_infinity;
	response.reset();
        response.function_value() = real::positive_infinity;
	termination_info = "Unknown";
	model_status = model_error_unknown;
	solver_status=solver_error_solver_failure;
	}

  /// The final/best point.
  DomainT point;

  /// The application response for this point.
  AppResponseT response;

  /// The value of this point.
  real& value()
	{ return values[0]; }

  /// The value of this point.
  const real& value() const
	{ return values[0]; }

  /// The value of this point.
  typename AppResponseT::realarray_t values;

  /// The constraint violation for this point.
  real constraint_violation;

  /// Termination information for the solver.
  std::string termination_info;

  /// The status of model use.
  model_status_enum model_status;

  /// The status of solver use.
  solver_status_enum solver_status;

  /// Write this object
  void write(std::ostream& os) const;

  /// Read this object
  void read(std::istream& is);

  /// Pack this object
  void write(PackBuffer& os) const;

  /// Unpack this object
  void read(UnPackBuffer& is);

};

template <class DomainT, class AppResponseT>
void OptResponse<DomainT,AppResponseT>::write(std::ostream& os) const
{
os << point << std::endl;
os << response << std::endl;
if (values.size() > 0) {
   os << values << " " << constraint_violation << std::endl;
   }
else {
   os << value() << " " << constraint_violation << std::endl;
   }
os << termination_info << " " << model_status << " " << solver_status << std::endl;
}

template <class DomainT, class AppResponseT>
void OptResponse<DomainT,AppResponseT>::read(std::istream& is)
{
is >> point >> response >> values >> constraint_violation
	>> termination_info >> model_status >> solver_status;
}

template <class DomainT, class AppResponseT>
void OptResponse<DomainT,AppResponseT>::write(PackBuffer& os) const
{
os << point << response << values << constraint_violation
	<< termination_info;
os << model_status << solver_status;
}

template <class DomainT, class AppResponseT>
void OptResponse<DomainT,AppResponseT>::read(UnPackBuffer& is)
{
is >> point >> response >> values >> constraint_violation
	>> termination_info >> model_status >> solver_status;
}

/// Copy response information from one colin::OptResponse object to another.
template <class DomainT, class AppResponseT, class LDomainT, class LAppResponseT>
void map_OptResponse(OptResponse<DomainT,AppResponseT>& to, OptResponse<LDomainT,LAppResponseT>& from)
{
to.values = from.values;
to.constraint_violation = from.constraint_violation;
to.termination_info = from.termination_info;
to.model_status = from.model_status;
to.solver_status = from.solver_status;
map_domain(to.point, from.point);
to.response.info = from.response.info;
map_AppResponseValues(to.response.values.data(),from.response.values.data(),to.response.info.data());
}

}

/// Copy colin::AppResponse information into a colin::OptResponse object.
template <class DomainT, class AppResponseT >
inline colin:: OptResponse<DomainT,AppResponseT>& operator<<( colin::OptResponse<DomainT,AppResponseT>& x, const  AppResponseT& info)
{
x.response << info;
info.function_values( x.values );
x.constraint_violation = info.l2_constraint_violation();
return x;
}


#endif
