/*  _________________________________________________________________________
 *
 *  COLIN: A Common Optimization Library INterface
 *  Copyright (c) 2003, Sandia National Laboratories.
 *  This software is distributed under the GNU Lesser General Public License.
 *  For more information, see the README.html file in the top COLIN directory.
 *  _________________________________________________________________________
 */

/**
 * \file OptProblemFrag_IntParams.h
 *
 * Defines the colin::OptProblemFragment_IntParams class.
 */

#ifndef colin_OptProblemFrag_IntParams_h
#define colin_OptProblemFrag_IntParams_h

#include <acro_config.h>
#include <colin/OptProblemFrag_Bounds.h>

namespace colin {


/**
 *  Defines the elements of an OptProblem that pertain to integer parameters.
 */
template <class DomainT, class ResponseT>
class OptProblemFragment_IntParams : virtual public OptProblemFragment_Bounds<DomainT,ResponseT>
{
public:

  /// Returns the number of integer parameters in the domain
  unsigned int num_int_params()
		{ return (this->state ? this->state->num_int_params : 0);}

  /// Returns the number of binary parameters in the domain
  unsigned int num_binary_params()
		{ return (this->state ? this->state->num_binary_params : 0);}

  /// Initializes the number of integer params in the domain
  void init_int_params(unsigned int num)
		{
		if (!(this->state))
		   EXCEPTION_MNGR(std::runtime_error, "OptProblemFragment_IntParams::init_int_params - no application!");
		this->state->num_int_params = num;
		if(this->state->real_lower_bounds.size()==0)
		  this->state->enforcing_bound_constraints = false;
		}

  /// Initializes the number of binary params in the domain
  void init_binary_params(unsigned int num)
		{
		if (!(this->state))
		   EXCEPTION_MNGR(std::runtime_error, "OptProblemFragment_IntParams::init_binary_params - no application!");
		this->state->num_binary_params = num;
		}

  /// Sets the type of an integer lower bound
  void int_lower_bound_type(int i, bound_type_enum type)
                {
                if (this->state->enforcing_bound_constraints == false)
		   EXCEPTION_MNGR(std::runtime_error, "Trying to set a bound type when integer bounds are not used.");
                this->state->int_lower_bound_type[i] = type;
                }

  /// Sets the type of an integer upper bound
  void int_upper_bound_type(int i, bound_type_enum type)
                {
                if (this->state->enforcing_bound_constraints == false)
		   EXCEPTION_MNGR(std::runtime_error, "Trying to set a bound type when integer bounds are not used.");
                this->state->int_upper_bound_type[i] = type;
                }

  /// Returns the type of the int lower bound
  bound_type_enum int_lower_bound_type(int i) const
                {
                if (this->state->enforcing_bound_constraints == false)
                   return no_bound;
                return this->state->int_lower_bound_type[i];
                }

  /// Returns the type of the int upper bound
  bound_type_enum int_upper_bound_type(int i) const
                {
                if (this->state->enforcing_bound_constraints == false)
                   return no_bound;
                return this->state->int_upper_bound_type[i];
                }

  /// Returns true if this problem has a lower bound on the $i$-th parameter
  bool has_int_lower_bound(int i) const
                {return int_lower_bound_type(i) == no_bound;}

  /// Returns true if this problem has an upper bound on the $i$-th parameter
  bool has_int_upper_bound(int i) const
                {return int_upper_bound_type(i) == no_bound;}

  /// Set integer boundary constraints - dense format
#ifdef ACRO_HAVE_TEMPLATES_AS_TEMPLATE_ARGUMENTS
  template <template <class TYPE> class LArrayT>
  void set_int_bounds(const LArrayT<int>& lower, const LArrayT<int>& upper)
#else
  template <class ArrayT>
  void set_int_bounds(const ArrayT& lower, const ArrayT& upper)
#endif
		{
		std::vector<int> lower_,upper_;
		lower_ << lower;
		upper_ << upper;
		this->state->set_int_bounds(lower_,upper_);
		}

  /// Get integer boundary constraints - dense format
#ifdef ACRO_HAVE_TEMPLATES_AS_TEMPLATE_ARGUMENTS
  template <template <class TYPE> class LArrayT>
  void get_int_bounds(LArrayT<int>& lower, LArrayT<int>& upper)
#else
  template <class ArrayT>
  void get_int_bounds(ArrayT& lower, ArrayT& upper)
#endif
		{
		if (!(this->state->enforcing_bound_constraints)) 
                   EXCEPTION_MNGR(std::runtime_error,"Requested bounds for a problem without enforced bounds.");
		std::vector<int> lower_,upper_;
		this->state->get_int_bounds(lower_,upper_);
		lower << lower_;
		upper << upper_;
		}

  /// Returns true if the bounds on the $i$-th integer variable are periodic
  bool periodic_int_bound(int i) const
		{return (this->state->enforcing_bound_constraints) &&
			(this->state->int_lower_bound_type[i] == periodic_bound);}
};


/// A subclass of OptProblemFragment_IntParams that defines domain_int().
template <class DomainT, class ResponseT, bool IsUsed>
class OptProblemInterface_IntParams : public OptProblemFragment_IntParams<DomainT,ResponseT>
{
public:

  /// Returns information about the integer domain.
  OptProblemFragment_IntParams<DomainT,ResponseT>& domain_int()
                {return *this;}

};


/// Specialiation of OptProblemInterface_IntParams for the case where no
/// integer parameters are used.
template <class DomainT, class ResponseT>
class OptProblemInterface_IntParams<DomainT,ResponseT,false> : virtual public OptProblemFragment_Base<DomainT,ResponseT>
{
public:

  /// Returns an error.
  OptProblemFragment_IntParams<DomainT,ResponseT>& domain_int()
                {
                EXCEPTION_MNGR(std::runtime_error,"Error attempting to access the IntParams domain information.");
                return *static_cast<OptProblemFragment_IntParams<DomainT,ResponseT>*>(0);
                }

};

}

#endif
