/*  _________________________________________________________________________
 *
 *  COLIN: A Common Optimization Library INterface
 *  Copyright (c) 2003, Sandia National Laboratories.
 *  This software is distributed under the GNU Lesser General Public License.
 *  For more information, see the README.html file in the top COLIN directory.
 *  _________________________________________________________________________
 */

/**
 * \file OptProblemAppHandle.h
 *
 * Defines the colin::OptProblemAppHandle class.
 */

#ifndef colin_OptProblemAppHandle_h
#define colin_OptProblemAppHandle_h

#include <acro_config.h>
#include <utilib/stl_auxillary.h>
#include <colin/OptApplication.h>
#include <colin/OptProblemState.h>

namespace colin {


template <class DomainT, class ResponseT>
class OptProblem;

template <class DomainT, class ResponseT>
class OptProblemAppBody;

/**
 * The 'body' class for a handle-body idiom that enables colin::OptProblem
 * classes to transparently point to the same data and application interface.
 * The colin::OptProblem classes actually point to the colin::OptProblemAppBody 
 * class, which derives from this class and defines domain-specific 
 * functionality.
 */
template <class DomainT, class ResponseT>
class OptProblemAppHandle
{
public:

  /// Constructor
  OptProblemAppHandle() {}

  /// Destructor
  virtual ~OptProblemAppHandle() {}

  /// Evaluate the current point, requesting information specified by 
  /// \c asv and using the mask defined by \c mode
  virtual void Eval(std::vector<int>& asv, int mode, ResponseT& response) = 0;

  /// Evaluate the current point
  virtual void Eval(int mode, ResponseT& response) = 0;

  /// Evaluate the current point and compute the function value
  virtual void EvalF(real& value) = 0;

  /// Evaluate the current point and compute function values
  virtual void EvalF(typename ResponseT::realarray_t& values) = 0;

  /// Evaluate the current point and compute the constraint function(s)
  virtual void EvalCF(typename ResponseT::vector_t& value) = 0;

  /// Evaluate the current point and compute the gradient
  virtual void EvalG(typename ResponseT::vector_t& value) = 0;
 
  /// Evaluate the current point and compute the function and gradient
  virtual void EvalFG(real& ans, typename ResponseT::vector_t& value) = 0;
 
  /// Evaluate the current point and compute the Jacobian
  virtual void EvalCG(typename ResponseT::vectorarray_t& value) = 0;
 
  /// Asynchronously evaluate the current point
  virtual void AsyncEval(std::vector<int>& asv, int mode, int& priority, ResponseT* response, int tag=-1) = 0;

  /// Asynchronously evaluate the current point
  virtual void AsyncEval(int mode, int& priority, ResponseT* response, int tag=-1) = 0;

  /// Asynchronously evaluate the current point
  virtual void AsyncEvalF(typename ResponseT::realarray_t* values, int& priority, int tag=-1) = 0;

  /// Asynchronously evaluate the current point
  virtual void AsyncEvalF(real* value, int& priority, int tag=-1) = 0;

  /// Asynchronously evaluate the current point
  virtual void AsyncEvalCF(typename ResponseT::vector_t* value, int& priority, int tag=-1) = 0;

  /// Asynchronously evaluate the current point
  virtual void AsyncEvalG(typename ResponseT::vector_t *value, int& priority, int tag=-1) = 0;
 
  /// Asynchronously evaluate the current point
  virtual void AsyncEvalFG(real* ans, typename ResponseT::vector_t *value, int& priority, int tag=-1) = 0;
 
  /// Asynchronously evaluate the current point
  virtual void AsyncEvalCG(typename ResponseT::vectorarray_t* value, int& priority, int tag=-1) = 0;

  /// Set the current point
  virtual void set_point(const DomainT& point_) = 0;

  /// Get the current point
  virtual DomainT& get_point() = 0;

  /// Set the linear constraints
  virtual void set_linear_constraints(const typename ResponseT::matrix_t& mat, const typename ResponseT::vector_t& rhs, 
					const int num_equality) = 0;

  /// Get the linear constraint matrix
  virtual const typename ResponseT::matrix_t& linear_constraint_mat() = 0;

  /// Get the linear constraints RHS
  virtual const typename ResponseT::vector_t& linear_constraint_rhs() = 0;

  /// Get the base state object
  virtual OptProblemStateBase* state() = 0;

  /// Get the base application object
  virtual OptApplicationBase* app() = 0;

  /// The current point
  DomainT point;

  /// Set the state in \c app to be the same as in the current Handle object
  virtual void extract_state(OptProblemAppBody<DomainT,ResponseT>* app) = 0;
};


} // namespace colin

#endif
