/*  _________________________________________________________________________
 *
 *  COLIN: A Common Optimization Library INterface
 *  Copyright (c) 2003, Sandia National Laboratories.
 *  This software is distributed under the GNU Lesser General Public License.
 *  For more information, see the README.html file in the top COLIN directory.
 *  _________________________________________________________________________
 */

/**
 * \file OptProblemAppBody.h
 *
 * Defines the colin::OptProblemAppBody class.
 */

#ifndef colin_ColinProblemAppBody_h
#define colin_ColinProblemAppBody_h

#include <acro_config.h>
#include <utilib/CachedAllocator.h>
#include <utilib/SmartPtr.h>
#include <colin/OptProblemAppHandle.h>

namespace colin {

/**
 * The subclass of colin::OptProblemAppHandle that is actually accessed by
 * colin::OptProblem.
 */
template <class DomainT, class ResponseT>
class OptProblemAppBody : public OptProblemAppHandle<DomainT,ResponseT>
{
  #if !defined(DOXYGEN)
  friend class OptProblem<DomainT,ResponseT>;
  #endif

public:

  /// Constructor.
  OptProblemAppBody(
		OptProblemState<DomainT,typename ResponseT::vector_t,typename ResponseT::matrix_t>* state_=0,
  		OptApplication<DomainT,ResponseT>* app_=0)
		: State(state_), App(app_)  {}

  ///
  void Eval(std::vector<int>& asv, int mode, ResponseT& response);

  ///
  void Eval(int mode, ResponseT& response);

  ///
  void EvalF(real& value);

  ///
  void EvalF(typename ResponseT::realarray_t& values);

  ///
  void EvalCF(typename ResponseT::vector_t& value);

  ///
  void EvalG(typename ResponseT::vector_t& value);
 
  ///
  void EvalFG(real& ans, typename ResponseT::vector_t& value);
 
  ///
  void EvalCG(typename ResponseT::vectorarray_t& value);
 
  ///
  void AsyncEval(std::vector<int>& asv, int mode, int& priority, 
					ResponseT* response, int tag=-1);

  ///
  void AsyncEval(int mode, int& priority, 
					ResponseT* response, int tag=-1);

  ///
  void AsyncEvalF(real* value, int& priority, int tag=-1);

  ///
  void AsyncEvalF(typename ResponseT::realarray_t* value, int& priority, int tag=-1);

  ///
  void AsyncEvalCF(typename ResponseT::vector_t* value, int& priority, int tag=-1);

  ///
  void AsyncEvalG(typename ResponseT::vector_t *value, int& priority, int tag=-1);
 
  ///
  void AsyncEvalFG(real* ans, typename ResponseT::vector_t *value, int& priority, int tag=-1);
 
  ///
  void AsyncEvalCG(typename ResponseT::vectorarray_t* value, int& priority, int tag=-1);

  ///
  void set_point(const DomainT& point_)
		{ map_domain(point,point_); }

  ///
  DomainT& get_point()
		{ return point; }

  ///
  void set_linear_constraints(const typename ResponseT::matrix_t& mat, const typename ResponseT::vector_t& rhs, 
					const int num_equality)
		{ State->set_linear_constraints(mat,rhs,num_equality); }

  ///
  const typename ResponseT::matrix_t& linear_constraint_mat()
		{ return State->linear_constraint_mat; }

  ///
  const typename ResponseT::vector_t& linear_constraint_rhs()
		{ return State->linear_constraint_rhs; }

  ///
  OptProblemStateBase* state()
		{ return State; }

  ///
  OptApplicationBase* app()
		{ return App; }

  ///
  void extract_state(OptProblemAppBody<DomainT,ResponseT>* app_)
  		{ app_->State = State; }

  /// A pointer to the application state.
  utilib::SmartPtr<OptProblemState<DomainT,typename ResponseT::vector_t,typename ResponseT::matrix_t> > State;

  /// Pointer to the application object.
  utilib::SmartPtr<OptApplication<DomainT,ResponseT> > App;

protected:

  /// A copy of the point being evaluated.
  DomainT point;

  /// Allocate a new response object.
  ResponseT* new_response()
		{
		ResponseT* response  = utilib::CachedAllocator<ResponseT> :: allocate();
		response->resize(State->numFunctions, State->numConstraints,
			State->num_real_params, State->using_gradients,
			State->using_hessians);
		return response;
		}
};



//============================================================================
//
//
template <class DomainT, class ResponseT>
void OptProblemAppBody<DomainT,ResponseT>::
	Eval(std::vector<int>& asv, int mode, ResponseT& response)
{
int tmp=-1;
App->Eval(point,mode,asv,tmp,&response,true,tmp);
}


//============================================================================
//
//
template <class DomainT, class ResponseT>
void OptProblemAppBody<DomainT,ResponseT>::
	Eval(int mode, ResponseT& response)
{
int tmp=-1;
State->init_request_asv(mode);
App->Eval(point,mode,State->REQUEST_ASV,tmp,&response,true,tmp);
}


//============================================================================
//
//
template <class DomainT, class ResponseT>
void OptProblemAppBody<DomainT,ResponseT>::
	EvalF(real& value)
{
ResponseT* response = new_response();
response->response_bufferF(&value);
int tmp=-1;
App->Eval(point,mode_f,State->F_ASV,tmp,response,true,tmp);
utilib::CachedAllocator<ResponseT> :: deallocate(response);
}
        

//============================================================================
//
//
template <class DomainT, class ResponseT>
void OptProblemAppBody<DomainT,ResponseT>::
	EvalF(typename ResponseT::realarray_t& value)
{
ResponseT* response = new_response();
response->response_bufferF( &(value[0]) );
int tmp=-1;
App->Eval(point,mode_f,State->F_ASV,tmp,response,true,tmp);
utilib::CachedAllocator<ResponseT> :: deallocate(response);
}
        

//============================================================================
//
//
template <class DomainT, class ResponseT>
void OptProblemAppBody<DomainT,ResponseT>::
	EvalCF(typename ResponseT::vector_t& value)
{
ResponseT* response = new_response();
response->response_bufferCF(&value);
int tmp=-1;
App->Eval(point,mode_cf,state()->CF_ASV,tmp,response,true,tmp);
utilib::CachedAllocator<ResponseT> :: deallocate(response);
}


//============================================================================
//
//
template <class DomainT, class ResponseT>
void OptProblemAppBody<DomainT,ResponseT>::
	EvalG(typename ResponseT::vector_t& value)
{
ResponseT* response = new_response();
response->response_bufferG(&value);
int tmp=-1;
App->Eval(point,mode_g,state()->G_ASV,tmp,response,true,tmp);
utilib::CachedAllocator<ResponseT> :: deallocate(response);
}
 

//============================================================================
//
//
template <class DomainT, class ResponseT>
void OptProblemAppBody<DomainT,ResponseT>::
	EvalFG(real& ans, typename ResponseT::vector_t& value)
{
ResponseT* response = new_response();
response->response_bufferF(&ans);
response->response_bufferG(&value);
int tmp=-1;
App->Eval(point,mode_f|mode_g,state()->FG_ASV,tmp,response,true,tmp);
utilib::CachedAllocator<ResponseT> :: deallocate(response);
}
 

//============================================================================
//
//
template <class DomainT, class ResponseT>
void OptProblemAppBody<DomainT,ResponseT>::
	EvalCG(typename ResponseT::vectorarray_t& value)
{
ResponseT* response = new_response();
response->response_bufferCG(&value);
int tmp=-1;
App->Eval(point,mode_cg,state()->CG_ASV,tmp,response,true,tmp);
utilib::CachedAllocator<ResponseT> :: deallocate(response);
}
 

//============================================================================
//
//
template <class DomainT, class ResponseT>
void OptProblemAppBody<DomainT,ResponseT>::
	AsyncEval(std::vector<int>& asv, int mode, int& priority, 
						ResponseT* response, int tag)
{
State->last_id = tag;
if (tag != -1){
   response->info->id = tag;
   response->info->id_generate=false;
}
else
   response->info->id_generate=true;
App->Eval(point,mode,asv,priority,response,false,State->last_id);
}


//============================================================================
//
//
template <class DomainT, class ResponseT>
void OptProblemAppBody<DomainT,ResponseT>::
	AsyncEval(int mode, int& priority, ResponseT* response, int tag)
{
State->init_request_asv(mode);
State->last_id = tag;
if (tag != -1) {
   response->info->id = tag;
   response->info->id_generate=false;
   }
else
   response->info->id_generate=true;
App->Eval(point,mode,State->REQUEST_ASV,priority,
				response,false,State->last_id);
}


//============================================================================
//
//
template <class DomainT, class ResponseT>
void OptProblemAppBody<DomainT,ResponseT>::
	AsyncEvalF(real *value, int& priority, int tag)
{
ResponseT* response = new_response();
response->response_bufferF(value);
AsyncEval(State->F_ASV,mode_f,priority,response,tag);
}
 
 
//============================================================================
//
//
template <class DomainT, class ResponseT>
void OptProblemAppBody<DomainT,ResponseT>::
	AsyncEvalF(typename ResponseT::realarray_t* values, int& priority, int tag)
{
ResponseT* response = new_response();
response->response_bufferF( &(values[0]) );
AsyncEval(State->F_ASV,mode_f,priority,response,tag);
}
 
 
//============================================================================
//
//
template <class DomainT, class ResponseT>
void OptProblemAppBody<DomainT,ResponseT>::
	AsyncEvalG(typename ResponseT::vector_t* value, int& priority, int tag)
{
ResponseT* response = new_response();
response->response_bufferG(value);
AsyncEval(State->G_ASV,mode_g,priority,response,tag);
}
 

//============================================================================
//
//
template <class DomainT, class ResponseT>
void OptProblemAppBody<DomainT,ResponseT>::
	AsyncEvalFG(real* ans, typename ResponseT::vector_t* value, int& priority, int tag)
{
ResponseT* response = new_response();
response->response_bufferF(ans);
response->response_bufferG(value);
AsyncEval(State->FG_ASV,mode_f|mode_g,priority,response,tag);
} 


//============================================================================
//
//
template <class DomainT, class ResponseT>
void OptProblemAppBody<DomainT,ResponseT>::
	AsyncEvalCF(typename ResponseT::vector_t* value, int& priority, int tag)
{
ResponseT* response = new_response();
response->response_bufferCF(value);
AsyncEval(State->CF_ASV,mode_cf,priority,response,tag);
}


//============================================================================
//
//
template <class DomainT, class ResponseT>
void OptProblemAppBody<DomainT,ResponseT>::
	AsyncEvalCG(typename ResponseT::vectorarray_t* value, int& priority, int tag)
{
ResponseT* response = new_response();
response->response_bufferCG(value);
AsyncEval(State->CG_ASV,mode_cg,priority,response,tag);
}



} // namespace colin

#endif
