/*  _________________________________________________________________________
 *
 *  COLIN: A Common Optimization Library INterface
 *  Copyright (c) 2003, Sandia National Laboratories.
 *  This software is distributed under the GNU Lesser General Public License.
 *  For more information, see the README.html file in the top COLIN directory.
 *  _________________________________________________________________________
 */

/**
 * \file OptApplicationBase.h
 *
 * Defines the colin::OptApplicationBase class.
 */

#ifndef colin_OptApplicationBase_h
#define colin_OptApplicationBase_h

#include <acro_config.h>
#include <iostream>
#include <list>
#include <stdexcept>
#include <utilib/LinkedList.h>
#include <utilib/exception_mngr.h>
#include <colin/AppResponseInfo.h>


namespace colin {


///
enum optimizationSense
    {
      minimization = 1,
      unknown = 0,
      maximization = -1
    };


/**
  * The base class for an application interface class, which encapsulates 
  * the mechanics needed to executed the code for an optimization application.
  */
class OptApplicationBase
{
public:
 
  /// Generic constructor
  OptApplicationBase();

  /// Destructor
  virtual ~OptApplicationBase() {}

  /// Initialize the application
  virtual void initialize(int* , char*** ) {}

  /// Finish up an application
  virtual void finalize() {}

  /// Start a batch set of async evaluations
  virtual void start_async_batch() {}

  /// Finish a batch set of async evaluations
  virtual void end_async_batch() {}

  /// This synchronizes the calculations requested
  virtual void synchronize()
		{
		async_request_ids.clear();
		async_completed_ids.clear();
		dead_ids.clear();
		}

  /// This returns the id of the next asynchronous evaluation that terminates
  virtual int next_eval()
		{
		if (async_completed_ids.size() == 0)
		   return -1;
		int tmp = async_completed_ids.front();
		async_completed_ids.pop_front();
		return tmp;
		}

  /// Terminate the evaluation for the requested evaluation id
  virtual bool terminate_eval(const int /*id*/) {return false;}

  /// Inform the application to ignore a given evaluation
  virtual void ignore_eval(const int id)
		{
		if (dead_ids.find(id) == dead_ids.end()) {
		   dead_ids.insert(id);
		   bool flag=true;
		   //
		   // Ignore from completed ids
		   //	
		   std::list<int>::iterator curr = async_completed_ids.begin();
		   std::list<int>::iterator end  = async_completed_ids.end();
		   while (curr != end) {
		     if (*curr == id) {
		        flag = false;
		        async_completed_ids.erase(curr++);
		        break;
		        }
		     curr++;
		     }
		   if (!flag) return;
		   //
		   // Ignore from requested ids
		   //	
		   curr = async_request_ids.begin();
		   end  = async_request_ids.end();
		   while (curr != end) {
		     if (*curr == id) {
		        flag = false;
		        async_request_ids.erase(curr++);
		        break;
		        }
		     curr++;
		     }
		   if (!flag) return;
		   EXCEPTION_MNGR(std::runtime_error, "OptApplicationBase::ignore_eval - id " << id << " is not an asynchronous evaluation!");
		   }
		}

  /// Returns the number of queued evaluations
  virtual size_type num_queued_evaluations()
	  	{ return async_request_ids.size() + async_completed_ids.size() + dead_ids.size(); }

  /// Returns the number of evaluation servers
  /// The value '0' indicates that this is a sequential application
  virtual unsigned int num_evaluation_servers()
	  	{ return 0; }

  /// Write out the information about the application
  virtual void write(std::ostream& ) const;

  /** Read in information about the application.
    * There is no sensible default for this, but a given application may
    * need to define this.
    */
  virtual void read(std::istream& ) {}

  /// # objective functions
  int num_objectives;

  /// # nonlinear equality constraints
  int num_eq_constr;

  /// # nonlinear inequality constraints
  int num_ineq_constr;

  /// # of times the application was queried for an evaluation.
  int nprob_ctr;

  /// # of function evaluations computed
  int neval_ctr;

  /// # of constraint evaluations computed
  int nconstr_ctr;

  /// # of evaluations requested.
  int neval_requested;

  /** The capabilities of the application.
    * This variable defines what the application \e can compute, 
    * using the \c ResponseMode enumeration types to define the 
    * information it contains (e.g. mode_f | mode_cf).
    */
  int app_mode;

  /// Copy data from an OptApplicationBase object.
  virtual void copy(OptApplicationBase* app)
  {
	  num_objectives = app->num_objectives;
	  num_eq_constr = app->num_eq_constr;
	  num_ineq_constr = app->num_ineq_constr;
	  nprob_ctr = app->nprob_ctr;
	  neval_ctr = app->neval_ctr;
	  nconstr_ctr = app->nconstr_ctr;
	  neval_requested = app->neval_requested;
	  app_mode = app->app_mode;
  }

  ///
  optimizationSense sense;

protected:

  /// Verify that the requested mode matches the application's capabilities.
  void verify(int mode);

  /// List of ids for requested async evals.
  std::list<int> async_request_ids;

  /// List of ids for completed async evals.
  std::list<int> async_completed_ids;

  /// List of ids for dead evals.
  std::set<int> dead_ids;
};



//============================================================================
//
//
inline OptApplicationBase::OptApplicationBase()
{
num_objectives=1;
num_eq_constr=num_ineq_constr=0;
nprob_ctr=0;
neval_ctr=0;
neval_requested=0;
nconstr_ctr=0;
app_mode=0;
sense=minimization;
}


//============================================================================
//
//
inline void OptApplicationBase::write(std::ostream& os) const
{
os << "#\n# Application Statistics\n#\n";
os << "Num times the app evaluated\t" << nprob_ctr << std::endl;
os << "Num func evals computed\t\t" << neval_ctr << std::endl;
os << "Num constraint evals computed\t" << nconstr_ctr << std::endl;

os << "Application Mode:\t\t";
if (app_mode & mode_f) os << "Func ";
if (app_mode & mode_g) os << "Grad ";
if (app_mode & mode_h) os << "Hess ";
if (app_mode & mode_cf) os << "ConstrFunc ";
if (app_mode & mode_cg) os << "ConstrGrad ";
if (app_mode & mode_ch) os << "ConstrHess ";
os << std::endl;
}


//============================================================================
//
//
inline void OptApplicationBase::verify(int mode)
{
if ((mode & mode_f) && !(app_mode & mode_f))
   EXCEPTION_MNGR(std::runtime_error, "OptApplicationBase::verify - This application does not support objective function evaluations.");

if ((mode & mode_g) && !(app_mode & mode_g))
   EXCEPTION_MNGR( std::runtime_error,"OptApplicationBase::verify - This application does not support gradient evaluations.");

if ((mode & mode_h) && !(app_mode & mode_h))
   EXCEPTION_MNGR( std::runtime_error,"OptApplicationBase::verify - This application does not support hessian evaluations.");

if ((mode & mode_cf) && !(app_mode & mode_cf))
   EXCEPTION_MNGR( std::runtime_error,"OptApplicationBase::verify - This application does not support constraint function evaluations.");

if ((mode & mode_cg) && !(app_mode & mode_cg))
   EXCEPTION_MNGR( std::runtime_error,"OptApplicationBase::verify - This application does not support constraint gradient evaluations.");

if ((mode & mode_ch) && !(app_mode & mode_ch))
   EXCEPTION_MNGR( std::runtime_error,"OptApplicationBase::verify - This application does not support constraint hessian evaluations.");
}


} // namespace colin

#endif
