/*  _______________________________________________________________________

    DAKOTA: Design Analysis Kit for Optimization and Terascale Applications
    Copyright (c) 2001, Sandia National Laboratories.
    This software is distributed under the GNU General Public License.
    For more information, see the README file in the top Dakota directory.
    _______________________________________________________________________ */

//- Class:        ForkAnalysisCode
//- Description:  Class implementation
//- Owner:        Mike Eldred

#include "ForkAnalysisCode.H"
#include "ParallelLibrary.H"
#include "ProblemDescDB.H"
#include "system_defs.h"
#include <sys/wait.h>
#include <unistd.h>

static const char rcsId[]="@(#) $Id: ForkAnalysisCode.C 1184 2002-10-23 23:11:35Z DEVsgopt $";


ForkAnalysisCode::ForkAnalysisCode(const ProblemDescDB& problem_db): 
  AnalysisCode(problem_db)
{
  // argList[3] does not change so it can be initialized here
  argList[3] = (const char *)NULL;
}


ForkAnalysisCode::~ForkAnalysisCode()
{ }


pid_t ForkAnalysisCode::fork_program(const short block_flag)
{
  pid_t pid = 0;

#if (!defined(TFLOPS_COMPUTE) && !defined(CPLANT_COMPUTE))
  // vfork() should be used here since there is an immediate execvp(). This 
  // conserves memory over fork().  If some platforms have problems with a
  // hybrid fork/vfork approach, add #ifdef's but make vfork the default.
#if ( defined(SGI) || defined(SOLARIS) || defined(TFLOPS_SERVICE) )
  // SGI/TFLOPS don't support vfork, Solaris vfork hangs intermittently in 3-pc
  pid = fork();
#else
  pid = vfork(); // replicate this process
#endif

  if (pid == 0) { // this is the child: replace process & execute application
    int return_code = execvp(argList[0], (char* const *)argList);
    // if execvp returns then it failed, exit gracefully. 
    _exit(return_code); // since this is a copy of the parent, use _exit
    // so that the parent i/o stream is not prematurely flushed and closed.
  }
  else if (block_flag) { // parent: wait for completion
    int status;
    waitpid(pid, &status, 0); // be explicit about waiting on the pid created
    // above (use waitpid instead of wait) so that this blocking fork works
    // properly in the possible presence of other nonblocking fork pid's 
    // (required by failure capturing routine, but also good form in general).
    check_status(status); // check the exit status
  }
#endif // TFLOPS_COMPUTE/CPLANT_COMPUTE

  return(pid);
}


/** Check to see if the 3-piece interface terminated abnormally
    (WIFEXITED(status)==0) or if either execvp or the application
    returned a status code of -1 (WIFEXITED(status)!=0 && (signed
    char)WEXITSTATUS(status)==-1). If one of these conditions is
    detected, output a failure message and abort. Note: the
    application code should not return a status code of -1 unless an
    immediate abort of dakota is wanted. If for instance, failure
    capturing is to be used, the application code should write the
    word "FAIL" to the appropriate results file and return a status
    code of 0 through exit(). */
void ForkAnalysisCode::check_status(const int status)
{
  // checks for an abnormal exit status and, if present, abort 
  if( (WIFEXITED(status) != 0 && (signed char)WEXITSTATUS(status) == -1)
    || WIFEXITED(status) == 0 ) {
    Cerr << "Fork application failure, aborting.\nSystem error message: "
         << strerror(errno) << '\n';
    abort_handler(-1);
    // The strerror function returns the system error associated
    // with errno. The variable errno is a system constant defined in
    // errno.h. It contains the number of the last observed error.
    // Note: If it becomes an issue, it would also be possible to 
    // kill simulation drivers or intermediate processes for the 
    // three-piece interface here, based on pid's obtained from any
    // asynch forks. This would be beneficial on systems that do not
    // perform a complete cleanup when one (asynch) child process dies.
  }
}
