/*
================================================================================
    PROJECT:

        John Eddy's Genetic Algorithms (JEGA)

    CONTENTSH:

        Implementation of class DesignGroup.

    NOTES:

        See notes of DesignGroup.hpp.

    PROGRAMMERS:

        John Eddy (jpeddy@sandia.gov) (JE)

    ORGANIZATION:

        Sandia National Laboratories

    COPYRIGHT:

        See the LICENSE file in the top level JEGA directory.

    VERSION:

        1.0.0

    CHANGES:

        Fri May 16 12:05:31 2003 - Original Version (JE)

================================================================================
*/




/*
================================================================================
Document This File
================================================================================
*/
/** \file
 * \brief Contains the implementation of the DesignGroup class.
 */



/*
================================================================================
Includes
================================================================================
*/
// JEGAConfig.hpp should be the first include in all JEGA files.
#include <../Utilities/include/JEGAConfig.hpp>

#include <cfloat>
#include <../Utilities/include/DesignGroup.hpp>
#include <utilities/include/EDDY_DebugScope.hpp>
#include <../Utilities/include/DesignStatistician.hpp>



/*
================================================================================
Namespace Using Directives
================================================================================
*/
using namespace std;
using namespace eddy::utilities;







/*
================================================================================
Begin Namespace
================================================================================
*/
namespace JEGA {
    namespace Utilities {







/*
================================================================================
Private Template Method Implementations
================================================================================
*/
template <typename Predicate>
DesignGroup::DVSortContainer::size_type
DesignGroup::Flush(
    Predicate pred
    )
{
    EDDY_FUNC_DEBUGSCOPE

    // prepare to count the removals.
    DVSortContainer::size_type nrems = 0;

    for(DVSortContainer::iterator it(this->_dvSort.begin());
        it!=this->_dvSort.end();)
    {
        Design* des = *it;

        if( pred(*des) )
        {
            it = this->FlushDesignRetDV(it);
            ++nrems;
        }
        else ++it;
    }
    return nrems;
}




/*
================================================================================
Static Member Data Definitions
================================================================================
*/








/*
================================================================================
Mutators
================================================================================
*/








/*
================================================================================
Accessors
================================================================================
*/








/*
================================================================================
Public Methods
================================================================================
*/

DoubleExtremes
DesignGroup::ComputeDVExtremes(
    ) const
{
    EDDY_FUNC_DEBUGSCOPE
    return DesignStatistician::GetDesignVariableExtremes(
            this->GetDVSortContainer()
            );
}

DoubleExtremes
DesignGroup::ComputeOFExtremes(
    ) const
{
    EDDY_FUNC_DEBUGSCOPE
    return DesignStatistician::GetObjectiveFunctionExtremes(
            this->GetOFSortContainer()
            );
}

bool
DesignGroup::AnyDesignsAreFeasible(
    ) const
{
    EDDY_FUNC_DEBUGSCOPE

    // use a DesignStatistician to see if we can find a feasible Design
    return DesignStatistician::FindFeasible(this->_dvSort) != 
           this->_dvSort.end();

} // DesignGroup::AnyDesignsAreFeasible



DesignGroup::DVSortContainer::iterator
DesignGroup::FlushDesignRetDV(
    DVSortContainer::iterator it
    )
{
    EDDY_FUNC_DEBUGSCOPE


    if(it == this->EndDV()) return this->EndDV();

    Design* des = *it;
    DVSortContainer::iterator ret(this->EraseRetDV(it));
    this->_target.TakeDesign(des);
    return ret;

} // DesignGroup::FlushDesignRetDV

DesignGroup::OFSortContainer::iterator
DesignGroup::FlushDesignRetOF(
    OFSortContainer::iterator it
    )
{
    EDDY_FUNC_DEBUGSCOPE

    if(it == this->EndOF()) return this->EndOF();

    Design* des = *it;
    OFSortContainer::iterator ret(this->EraseRetOF(it));
    this->_target.TakeDesign(des);
    return ret;

} // DesignGroup::FlushDesignRetOF


std::size_t
DesignGroup::CountFeasible(
    ) const
{
    EDDY_FUNC_DEBUGSCOPE

    std::size_t ret = 0;

    // go through and count all evaluated, feasible Designs
    for(DVSortContainer::const_iterator it(this->_dvSort.begin());
        it!=this->_dvSort.end(); ++it)
            if((*it)->IsEvaluated() && (*it)->IsFeasible()) ++ret;

    return ret;
}

void
DesignGroup::ClearContainers(
    )
{
    EDDY_FUNC_DEBUGSCOPE

    this->ClearDVContainer();
    this->ClearOFContainer();
}

void
DesignGroup::FlushAll(
    )
{
    EDDY_FUNC_DEBUGSCOPE

    // send all Designs to the target.
    if(!this->IsEmpty())
    {
        while(!this->_dvSort.empty())
        {
            this->_target.TakeDesign(this->_dvSort.front());
            this->_dvSort.pop_front();
        }

        this->ClearContainers();
    }
}

bool
DesignGroup::SynchronizeOFAndDVContainers(
    ) const
{
    EDDY_FUNC_DEBUGSCOPE

    // if the objective sort is current, we needn't do anything.
    if(IsOFSortCurrent()) return true;

    DesignGroup* ucthis = const_cast<DesignGroup*>(this);
    ucthis->ClearOFContainer();
    const DVSortContainer::const_iterator e(this->_dvSort.end());

    for(DVSortContainer::const_iterator it(this->_dvSort.begin()); it!=e; ++it)
        ucthis->InsertIntoOFContainer(*it);

    return this->IsOFSortCurrent();
}

const DesignGroup&
DesignGroup::operator = (
   const DesignGroup& right
   )
{
    EDDY_FUNC_DEBUGSCOPE
    EDDY_ASSERT(&_target == &right._target);

    if(this== &right) return *this;

    this->_dvSort.clear();
    this->_ofSort.clear();

    this->_dvSort.insert(right._dvSort.begin(), right._dvSort.end());
    this->_ofSort.insert(right._ofSort.begin(), right._ofSort.end());

    return *this;
}

void
DesignGroup::CopyIn(
    const DesignGroup& other
    )
{
    EDDY_FUNC_DEBUGSCOPE

    EDDY_ASSERT(&this->_target == &other.GetDesignTarget());

    // do the DVSortContainer first.
    DVSortContainer::const_iterator dvit(other.BeginDV());
    const DVSortContainer::const_iterator dve(other.EndDV());

    DVSortContainer::iterator dvloc(BeginDV());
    for(; dvit!=dve; ++dvit) dvloc = this->InsertIntoDVContainer(*dvit, dvloc);

    // now do the OFSortContainer.
    OFSortContainer::const_iterator ofit(other.BeginOF());
    const OFSortContainer::const_iterator ofe(other.EndOF());

    OFSortContainer::iterator ofloc(BeginOF());
    for(; ofit!=ofe; ++ofit) ofloc = this->InsertIntoOFContainer(*ofit, ofloc);

} // DesignGroup::CopyIn

void
DesignGroup::MergeIn(
    DesignGroup& other
    )
{
    EDDY_FUNC_DEBUGSCOPE
    this->CopyIn(other);
    other.ClearContainers();

} // DesignGroup::MergeIn

void
DesignGroup::Insert(
   Design* des
   )
{
    EDDY_FUNC_DEBUGSCOPE
    this->InsertRetDV(des);

} // DesignGroup::Insert

DesignGroup::DVSortContainer::iterator
DesignGroup::InsertRetDV(
   Design* des
   )
{
    EDDY_FUNC_DEBUGSCOPE
    EDDY_ASSERT(des != 0x0);

    this->InsertIntoOFContainer(des);
    return this->InsertIntoDVContainer(des);

} // DesignGroup::InsertRetDV

DesignGroup::OFSortContainer::iterator
DesignGroup::InsertRetOF(
   Design* des
   )
{
    EDDY_FUNC_DEBUGSCOPE
    EDDY_ASSERT(des != 0x0);

    this->InsertIntoDVContainer(des);
    return this->InsertIntoOFContainer(des);

} // DesignGroup::InsertRetOF

DesignGroup::DVSortContainer::iterator
DesignGroup::ReplaceRetDV(
   DVSortContainer::iterator where,
   Design* des
   )
{
    EDDY_FUNC_DEBUGSCOPE
    EDDY_ASSERT(des != 0x0);

    this->EraseRetDV(where);
    return this->InsertRetDV(des);

} // DesignGroup::ReplaceRetDV

DesignGroup::OFSortContainer::iterator
DesignGroup::ReplaceRetOF(
   OFSortContainer::iterator where,
   Design* des
   )
{
    EDDY_FUNC_DEBUGSCOPE
    EDDY_ASSERT(des != 0x0);

    this->EraseRetOF(where);
    return this->InsertRetOF(des);

} // DesignGroup::ReplaceRetOF

bool
DesignGroup::Erase(
   const Design* des
   )
{
    EDDY_FUNC_DEBUGSCOPE
    DesignDVSortSet::iterator where(
        this->_dvSort.find_exact(const_cast<Design*>(des))
        );

    bool ret = (where != this->_dvSort.end());
    if(ret) this->EraseRetDV(where);
    return ret;

} // DesignGroup::Erase

DesignGroup::DVSortContainer::iterator
DesignGroup::EraseRetDV(
   DVSortContainer::iterator where
   )
{
    EDDY_FUNC_DEBUGSCOPE
    this->EraseFromOFContainer(this->_ofSort.find_exact(*where));
    return this->EraseFromDVContainer(where);

} // DesignGroup::EraseRetDV

DesignGroup::OFSortContainer::iterator
DesignGroup::EraseRetOF(
   OFSortContainer::iterator where
   )
{
    EDDY_FUNC_DEBUGSCOPE
    this->EraseFromDVContainer(this->_dvSort.find_exact(*where));
    return this->EraseFromOFContainer(where);

} // DesignGroup::EraseRetOF

DesignGroup::DVSortContainer::size_type
DesignGroup::FlushNonUnique()
{
    EDDY_FUNC_DEBUGSCOPE

    size_t nMarked = this->_dvSort.mark_non_unique();
    return nMarked > 0 ? this->FlushDesigns(DVSortContainer::MARK) : 0;

} // FlushNonUnique

DesignGroup::DVSortContainer::size_type
DesignGroup::FlushDesigns(
    const Design::AttributeMask& attrs,
    bool hasOrNot
    )
{
    EDDY_FUNC_DEBUGSCOPE

    Design::AttributePred pred(attrs, hasOrNot);
    return this->Flush(pred);

} // DesignGroup::FlushDesigns

DesignGroup::DVSortContainer::size_type
DesignGroup::FlushDesigns(
    std::size_t attrs,
    bool hasOrNot
    )
{
    EDDY_FUNC_DEBUGSCOPE

    Design::AttributePred pred(attrs, hasOrNot);
    return this->Flush(pred);

} // DesignGroup::FlushDesigns

/*
================================================================================
Subclass Visible Methods
================================================================================
*/

DesignGroup::DVSortContainer::iterator
DesignGroup::EraseFromDVContainer(
   DVSortContainer::iterator where
   )
{
    EDDY_FUNC_DEBUGSCOPE
    if(where == this->_dvSort.end()) return where;
    this->_dvSort.erase(where++);
    return where;

} // DesignGroup::EraseFromDVContainer

DesignGroup::OFSortContainer::iterator
DesignGroup::EraseFromOFContainer(
   OFSortContainer::iterator where
   )
{
    EDDY_FUNC_DEBUGSCOPE
    if(where == this->_ofSort.end()) return where;
    this->_ofSort.erase(where++);
    return where;

} // DesignGroup::EraseFromOFContainer

void
DesignGroup::EraseFromDVContainer(
   const Design* des
   )
{
    EDDY_FUNC_DEBUGSCOPE
    DesignDVSortSet::iterator where(
        this->_dvSort.find_exact(const_cast<Design*>(des))
        );
    if(where != this->_dvSort.end()) this->EraseFromDVContainer(where);

} // DesignGroup::EraseFromDVContainer

void
DesignGroup::EraseFromOFContainer(
   const Design* des
   )
{
    EDDY_FUNC_DEBUGSCOPE
    DesignOFSortSet::iterator where(
        this->_ofSort.find_exact(const_cast<Design*>(des))
        );
    if(where != this->_ofSort.end()) this->EraseFromOFContainer(where);

} // DesignGroup::EraseFromOFContainer

DesignGroup::DVSortContainer::iterator
DesignGroup::InsertIntoDVContainer(
   Design* des
   )
{
    EDDY_FUNC_DEBUGSCOPE
    return this->InsertIntoDVContainer(des, this->BeginDV());

} // DesignGroup::InsertIntoDVContainer

DesignGroup::DVSortContainer::iterator
DesignGroup::InsertIntoDVContainer(
   Design* des,
   DVSortContainer::iterator first
   )
{
    EDDY_FUNC_DEBUGSCOPE
    EDDY_ASSERT(des != 0x0);

    return this->_dvSort.insert(first, des);

} // DesignGroup::InsertIntoDVContainer

DesignGroup::OFSortContainer::iterator
DesignGroup::InsertIntoOFContainer(
   Design* des
   )
{
    EDDY_FUNC_DEBUGSCOPE
    return this->InsertIntoOFContainer(des, this->BeginOF());

} // DesignGroup::InsertIntoOFContainer

DesignGroup::OFSortContainer::iterator
DesignGroup::InsertIntoOFContainer(
   Design* des,
   OFSortContainer::iterator first
   )
{
    EDDY_FUNC_DEBUGSCOPE
    EDDY_ASSERT(des != 0x0);

    return des->IsEvaluated() ?
        this->_ofSort.insert(first, des) : this->_ofSort.end();

} // DesignGroup::InsertIntoOFContainer


DesignGroup::DVSortContainer::iterator
DesignGroup::ReplaceInDVContainer(
   DVSortContainer::iterator where,
   Design* des
   )
{
    EDDY_FUNC_DEBUGSCOPE
    return this->_dvSort.replace(where, des);

} // DesignGroup::ReplaceInDVContainer

DesignGroup::OFSortContainer::iterator
DesignGroup::ReplaceInOFContainer(
   OFSortContainer::iterator where,
   Design* des
   )
{
    EDDY_FUNC_DEBUGSCOPE
    return this->_ofSort.replace(where, des);

} // DesignGroup::ReplaceInOFContainer


DesignGroup::DVSortContainer::size_type
DesignGroup::FlushIfTrue(
    Design::BoolPred::BoolFunc func
    )
{
    EDDY_FUNC_DEBUGSCOPE
    EDDY_ASSERT(func != 0x0);

    Design::BoolPred pred(func, true);
    return this->Flush(pred);

} // DesignGroup::FlushIfTrue

DesignGroup::DVSortContainer::size_type
DesignGroup::FlushIfFalse(
    Design::BoolPred::BoolFunc func
    )
{
    EDDY_FUNC_DEBUGSCOPE
    EDDY_ASSERT(func != 0x0);

    Design::BoolPred pred(func, false);
    return this->Flush(pred);

} // DesignGroup::FlushIfFalse






/*
================================================================================
Subclass Overridable Methods
================================================================================
*/



/*
================================================================================
Private Methods
================================================================================
*/

void
DesignGroup::ClearDVContainer()
{
    EDDY_FUNC_DEBUGSCOPE
    this->_dvSort.clear();
}





/*
================================================================================
Structors
================================================================================
*/
DesignGroup::DesignGroup(
    DesignTarget& target
    ) :
        _dvSort(),
        _ofSort(),
        _target(target),
        _title()
{
    EDDY_FUNC_DEBUGSCOPE

} // DesignGroup::DesignGroup

DesignGroup::DesignGroup(
    const DesignGroup& copy
    ) :
        _dvSort(copy._dvSort),
        _ofSort(copy._ofSort),
        _target(copy._target),
        _title(copy._title)
{
    EDDY_FUNC_DEBUGSCOPE

} // DesignGroup::DesignGroup

DesignGroup::~DesignGroup(
    )
{
    EDDY_FUNC_DEBUGSCOPE
}







/*
================================================================================
End Namespace
================================================================================
*/
    } // namespace Utilities
} // namespace JEGA
