/*
================================================================================
    PROJECT:

        John Eddy's Genetic Algorithms (JEGA) Managed Front End

    CONTENTS:

        Implementation of class MEvaluator.

    NOTES:

        See notes of MEvaluator.hpp.

    PROGRAMMERS:

        John Eddy (jpeddy@sandia.gov) (JE)

    ORGANIZATION:

        Sandia National Laboratories

    COPYRIGHT:

        See the LICENSE file in the top level JEGA directory.

    VERSION:

        1.0.0

    CHANGES:

        Thu Feb 09 10:51:11 2006 - Original Version (JE)

================================================================================
*/




/*
================================================================================
Document This File
================================================================================
*/
/** \file
 * \brief Contains the implementation of the MEvaluator class.
 */




/*
================================================================================
Includes
================================================================================
*/
#include <stdafx.h>
#include <vcclr.h>
#include <MDesign.hpp>
#include <MEvaluator.hpp>
#include <ManagedUtils.hpp>

#pragma unmanaged
#include <GeneticAlgorithmEvaluator.hpp>
#include <../Utilities/include/Design.hpp>
#include <../Utilities/include/Logging.hpp>
#include <../Utilities/include/DesignGroup.hpp>
#include <utilities/include/EDDY_DebugScope.hpp>
#include <../Utilities/include/DesignTarget.hpp>
#include <../Utilities/include/DesignVariableInfo.hpp>
#include <../FrontEnd/Core/include/EvaluatorCreator.hpp>
#pragma managed





/*
================================================================================
Namespace Using Directives
================================================================================
*/
using namespace std;
using namespace JEGA::Logging;
using namespace JEGA::FrontEnd;
using namespace JEGA::Utilities;
using namespace JEGA::Algorithms;
using namespace System::Collections;



/*
================================================================================
Begin Namespace
================================================================================
*/
namespace JEGA {
    namespace FrontEnd {
        namespace Managed {



/*
===========================================================================
Nested Utility Class Implementations
===========================================================================
*/

class BaseEvaluator :
    public GeneticAlgorithmEvaluator
{
    /*
    ===========================================================================
    Member Data Declarations
    ===========================================================================
    */
    private:

        gcroot<MEvaluator MOH> _managedEvaluator;


    /*
    ===========================================================================
    Subclass Overridable Methods
    ===========================================================================
    */
    public:

        /**
         * \brief This is the method in which this evaluator does what it does.
         *
         * It may be called either by the genetic algorithm or by the
         * overloaded version of this method taking a DesignGroup argument.
         *
         * This implementation makes use of the managed global evaluator
         * pointer in this file to call to what should resolve to some
         * code possibly in some other language.
         *
         * Note that evaluation in this algorithm has nothing to do with
         * fitness assessment.  This method only installs raw constraint and
         * objective function values.  Fitness assessment based on these values
         * is the job of a GeneticAlgorithmFitnessAssessor.
         *
         * \param des The Design object to evaluate.
         * \return True if "des" winds up with proper responses and everything
         *         is ok and false otherwise.  This base class implementation
         *            always returns true.
         */
        virtual
        bool
        Evaluate(
            EvaluationJob& evalJob
            )
        {
            EDDY_FUNC_DEBUGSCOPE
            return this->Evaluate(evalJob.GetDesign());
        }

        virtual
        bool
        Evaluate(
            Design& des
            )
        {
            EDDY_FUNC_DEBUGSCOPE

            bool ret = false;

            try {
                ret = this->_managedEvaluator->PerformEvaluation(des);

                // if PerformEvaluation returned badly, we mark the
                // design illconditioned.
                if(!ret) des.SetIllconditioned(true);
            }
            catch(...) {
                des.SetIllconditioned(true);
            }

            // Do base class evaluation finalization no matter what. 
            // It knows enough to properly handle illconditioned designs.
            return this->GeneticAlgorithmEvaluator::PostEvaluate(des) && ret;
        }

        /// Returns the proper name of this operator.
        /**
         * \return The string "null_evaluation".
         */
        virtual
        string
        GetName(
            ) const
        {
            EDDY_FUNC_DEBUGSCOPE
            return "managed_evaluator";
        }

        /// Returns a full description of what this operator does and how.
        /**
         * \verbatim
            This evaluator is a derivative of the
            GeneticAlgorithmEvaluator and uses managed extensions to
            hold and call-forward to a managed evaluator (MEvaluator)
            class instance's PerformEvaluation method.
           \endverbatim.
         *
         * \return A description of the operation of this operator.
         */
        virtual
        string
        GetDescription(
            ) const
        {
            EDDY_FUNC_DEBUGSCOPE
            return "This evaluator is a derivative of the "
                "GeneticAlgorithmEvaluator and uses managed extensions to "
                "hold and call-forward to a managed evaluator (MEvaluator) "
                "class instance's PerformEvaluation method.";
        }

        /**
         * \brief Returns a new instance of this operator class for use by
         *        \a algorithm.
         *
         * \param algorithm The GA for which the new evaluator is to be used.
         * \return A new, default instance of a NullEvaluator.
         */
        virtual
        GeneticAlgorithmOperator*
        Clone(
            GeneticAlgorithm& algorithm
            ) const
        {
            EDDY_FUNC_DEBUGSCOPE
            return new BaseEvaluator(algorithm, this->_managedEvaluator);
        }


    /*
    ===========================================================================
    Structors
    ===========================================================================
    */
    public:

        BaseEvaluator(
            GeneticAlgorithm& algorithm,
            gcroot<MEvaluator MOH> managedEvaluator
            ) :
                GeneticAlgorithmEvaluator(algorithm),
                _managedEvaluator(managedEvaluator)
        {
            EDDY_FUNC_DEBUGSCOPE
        }

}; // class BaseEvaluator


class BaseEvaluatorCreator :
    public JEGA::FrontEnd::EvaluatorCreator
{
    /*
    ===========================================================================
    Member Data Declarations
    ===========================================================================
    */
    private:

        gcroot<MEvaluator MOH> _managedEvaluator;

        BaseEvaluator* _theEvaler;

    /*
    ===========================================================================
    Public Methods
    ===========================================================================
    */
    public:

        BaseEvaluator*
        CurrentEvaluator(
            ) const
        {
            EDDY_FUNC_DEBUGSCOPE
            return this->_theEvaler;
        }

    /*
    ===========================================================================
    Subclass Overridable Methods
    ===========================================================================
    */
    public:

        virtual
        JEGA::Algorithms::GeneticAlgorithmEvaluator*
        CreateEvaluator(
            JEGA::Algorithms::GeneticAlgorithm& algorithm
            )
        {
            EDDY_FUNC_DEBUGSCOPE
            if((this->_theEvaler == 0x0) || 
               (&this->_theEvaler->GetAlgorithm() != &algorithm))
                return new BaseEvaluator(algorithm, this->_managedEvaluator);
            return this->CurrentEvaluator();
        };

    /*
    ===========================================================================
    Structors
    ===========================================================================
    */
    public:

        BaseEvaluatorCreator(
            gcroot<MEvaluator MOH> managedEvaluator
            ) :
                _managedEvaluator(managedEvaluator),
                _theEvaler(0x0)
        {
            EDDY_FUNC_DEBUGSCOPE
        }

}; // class BaseEvaluatorCreator


/*
================================================================================
Static Member Data Definitions
================================================================================
*/








/*
================================================================================
Mutators
================================================================================
*/

void
MEvaluator::SetEvaluationDelegate(
    MEvaluationFunctor MOH del
    )
{
    EDDY_FUNC_DEBUGSCOPE
    this->_theDelegate = del;
}






/*
================================================================================
Accessors
================================================================================
*/

MEvaluationFunctor MOH
MEvaluator::GetEvaluationDelegate(
    )
{
    EDDY_FUNC_DEBUGSCOPE
    return this->_theDelegate;
}






/*
================================================================================
Public Methods
================================================================================
*/

bool
MEvaluator::PerformEvaluation(
    Design& des
    )
{
    EDDY_FUNC_DEBUGSCOPE
    try
    {
        MDesign MOH toEval = MANAGED_GCNEW MDesign(&des);
        return this->_theDelegate->Evaluate(toEval);
    }
    catch(System::Exception MOH JEGA_LOGGING_IF_ON(ex))
    {
        JEGALOG_G(lquiet(),
            text_entry(lverbose(),
                "Managed Evaluator caught an exception thrown by the "
                "evaluation functor reading \"" + ToStdStr(ex->Message) +
                "\".  The design being evaluated will be considered "
                "illconditioned."
                )
            )

        return false;
    }
}

EvaluatorCreator&
MEvaluator::GetTheEvaluatorCreator(
    )
{
    EDDY_FUNC_DEBUGSCOPE
    return *this->_theEvalCreator;
}






/*
================================================================================
Subclass Visible Methods
================================================================================
*/





/*
================================================================================
Subclass Overridable Methods
================================================================================
*/
void
MEvaluator::MANAGED_DISPOSE(
    )
{
    EDDY_FUNC_DEBUGSCOPE
    delete this->_theEvalCreator;
    this->_theEvalCreator = 0x0;
}




/*
================================================================================
Private Methods
================================================================================
*/








/*
================================================================================
Structors
================================================================================
*/

MEvaluator::MEvaluator(
    ) :
        _theDelegate(MANAGED_NULL_HANDLE),
        _theEvalCreator(MANAGED_NULL_HANDLE)
{
    EDDY_FUNC_DEBUGSCOPE
    this->_theEvalCreator = new BaseEvaluatorCreator(this);
}


MEvaluator::MEvaluator(
    MEvaluationFunctor MOH theDelegate
    ) :
        _theDelegate(theDelegate),
        _theEvalCreator(MANAGED_NULL_HANDLE)
{
    EDDY_FUNC_DEBUGSCOPE
    this->_theEvalCreator = new BaseEvaluatorCreator(this);
}

MEvaluator::~MEvaluator(
    )
{
    EDDY_FUNC_DEBUGSCOPE
    this->MANAGED_DISPOSE();
}







/*
================================================================================
End Namespace
================================================================================
*/
        } // namespace Managed
    } // namespace FrontEnd
} // namespace JEGA


