/// Creates a '1-Way Analysis of Variance' table

///
/// Programmed: EJW, 2005
/// Modifiers:  MMC, 1/30/2006    Added comments 
///                               Call to gslib to print out p-value in ANOVA table  
///                               Renamed to OneWayANOVA

#include "OneWayANOVA.h"
#include <iomanip>
#include <iostream>

namespace DDaceMainEffects {


OneWayANOVA::OneWayANOVA(std::vector<Factor> factors)
	: factors_(factors)
{

	/// Check to make sure that there are factors (i.e. that the vector
	/// is not empty. And if it is, throw a runtime_error
	if(factors_.empty()) 
	  throw std::runtime_error("Error in MainEffects ctor: factors is empty");

	/// Check to make sure that the size of each Factor vector is the
	/// same. If they are not the same, then throw a runtime_error
	int factorSize = factors_[0].getNumberOfObservations();
	
	for(int i = 1; i < (int) factors_.size(); i++)
	{
	  if(factorSize != factors_[i].getNumberOfObservations())
	    throw runtime_error("Error in MainEffects ctor: factors are different sizes");				
	}

}

#ifdef HAVE_GSL
#include <gsl/gsl_randist.h>
#endif
std::string OneWayANOVA::getANOVATable(int factor)
{  	
#ifdef HAVE_SSTREAM
        std::ostringstream ss;
#else
        std::ostrstream ss;
#endif

	/// The 1-Way ANOVA table contains the analysis of variance components
	/// associated with apportioning the variance in a response variable
	/// to just one factor ("Between Groups") and to the remaining error ("Within Groups"):
	///
	/// Source of                  Sum of  
	/// Variation         DoF     Squares     Fdata    P
	/// -----------------------------------------------------------------
	/// Between Groups    N1        XXX       XXX      XXX
	/// Within Groups     N2        XXX             
	/// Total             N1+N2     XXX
	///
	///
	/// The "Fdata" term corresponds to the ratio of these variances and implies importance.
	/// The p-value gives significance of importance.  
	/// If the p-value is smaller than your acceptable Type I error (or risk alpha), then this implies
	/// a (1-p) Confidence that the "Between Groups" effect on the response variable variance 
	/// due to the Factor is significant.  


        Factor currFac = getFactor(factor);
	double SSBG = currFac.sumOfSquaresBetweenGroups();
	double SSWG = currFac.sumOfSquaresWithinGroups();
	int doFBetween = currFac.doFBetween();
	int doFWithin = currFac.doFWithin();
	double Fdata = currFac.Fdata();
#ifdef HAVE_GSL
	double p_value = gsl_ran_fdist_pdf(Fdata,doFBetween,doFWithin);
#endif

	ss.setf(ios::scientific);
        ss<<setprecision(5);
	ss <<"Source of        "  << "      " << "        Sum of" << "      Mean Sum" << std::endl;
        ss <<"Variation        " << "   DoF"  << "       Squares" << "    of Squares"  << "         Fdata" << "       p-value"  << std::endl;
#ifdef HAVE_GSL
        ss <<"Between Groups   " << setw(6) << doFBetween << setw(14) << SSBG << setw(14) << currFac.varianceBetweenGroups() << setw(14) << Fdata       << setw(14)    << p_value     << std::endl;
#else
        ss <<"Between Groups   " << setw(6) << doFBetween << setw(14) << SSBG << setw(14) << currFac.varianceBetweenGroups() << setw(14) << Fdata       << setw(14)    << " ************ "     << std::endl;
#endif
        ss  <<"Within Groups    " << setw(6)<< doFWithin  << setw(14) << SSWG << setw(14) <<currFac.varianceWithinGroups() << std::endl;
	ss  << "Total            " << setw(6) << (doFBetween + doFWithin)  << setw(14) << ( SSBG + SSWG ) << std::endl;  
        return(ss.str());

}


std::string OneWayANOVA::getANOVATables()
{
#ifdef HAVE_SSTREAM
        std::ostringstream ss;
#else
        std::ostrstream ss;
#endif
        for(int i = 0; i < (int) factors_.size(); i++)
        {
                
              ss << "\n"  << "ANOVA Table for Factor (Variable) " << i+1 
                 << "\n"  << getANOVATable(i);
        }

	//         ss << getANOVATable(0);
         return(ss.str());                         
        
}

void OneWayANOVA::printANOVATable(int factor)
{
        std::cout << getANOVATable(factor);                               
}
 
void OneWayANOVA::printANOVATables()
{
        std::cout << getANOVATables();
}

}//namespace
