/*\file ModelProcessor.c
 *\brief: processes model inputs from the matlab workspace, and transforms them into datasets or vectors 
 *        that will be used throughout the matlab solution sequences.
 */

#include "./ModelProcessor.h"

void mexFunction( int nlhs, mxArray* plhs[], int nrhs, const mxArray* prhs[]){ 

	int i;

	/*inputs: */
	char* inputfilename=NULL;
	FILE* IOMODEL;
	int     solution_type;
	double* double_analyses=NULL;
	int*    analyses=NULL;
	int     numanalyses;
	
	/* output datasets: */
	Elements* elements=NULL;
	Nodes* nodes=NULL;
	Vertices* vertices=NULL;
	Constraints* constraints=NULL;
	Loads* loads=NULL;
	Materials* materials=NULL;
	Parameters* parameters=NULL;
	
	/*Boot module: */
	MODULEBOOT();

	/*checks on arguments on the matlab side: */
	CheckNumMatlabArguments(nlhs,NLHS,nrhs,NRHS,__FUNCT__,&ModelProcessorUsage);
	
	/*open file input for reading: */
	FetchMatlabData(&inputfilename,INPUTFILE);
	IOMODEL= pfopen(inputfilename ,"rb");

	/*retrieve solution type and analyses: */
	FetchMatlabData(&solution_type,SOLUTIONTYPE);
	FetchMatlabData(&double_analyses,&numanalyses,ANALYSES);
	analyses=(int*)xmalloc(numanalyses*sizeof(int));
	for(i=0;i<numanalyses;i++)analyses[i]=(int)double_analyses[i];
	
	/*call x code ModelProcessorx: */
	ModelProcessorx(&elements,&nodes,&vertices,&materials,&constraints, &loads, &parameters, IOMODEL,solution_type,numanalyses,analyses);

	/*Write output data: */
	WriteMatlabData(ELEMENTS,elements);
	WriteMatlabData(NODES,nodes);
	WriteMatlabData(VERTICES,vertices);
	WriteMatlabData(CONSTRAINTS,constraints);
	WriteMatlabData(LOADS,loads);
	WriteMatlabData(MATERIALS,materials);
	WriteMatlabData(PARAMETERS,parameters);

	/*Free ressources: */
	delete elements;
	delete nodes;
	delete vertices;
	delete constraints;
	delete loads;
	delete materials;
	delete parameters;
	xfree((void**)&analyses);
	xfree((void**)&double_analyses);
	xfree((void**)&inputfilename);
	
	/*Close input file: */
	pfclose(IOMODEL,inputfilename);
	
	/*end module: */
	MODULEEND();
}
	
void ModelProcessorUsage(void) {
	_printf_(true,"\n");
	_printf_(true,"   usage: [elements,vertices,constraints,loads,materials,parameters,part,tpart]=ModelProcessor(model,analyses,inputfile)\n");
	_printf_(true,"   where: model is an instance of the matlab @model class and analyses is a list of analysis types being performed.\n");
	_printf_(true,"          inputfile is the file holding the marshalled datasets from the model.\n");
	_printf_(true,"\n");
}
