/*!\file: control_core.cpp
 * \brief: core of the control solution 
 */ 
#include <config.h>
#include "../toolkits/toolkits.h"
#include "../objects/objects.h"
#include "../shared/shared.h"
#include "../io/io.h"
#include "../EnumDefinitions/EnumDefinitions.h"
#include "./solutions.h"
#include "../modules/modules.h"
#include "../include/include.h"
#include "../solvers/solvers.h"

#if defined (_HAVE_TAO_) && (_PETSC_MAJOR_ == 3 && _PETSC_MINOR_ == 2)
#include <tao.h>

/*Local prototype*/
int FormFunctionGradient(TaoSolver tao,Vec,double*,Vec,void*);
typedef struct {
	FemModel* femmodel;
} AppCtx;

void controltao_core(FemModel* femmodel){

	/*TAO*/
	int        ierr,numberofvertices;
	int        num_controls;
	AppCtx     user;
	TaoSolver  tao;
	int       *control_list   = NULL;
	Vec        X              = NULL;
	Vec        XL             = NULL;
	Vec        XU             = NULL;

	/*Initialize TAO*/
	int argc; char **args=NULL;
	PetscGetArgs(&argc,&args);
	ierr = TaoInitialize(&argc,&args,(char*)0,"");
	if(ierr) _error_("Could not initialize Tao");

	/*Recover some parameters*/
	femmodel->parameters->FindParam(&num_controls,InversionNumControlParametersEnum);
	femmodel->parameters->FindParam(&control_list,NULL,InversionControlParametersEnum);

	/*Initialize TAO*/
	TaoCreate(PETSC_COMM_WORLD,&tao);
	PetscOptionsSetValue("-tao_monitor","");
	TaoSetFromOptions(tao);
	TaoSetType(tao,"tao_blmvm");

	/*Prepare all TAO parameters*/
	TaoSetMaximumFunctionEvaluations(tao,50);
	TaoSetMaximumIterations(tao,10);
	TaoSetTolerances(tao,10e-28,0.0000,0.0000,0.0000,10e-28);

	numberofvertices = femmodel->vertices->NumberOfVertices();
	XL=NewVec(numberofvertices); VecSet(XL,1.);
	XU=NewVec(numberofvertices); VecSet(XU,200.);
	//TaoSetVariableBounds(tao,XL,XU); VecFree(&XL); VecFree(&XU);

	GetVectorFromControlInputsx(&X,femmodel->elements,femmodel->nodes,femmodel->vertices,femmodel->loads,femmodel->materials,femmodel->parameters);
	TaoSetInitialVector(tao,X);

	user.femmodel=femmodel;
	TaoSetObjectiveAndGradientRoutine(tao,FormFunctionGradient,(void*)&user); 

	/*Solver optimization problem*/
	TaoSolve(tao);
	TaoView(tao,PETSC_VIEWER_STDOUT_WORLD);
	TaoGetSolutionVector(tao,&X);
	SetControlInputsFromVectorx(femmodel->elements,femmodel->nodes,femmodel->vertices,femmodel->loads,femmodel->materials,femmodel->parameters,X);
	for(int i=0;i<num_controls;i++){
		InputToResultx(femmodel->elements,femmodel->nodes,femmodel->vertices,femmodel->loads,femmodel->materials,femmodel->parameters,control_list[i]);
	}

	/*Clean up and return*/
	xfree((void**)&control_list);
	VecFree(&X);
	TaoDestroy(&tao);
	TaoFinalize();
}
int FormFunctionGradient(TaoSolver tao, Vec X, double *fcn,Vec G,void *userCtx){

	/*Retreive arguments*/
	int       solution_type,num_cost_functions;
	AppCtx   *user           = (AppCtx *)userCtx;
	FemModel *femmodel       = user->femmodel;
	int      *cost_functions = NULL;
	double   *cost_functionsd= NULL;
	Vec       gradient       = NULL;

	/*Set new variable*/
	SetControlInputsFromVectorx(femmodel->elements,femmodel->nodes,femmodel->vertices,femmodel->loads,femmodel->materials,femmodel->parameters,X);

	/*Recover some parameters*/
	femmodel->parameters->FindParam(&solution_type,SolutionTypeEnum);
	femmodel->parameters->FindParam(&num_cost_functions,InversionNumCostFunctionsEnum);
	femmodel->parameters->FindParam(&cost_functionsd,NULL,NULL,InversionCostFunctionsEnum);

	/*Prepare objective function*/
	cost_functions=(int*)xmalloc(num_cost_functions*sizeof(int));
	for(int i=0;i<num_cost_functions;i++) cost_functions[i]=(int)cost_functionsd[i]; //FIXME
	femmodel->parameters->SetParam(cost_functions,1,num_cost_functions,StepResponsesEnum);

	/*Compute solution and adjoint*/
	void (*solutioncore)(FemModel*)=NULL;
	void (*adjointcore)(FemModel*)=NULL;
	CorePointerFromSolutionEnum(&solutioncore,femmodel->parameters,solution_type);
	AdjointCorePointerFromSolutionEnum(&adjointcore,solution_type);
	solutioncore(femmodel);
	adjointcore(femmodel);

	/*Compute objective function*/
	CostFunctionx(fcn,femmodel->elements,femmodel->nodes,femmodel->vertices,femmodel->loads,femmodel->materials,femmodel->parameters);

	/*Compute gradient*/
	Gradjx(&gradient,NULL,femmodel->elements,femmodel->nodes,femmodel->vertices,femmodel->loads,femmodel->materials,femmodel->parameters);
	VecCopy(gradient,G); VecFree(&gradient);
	VecScale(G,-1.);

	/*Clean-up and return*/
	xfree((void**)&cost_functions);
	xfree((void**)&cost_functionsd);
	return 0;
}

#else
void controltao_core(FemModel* femmodel){
	_error_("TAO not installed or PETSc version not supported");
}
#endif //_HAVE_TAO_ 
