/*!\file: control_core.cpp
 * \brief: core of the control solution 
 */ 
#include <config.h>
#include "../toolkits/toolkits.h"
#include "../objects/objects.h"
#include "../shared/shared.h"
#include "../io/io.h"
#include "../EnumDefinitions/EnumDefinitions.h"
#include "./solutions.h"
#include "../modules/modules.h"
#include "../include/include.h"
#include "../solvers/solvers.h"

#if defined (_HAVE_TAO_) && (_PETSC_MAJOR_ == 3 && _PETSC_MINOR_ == 2)
#include <tao.h>

/*Local prototype*/
int FormFunctionGradient(TaoSolver tao,Vec,double*,Vec,void*);
typedef struct {
	FemModel* femmodel;
} AppCtx;

void controltao_core(FemModel* femmodel){

	/*TAO*/
	int       i,n,info;
	TaoSolver tao;
	Vec       initial_solution = NULL;
	AppCtx    user;

	/*Initialize TAO*/
	int argc; char **args; PetscGetArgs(&argc,&args);
	int ierr=TaoInitialize(&argc,&args,(char*)0,"");
	if(ierr) _error_("Could not initialize Tao");

	/*Line search options*/
	/*TAO options: http://www.mcs.anl.gov/research/projects/tao/docs/manualpages/solver/TaoSetFromOptions.html*/
	info = PetscOptionsSetValue("-tao_monitor",""); if(info) _error_("STOP");
	info = PetscOptionsSetValue("-tao_max_its","20"); if(info) _error_("STOP");
	info = PetscOptionsSetValue("-tao_max_funcs","50"); if(info) _error_("STOP");

	/*Initialize argument*/
	user.femmodel=femmodel;

	/*Set up and solve TAO*/
	info = TaoCreate(PETSC_COMM_WORLD,&tao); if(info) _error_("STOP");
	info = TaoSetType(tao,"tao_blmvm"); if(info) _error_("STOP");
	//info = TaoSetType(tao,"tao_cg"); if(info) _error_("STOP");

	int size=femmodel->vertices->NumberOfVertices();
	Vec XL=NULL;
	Vec XU=NULL;
	VecCreate(PETSC_COMM_WORLD,&XL);                               VecCreate(PETSC_COMM_WORLD,&XU);
	VecSetSizes(XL,PetscDetermineLocalSize(size),PETSC_DECIDE);    VecSetSizes(XU,PetscDetermineLocalSize(size),PETSC_DECIDE);
	VecSetFromOptions(XL);                                         VecSetFromOptions(XU);
	VecSet(XL,1);                                                  VecSet(XU,200.);
	TaoSetVariableBounds(tao,XL,XU); VecFree(&XL); VecFree(&XU);

	GetVectorFromInputsx(&initial_solution,femmodel->elements,femmodel->nodes, femmodel->vertices, femmodel->loads, femmodel->materials, femmodel->parameters,FrictionCoefficientEnum,VertexEnum);
	info = TaoSetInitialVector(tao,initial_solution);  if(info) _error_("STOP");
	VecFree(&initial_solution);

	info = TaoSetObjectiveAndGradientRoutine(tao,FormFunctionGradient,(void*)&user);  if(info) _error_("STOP");
	info = TaoSetFromOptions(tao);  if(info) _error_("STOP");
	/* http://www.mcs.anl.gov/research/projects/tao/docs/manpages/taosolver/TaoSetTolerances.html*/
	/*                          fatol ,frtol ,gatol ,grtol ,gttol*/
	info = TaoSetTolerances(tao,10e-28,0.0000,0.0000,0.0000,10e-28); if(info) _error_("STOP");
	info = TaoSolve(tao); if(info) _error_("STOP");

	/*Get solution status*/
	info = TaoView(tao,PETSC_VIEWER_STDOUT_WORLD);  if(info) _error_("STOP");

	/*Clean up*/
	info = TaoDestroy(&tao);  if(info) _error_("STOP");
	InputToResultx(femmodel->elements,femmodel->nodes,femmodel->vertices,femmodel->loads,femmodel->materials,femmodel->parameters,FrictionCoefficientEnum);

	/* Finalize TAO */
	TaoFinalize();
}
int FormFunctionGradient(TaoSolver tao, Vec X, double *fcn,Vec G,void *userCtx){

	/*Retreive arguments*/
	AppCtx   *user     = (AppCtx*)userCtx;
	FemModel *femmodel = user->femmodel;
	Vec       gradient = NULL;

	int costfunction=SurfaceAbsVelMisfitEnum;
	femmodel->parameters->SetParam(&costfunction,1,1,StepResponsesEnum);
	//InputUpdateFromVectorx(femmodel->elements,femmodel->nodes, femmodel->vertices, femmodel->loads, femmodel->materials, femmodel->parameters,X,MaterialsRheologyBbarEnum,VertexEnum);
	InputUpdateFromVectorx(femmodel->elements,femmodel->nodes,femmodel->vertices,femmodel->loads,femmodel->materials,femmodel->parameters,X,FrictionCoefficientEnum,VertexEnum);
	adjointdiagnostic_core(user->femmodel);
	//Gradjx(&gradient, femmodel->elements,femmodel->nodes, femmodel->vertices,femmodel->loads, femmodel->materials,femmodel->parameters, MaterialsRheologyBbarEnum);
	Gradjx(&gradient, femmodel->elements,femmodel->nodes, femmodel->vertices,femmodel->loads, femmodel->materials,femmodel->parameters,FrictionCoefficientEnum);
	VecScale(gradient,-1.);
	VecCopy(gradient,G);
	VecFree(&gradient);
	CostFunctionx(fcn, femmodel->elements,femmodel->nodes, femmodel->vertices,femmodel->loads, femmodel->materials, femmodel->parameters);
	return 0;
}

#else
void controltao_core(FemModel* femmodel){
	_error_("TAO not installed or PETSc version not supported");
}
#endif //_HAVE_TAO_ 
