/*!\file: control_core.cpp
 * \brief: core of the control solution 
 */ 
#include <config.h>
#include "../toolkits/toolkits.h"
#include "../objects/objects.h"
#include "../shared/shared.h"
#include "../io/io.h"
#include "../EnumDefinitions/EnumDefinitions.h"
#include "./solutions.h"
#include "../modules/modules.h"
#include "../include/include.h"
#include "../solvers/solvers.h"

#if defined (_HAVE_TAO_) && (_PETSC_MAJOR_ == 3 && _PETSC_MINOR_ == 2)
#include <tao.h>

/*Local prototype*/
int FormFunction(TaoSolver tao,Vec,double *,void*);
int FormFunctionGradient(TaoSolver tao,Vec,double *,Vec,void*);
typedef struct {
	FemModel* femmodel;
} AppCtx;

void controltao_core(FemModel* femmodel){

	/*TAO*/
	int                        i,n,info;
	TaoSolverTerminationReason reason;
	TaoSolver                  tao;
	Vec                        initial_solution = NULL;
	AppCtx                     user;
	PetscInt                   iter;
	double                     ff                 ,gnorm;

	/*Initialize TAO*/
	int argc; char **args; PetscGetArgs(&argc,&args);
	int ierr=TaoInitialize(&argc,&args,(char*)0,"");
	if(ierr) _error_("Could not initialize Tao");

	/*Line search options*/
	//info = PetscOptionsSetValue("-tao_ls_stepmax","10e11"); if(info) _error_("STOP"); //does not work
	//info = PetscOptionsSetValue("-tao_ls_stepmin","10e5"); if(info) _error_("STOP");    //does not work
	info = PetscOptionsSetValue("-tao_ls_maxfev","3"); if(info) _error_("STOP");
	/*TAO options: http://www.mcs.anl.gov/research/projects/tao/docs/manualpages/solver/TaoSetFromOptions.html*/
	info = PetscOptionsSetValue("-tao_monitor",""); if(info) _error_("STOP");
	info = PetscOptionsSetValue("-tao_max_its","10"); if(info) _error_("STOP");
	info = PetscOptionsSetValue("-tao_max_funcs","30"); if(info) _error_("STOP");

	/*Additional options*/
	//info = PetscOptionsSetValue("-info","/u/astrid-r1b/morlighe/svn/issm/trunk/test/NightlyRun/taolog.txt"); if(info) _error_("STOP");

	/*Initialize argument*/
	user.femmodel=femmodel;

	/*Set up and solve TAO*/
	GetVectorFromInputsx(&initial_solution,femmodel->elements,femmodel->nodes, femmodel->vertices, femmodel->loads, femmodel->materials, femmodel->parameters,MaterialsRheologyBbarEnum,VertexEnum);
	info = TaoCreate(PETSC_COMM_WORLD,&tao); if(info) _error_("STOP");
	//info = TaoSetType(tao,"tao_blmvm"); if(info) _error_("STOP");
	info = TaoSetType(tao,"tao_cg"); if(info) _error_("STOP");
	info = TaoSetInitialVector(tao,initial_solution);  if(info) _error_("STOP");
	info = TaoSetObjectiveRoutine(tao,FormFunction,(void*)&user);  if(info) _error_("STOP");
	info = TaoSetObjectiveAndGradientRoutine(tao,FormFunctionGradient,(void*)&user);  if(info) _error_("STOP");
	info = TaoSetFromOptions(tao);  if(info) _error_("STOP");
	/* http://www.mcs.anl.gov/research/projects/tao/docs/manpages/taosolver/TaoSetTolerances.html*/
	/*                          fatol, frtol, gatol, grtol, gttol*/
	info = TaoSetTolerances(tao,10e-18,10e-18,10e-18,10e-18,10e-18); if(info) _error_("STOP");
	info = TaoSolve(tao); if(info) _error_("STOP");

	/*Get solution status*/
	info = TaoGetSolutionStatus(tao,&iter,&ff,&gnorm,0,0,&reason); //CHKERRQ(info);
	switch(reason){
		/*http://www.mcs.anl.gov/research/projects/tao/docs/manpages/taosolver/TaoGetTerminationReason.html*/
		case TAO_DIVERGED_MAXITS:      _printf_(true,"TAO_DIVERGED_MAXITS (its>maxits)\n"); break;
		case TAO_DIVERGED_NAN:         _printf_(true,"TAO_DIVERGED_NAN (Numerical problems)\n"); break;
		case TAO_DIVERGED_MAXFCN:      _printf_(true,"TAO_DIVERGED_MAXFCN (nfunc > maxnfuncts)\n"); break;
		case TAO_DIVERGED_LS_FAILURE:  _printf_(true,"TAO_DIVERGED_LS_FAILURE (line search failure)\n"); break;
		case TAO_DIVERGED_TR_REDUCTION:_printf_(true,"TAO_DIVERGED_TR_REDUCTION \n"); break;
		case TAO_DIVERGED_USER:        _printf_(true,"TAO_DIVERGED_USER (user defined)\n"); break;
		default: _printf_(true,"unknown reason");
	}
	if (reason<=0){
		_printf_(true,"Try a different TAO method, adjust some parameters, or check the function evaluation routines\n");
		_printf_(true," Iterations: %d,  Function Value: %4.2e, Residual: %4.2e \n",iter,ff,gnorm);
	}
	else{
	 _printf_(true,"TAO found a solution");
	}
	info = TaoView(tao,PETSC_VIEWER_STDOUT_SELF);  if(info) _error_("STOP");

	/*Clean up*/
	info = TaoDestroy(&tao);  if(info) _error_("STOP");
	VecFree(&initial_solution);

	/* Finalize TAO */
	TaoFinalize();
}

int FormFunction(TaoSolver tao, Vec X, double *fcn,void *userCtx){
	AppCtx   *user     = (AppCtx*)userCtx;
	FemModel *femmodel = user->femmodel;

	int costfunction=SurfaceAbsVelMisfitEnum;
	femmodel->parameters->SetParam(&costfunction,1,1,StepResponsesEnum);
	InputUpdateFromVectorx(femmodel->elements,femmodel->nodes, femmodel->vertices, femmodel->loads, femmodel->materials, femmodel->parameters,X,MaterialsRheologyBbarEnum,VertexEnum);
	diagnostic_core(user->femmodel);
	CostFunctionx(fcn, femmodel->elements,femmodel->nodes, femmodel->vertices,femmodel->loads, femmodel->materials, femmodel->parameters);
	printf("f(x) = %g\n",*fcn);
	return 0;
}
int FormFunctionGradient(TaoSolver tao, Vec X, double *fcn,Vec G,void *userCtx){

	/*Retreive arguments*/
	AppCtx   *user     = (AppCtx*)userCtx;
	FemModel *femmodel = user->femmodel;
	Vec       gradient = NULL;

	int costfunction=SurfaceAbsVelMisfitEnum;
	femmodel->parameters->SetParam(&costfunction,1,1,StepResponsesEnum);
	InputUpdateFromVectorx(femmodel->elements,femmodel->nodes, femmodel->vertices, femmodel->loads, femmodel->materials, femmodel->parameters,X,MaterialsRheologyBbarEnum,VertexEnum);
	adjointdiagnostic_core(user->femmodel);
	Gradjx(&gradient, femmodel->elements,femmodel->nodes, femmodel->vertices,femmodel->loads, femmodel->materials,femmodel->parameters, MaterialsRheologyBbarEnum);
	VecScale(gradient,-10e7);
	//VecScale(gradient,-1.);
	VecCopy(gradient,G);
	CostFunctionx(fcn, femmodel->elements,femmodel->nodes, femmodel->vertices,femmodel->loads, femmodel->materials, femmodel->parameters);

	printf("f2(x) = %g\n",*fcn);
	return 0;
}
#else
void controltao_core(FemModel* femmodel){
	_error_("TAO not installed or PETSc version not supported");
}
#endif //_HAVE_TAO_ 
