%FLAIM class definition
%
%   Usage:
%      flaim=flaim();

classdef flaim
	properties (SetAccess=public) 
		targets            = modelfield('default','','marshall',false);
		tracks             = modelfield('default','','marshall',false);
		flightreqs         = modelfield('default',struct(),'marshall',false);
		criterion          = modelfield('default',NaN,'marshall',false);
		gridsatequator     = modelfield('default',200000,'marshall',false);
		usevalueordering   = modelfield('default',true,'marshall',false);
		split_antimeridian = modelfield('default',true,'marshall',false);
		solution           = modelfield('default','','marshall',false);
		quality            = modelfield('default',0,'marshall',false);
		path_optimize      = modelfield('default',false,'marshall',false);
		opt_ndir           = modelfield('default',1,'marshall',false);
		opt_dist           = modelfield('default',25,'marshall',false);
		opt_niter          = modelfield('default',30000,'marshall',false);
	end
	methods
		function obj = flaim(varargin) % {{{
			switch nargin
				case 0
					obj=setdefaultparameters(obj);
				case 1
					in=varargin{1};
					if (isa(in,'numeric') & in==0),
						% requesting templates do nothing
					else
						error('constructor not supported');
					end 
				otherwise
					error('constructor not supported');
			end
		end % }}}
		function obj = setdefaultparameters(obj) % {{{

			%first, use the defaults provided by the properties definition above. 
			fieldnames=fields(obj);
			for i=1:length(fieldnames),
				fieldname=fieldnames{i};
				obj.(fieldname)=obj.(fieldname).default;
			end

		end % }}}
		function checkconsistency(obj,md,solution,analyses) % {{{

			%Early return
			if solution~=FlaimSolutionEnum, return; end

			checkfield(md,'flaim.tracks','file',1);
			if any(isnan(md.flaim.criterion)) || isempty(md.flaim.criterion)
				checkfield(md,'flaim.targets','file',1);
			else
				checkfield(md,'flaim.criterion','numel',[md.mesh.numberofvertices md.mesh.numberofelements]);
			end

		end % }}}
		function disp(obj) % {{{

			disp(sprintf('   FLAIM - Flight Line Adaptation using Ice sheet Modeling:')); 

			disp(sprintf('\n      Input:'));
			fielddisplay(obj,'targets'            ,'name of kml output targets file ');
			fielddisplay(obj,'tracks'             ,'name of kml input tracks file ');
			fielddisplay(obj,'flightreqs'         ,'structure of kml flight requirements (not used yet)');
			fielddisplay(obj,'criterion'          ,'element or nodal criterion for flight path evaluation (metric)');

			disp(sprintf('\n      Arguments:'));
			fielddisplay(obj,'gridsatequator'     ,'number of grids at equator (determines resolution)');
			fielddisplay(obj,'usevalueordering'   ,'flag to consider target values for flight path evaluation');
			fielddisplay(obj,'split_antimeridian' ,'flag to split polygons on the antimeridian');
			
			disp(sprintf('\n      Optimization:'));
			fielddisplay(obj,'path_optimize'     ,'optimize? (default false)');
			fielddisplay(obj,'opt_ndir'     ,{'number of directions to test when moving a point.  If this value = 1, a random direction is tested.',...
											  'A value > 1 results in directions equally spaced from [0, 2*PI] being tested.',...
											  'For example, 4 would result in directions [0, PI/2, PI, 3PI/2].'});
			fielddisplay(obj,'opt_dist'     ,'specifies the distance in km (default 25) to move a randomly selected path point on each iteration');
			fielddisplay(obj,'opt_niter'     ,{'number of iterations (default 30,000) to run for flightplan optimization',...
											   'i.e. the number of times to randomly select a point and move it.'});

			disp(sprintf('\n      Output:'));
			fielddisplay(obj,'solution'           ,'name of kml solution file');
			fielddisplay(obj,'quality'            ,'quality of kml solution');
		end % }}}
	end
end
