%MESH class definition
%
%   Usage:
%      mesh=mesh();

classdef mesh
	properties (SetAccess=public) 
		x                           = modelfield('default',NaN,'marshall',true,'format','DoubleMat','mattype',1);
		y                           = modelfield('default',NaN,'marshall',true,'format','DoubleMat','mattype',1);
		z                           = modelfield('default',NaN,'marshall',true,'format','DoubleMat','mattype',1);
		elements                    = modelfield('default',NaN,'marshall',true,'format','DoubleMat','mattype',2);
		dimension                   = modelfield('default',0,'marshall',true,'format','Integer');
		numberoflayers              = modelfield('default',0,'marshall',true,'format','Integer');
		numberofelements            = modelfield('default',0,'marshall',true,'format','Integer');
		numberofvertices            = modelfield('default',0,'marshall',true,'format','Integer');
		numberofedges               = modelfield('default',0,'marshall',true,'format','Integer');
		
		lat                         = modelfield('default',NaN,'marshall',false);
		long                        = modelfield('default',NaN,'marshall',false);
		hemisphere                  = modelfield('default',NaN,'marshall',false);

		elementonbed                = modelfield('default',NaN,'marshall',true,'format','BooleanMat','mattype',2);
		elementonsurface            = modelfield('default',NaN,'marshall',true,'format','BooleanMat','mattype',2);
		vertexonbed                 = modelfield('default',NaN,'marshall',true,'format','BooleanMat','mattype',1);
		vertexonsurface             = modelfield('default',NaN,'marshall',true,'format','BooleanMat','mattype',1);
		lowerelements               = modelfield('default',NaN,'marshall',true,'format','DoubleMat','mattype',2);
		lowervertex                 = modelfield('default',NaN,'marshall',false);
		upperelements               = modelfield('default',NaN,'marshall',true,'format','DoubleMat','mattype',2);
		uppervertex                 = modelfield('default',NaN,'marshall',false);
		vertexonboundary            = modelfield('default',NaN,'marshall',false);

		edges                       = modelfield('default',NaN,'marshall',true,'format','DoubleMat','mattype',3);
		segments                    = modelfield('default',NaN,'marshall',false);
		segmentmarkers              = modelfield('default',NaN,'marshall',false);
		vertexconnectivity          = modelfield('default',NaN,'marshall',false);
		elementconnectivity         = modelfield('default',NaN,'marshall',true,'format','DoubleMat','mattype',3);
		average_vertex_connectivity = modelfield('default',0,'marshall',true,'format','Integer');

		x2d                         = modelfield('default',NaN,'marshall',false);
		y2d                         = modelfield('default',NaN,'marshall',false);
		elements2d                  = modelfield('default',NaN,'marshall',true,'format','DoubleMat','mattype',3);
		numberofvertices2d          = modelfield('default',0,'marshall',true,'format','Integer');
		numberofelements2d          = modelfield('default',0,'marshall',true,'format','Integer');

		extractedvertices           = modelfield('default',NaN,'marshall',false);
		extractedelements           = modelfield('default',NaN,'marshall',false);
	end
	methods
		function obj = mesh(varargin) % {{{
			switch nargin
				case 0
					obj=setdefaultparameters(obj);
				case 1
					in=varargin{1};
					if (isa(in,'numeric') & in==0),
						% requesting templates do nothing
					else
						error('constructor not supported');
					end 
				otherwise
					error('constructor not supported');
			end
		end % }}}
		function obj = setdefaultparameters(obj) % {{{

			%first, use the defaults provided by the properties definition above. 
			fieldnames=fields(obj);
			for i=1:length(fieldnames),
				fieldname=fieldnames{i};
				obj.(fieldname)=obj.(fieldname).default;
			end

			%the connectivity is the avergaded number of nodes linked to a
			%given node through an edge. This connectivity is used to initially
			%allocate memory to the stiffness matrix. A value of 16 seems to
			%give a good memory/time ration. This value can be checked in
			%trunk/test/Miscellaneous/runme.m
			obj.average_vertex_connectivity=25;
		end % }}}
		function checkconsistency(obj,md,solution,analyses) % {{{

			checkfield(md,'mesh.x','NaN',1,'size',[md.mesh.numberofvertices 1]);
			checkfield(md,'mesh.y','NaN',1,'size',[md.mesh.numberofvertices 1]);
			checkfield(md,'mesh.z','NaN',1,'size',[md.mesh.numberofvertices 1]);
			checkfield(md,'mesh.elements','NaN',1,'>',0,'values',1:md.mesh.numberofvertices);
			if(md.mesh.dimension==2),
				checkfield(md,'mesh.elements','size',[md.mesh.numberofelements 3]);
			else
				checkfield(md,'mesh.elements','size',[md.mesh.numberofelements 6]);
			end
			if any(~ismember(1:md.mesh.numberofvertices,sort(unique(md.mesh.elements(:)))));
				checkmessage('orphan nodes have been found. Check the mesh outline');
			end
			checkfield(md,'mesh.dimension','values',[2 3]);
			checkfield(md,'mesh.numberoflayers','>=',0);
			checkfield(md,'mesh.numberofelements','>',0);
			checkfield(md,'mesh.numberofvertices','>',0);
			%no checks for numberofedges lat long and hemisphere
			checkfield(md,'mesh.elementonbed','size',[md.mesh.numberofelements 1],'values',[0 1]);
			checkfield(md,'mesh.elementonsurface','size',[md.mesh.numberofelements 1],'values',[0 1]);
			checkfield(md,'mesh.vertexonbed','size',[md.mesh.numberofvertices 1],'values',[0 1]);
			checkfield(md,'mesh.vertexonsurface','size',[md.mesh.numberofvertices 1],'values',[0 1]);
			if (md.mesh.dimension==2),
				checkfield(md,'mesh.average_vertex_connectivity','>=',9,'message','''mesh.average_vertex_connectivity'' should be at least 9 in 2d');
			else
				checkfield(md,'mesh.average_vertex_connectivity','>=',24,'message','''mesh.average_vertex_connectivity'' should be at least 24 in 3d');
			end

			%Solution specific checks
			switch(solution),
				case PrognosticSolutionEnum,
					if md.prognostic.stabilization==3,
						checkfield(md,'mesh.dimension','values',2,'message','Discontinuous Galerkin only supported for 2d meshes');
						checkfield(md,'mesh.edges','size',[NaN 4]);
						checkfield(md,'mesh.edges(:,1:3)','>',0);
					end
				case BalancethicknessSolutionEnum,
					if md.balancethickness.stabilization==3,
						checkfield(md,'mesh.dimension','values',2,'message','Discontinuous Galerkin only supported for 2d meshes');
						checkfield(md,'mesh.edges','size',[NaN 4]);
						checkfield(md,'mesh.edges(:,1:3)','>',0);
					end
				case TransientSolutionEnum,
					if md.transient.isprognostic & md.prognostic.stabilization==3,
						checkfield(md,'mesh.dimension','values',2,'message','Discontinuous Galerkin only supported for 2d meshes');
						checkfield(md,'mesh.edges','size',[NaN 4]);
						checkfield(md,'mesh.edges(:,1:3)','>',0);
					end
				case ThermalSolutionEnum,
					checkfield(md,'mesh.dimension','values',3,'message','thermal solution only supported on 3d meshes');
			end
		end % }}}
		function disp(obj) % {{{
			disp(sprintf('   Mesh:')); 

			if obj.dimension==3,
				disp(sprintf('\n      Elements and vertices of the original 2d mesh:'));
				fielddisplay(obj,'numberofelements2d','number of elements');
				fielddisplay(obj,'numberofvertices2d','number of vertices');
				fielddisplay(obj,'elements2d','index into (x,y,z), coordinates of the vertices');
				fielddisplay(obj,'x2d','vertices x coordinate');
				fielddisplay(obj,'y2d','vertices y coordinate');

				disp(sprintf('\n      Elements and vertices of the extruded 3d mesh:'));
			else
				disp(sprintf('\n      Elements and vertices:'));
			end
			fielddisplay(obj,'numberofelements','number of elements');
			fielddisplay(obj,'numberofvertices','number of vertices');
			fielddisplay(obj,'elements','index into (x,y,z), coordinates of the vertices');
			fielddisplay(obj,'x','vertices x coordinate');
			fielddisplay(obj,'y','vertices y coordinate');
			fielddisplay(obj,'z','vertices z coordinate');
			fielddisplay(obj,'edges','edges of the 2d mesh (vertex1 vertex2 element1 element2)');
			fielddisplay(obj,'numberofedges','number of edges of the 2d mesh');

			disp(sprintf('\n      Properties:'));
			fielddisplay(obj,'dimension','mesh dimension (2d or 3d)');
			fielddisplay(obj,'numberoflayers','number of extrusion layers');
			fielddisplay(obj,'vertexonbed','lower vertices flags list');
			fielddisplay(obj,'elementonbed','lower elements flags list');
			fielddisplay(obj,'vertexonsurface','upper vertices flags list');
			fielddisplay(obj,'elementonsurface','upper elements flags list');
			fielddisplay(obj,'uppervertex','upper vertex list (NaN for vertex on the upper surface)');
			fielddisplay(obj,'upperelements','upper element list (NaN for element on the upper layer)');
			fielddisplay(obj,'lowervertex','lower vertex list (NaN for vertex on the lower surface)');
			fielddisplay(obj,'lowerelements','lower element list (NaN for element on the lower layer');
			fielddisplay(obj,'vertexonboundary','vertices on the boundary of the domain flag list');
			
			fielddisplay(obj,'segments','edges on domain boundary (vertex1 vertex2 element)');
			fielddisplay(obj,'segmentmarkers','number associated to each segment');
			fielddisplay(obj,'vertexconnectivity','list of vertices connected to vertex_i');
			fielddisplay(obj,'elementconnectivity','list of vertices connected to element_i');
			fielddisplay(obj,'average_vertex_connectivity','average number of vertices connected to one vertex');

			disp(sprintf('\n      Extracted model:'));
			fielddisplay(obj,'extractedvertices','vertices extracted from the model');
			fielddisplay(obj,'extractedelements','elements extracted from the model');

			disp(sprintf('\n      Projection:'));
			fielddisplay(obj,'lat','vertices latitude');
			fielddisplay(obj,'long','vertices longitude');
			fielddisplay(obj,'hemisphere','Indicate hemisphere ''n'' or ''s'' ');
		end % }}}
	end
end
