/*!\file: solver_thermal_nonlinear.cpp
 * \brief: core of the thermal solution 
 */ 

#include "../toolkits/toolkits.h"
#include "../objects/objects.h"
#include "../io/io.h"
#include "../EnumDefinitions/EnumDefinitions.h"
#include "../modules/modules.h"

void solver_thermal_nonlinear(FemModel* femmodel){

	/*solution : */
	Vec tg=NULL; 
	Vec tf=NULL; 
	Vec tf_old=NULL; 
	Vec ys=NULL; 
	double melting_offset;

	/*intermediary: */
	Mat Kff=NULL;
	Mat Kfs=NULL;
	Vec pf=NULL;
	Vec df=NULL;

	bool converged;
	int constraints_converged;
	int num_unstable_constraints;
	int count;
	int thermal_penalty_threshold;
	int thermal_maxiter;
	bool reset_penalties;

	/*parameters:*/
	int kflag,pflag;
	bool lowmem=0;
	int  configuration_type;


	/*Recover parameters: */
	kflag=1; pflag=1;
	femmodel->parameters->FindParam(&lowmem,SettingsLowmemEnum);
	femmodel->parameters->FindParam(&thermal_penalty_threshold,ThermalPenaltyThresholdEnum);
	femmodel->parameters->FindParam(&configuration_type,ConfigurationTypeEnum);
	femmodel->parameters->FindParam(&thermal_maxiter,ThermalMaxiterEnum);

	count=1;
	converged=false;

	_printf_(VerboseSolution(),"%s\n","starting direct shooting method");
	InputUpdateFromConstantx( femmodel->elements,femmodel->nodes, femmodel->vertices, femmodel->loads, femmodel->materials, femmodel->parameters,reset_penalties,ResetPenaltiesEnum);
	InputUpdateFromConstantx( femmodel->elements,femmodel->nodes, femmodel->vertices, femmodel->loads, femmodel->materials, femmodel->parameters,false,ConvergedEnum);
	UpdateConstraintsx(femmodel->nodes,femmodel->constraints,femmodel->parameters);

	for(;;){

		SystemMatricesx(&Kff, &Kfs, &pf,&df, &melting_offset,femmodel->elements,femmodel->nodes,femmodel->vertices,femmodel->loads,femmodel->materials,femmodel->parameters);
		CreateNodalConstraintsx(&ys,femmodel->nodes,configuration_type);
		Reduceloadx(pf, Kfs, ys); MatFree(&Kfs); VecFree(&tf);
		Solverx(&tf, Kff, pf,tf_old, df, femmodel->parameters);
		VecFree(&tf_old); VecDuplicatePatch(&tf_old,tf);
		MatFree(&Kff);VecFree(&pf);VecFree(&tg); VecFree(&df);
		Mergesolutionfromftogx(&tg, tf,ys,femmodel->nodes,femmodel->parameters); VecFree(&ys);
		InputUpdateFromSolutionx(femmodel->elements,femmodel->nodes, femmodel->vertices, femmodel->loads, femmodel->materials, femmodel->parameters,tg);

		ConstraintsStatex(&constraints_converged, &num_unstable_constraints, femmodel->elements,femmodel->nodes,femmodel->vertices,femmodel->loads,femmodel->materials,femmodel->parameters);

		if (!converged){
			_printf_(VerboseConvergence(),"%s%i\n","   #unstable constraints = ",num_unstable_constraints);
			if (num_unstable_constraints <= thermal_penalty_threshold)converged=true;
			if (count>=thermal_maxiter){
				converged=true;
				_printf_(true,"   maximum number of iterations (%i) exceeded\n",thermal_maxiter); 
			}
		}
		count++;

		InputUpdateFromConstantx( femmodel->elements,femmodel->nodes, femmodel->vertices, femmodel->loads, femmodel->materials, femmodel->parameters,converged,ConvergedEnum);
		
		if(converged)break;
	}

	InputUpdateFromSolutionx(femmodel->elements,femmodel->nodes, femmodel->vertices, femmodel->loads, femmodel->materials, femmodel->parameters,tg);
	InputUpdateFromConstantx( femmodel->elements,femmodel->nodes, femmodel->vertices, femmodel->loads, femmodel->materials, femmodel->parameters,melting_offset,MeltingOffsetEnum);

	/*Free ressources: */
	VecFree(&tg);
	VecFree(&tf);
	VecFree(&tf_old);
	VecFree(&ys);
}
