/*\file InterpFromMeshToMesh2d.c
 *\brief: bamg module.
 */
#include "./InterpFromMeshToMesh2d.h"

void mexFunction( int nlhs, mxArray* plhs[], int nrhs, const mxArray* prhs[]){

	int i;

	/*input: */
	double* index=NULL;
	int     index_cols;

	double* x_data=NULL;
	int     x_data_rows;

	double* y_data=NULL;
	int     y_data_rows;

	double* data=NULL; 
	int     data_rows;
	int     data_cols;

	double* x_interp=NULL;
	double* y_interp=NULL;

	int     x_interp_rows;
	int     y_interp_rows;

	double* default_values=NULL;
	int     num_default_values=0;

	//contours
	mxArray*  matlabstructure=NULL;
	Contour** contours=NULL;
	int       numcontours;
	Contour*  contouri=NULL;

	/*Intermediary*/
	int nods_data;
	int nels_data;
	int nods_interp;
	int verbose=0;

	/* output: */
	double* data_interp=NULL;

	/*Boot module: */
	MODULEBOOT();

	/*checks on arguments on the matlab side: */
	if(nlhs!=NLHS){
		InterpFromMeshToMesh2dUsage();
		_error_("InterpFromMeshToMesh2dUsage usage error");
	}
	if((nrhs!=6) & (nrhs!=8)){
		InterpFromMeshToMesh2dUsage();
		_error_("InterpFromMeshToMesh2dUsage usage error");
	}

	/*Input datasets: */
	if (verbose) printf("Fetching inputs\n");
	FetchMatlabData(&index,&nels_data,&index_cols,INDEXHANDLE);
	FetchMatlabData(&x_data,&x_data_rows,NULL,XHANDLE);
	FetchMatlabData(&y_data,&y_data_rows,NULL,YHANDLE);
	FetchMatlabData(&data,&data_rows,&data_cols,DATAHANDLE);
	FetchMatlabData(&x_interp,&x_interp_rows,NULL,XINTERPHANDLE);
	FetchMatlabData(&y_interp,&y_interp_rows,NULL,YINTERPHANDLE);

	if(nrhs==8){
		
		/*Call expread on filename to build a contour array in the matlab workspace: */
		mexCallMATLAB( 1, &matlabstructure, 1, (mxArray**)&FILENAME, "expread");

		/*default values: */
		FetchMatlabData(&default_values,&num_default_values,DEFAULTHANDLE);

		/*contours: */
		numcontours=mxGetNumberOfElements(matlabstructure);
		contours=(Contour**)xmalloc(numcontours*sizeof(Contour*));
		for(i=0;i<numcontours;i++){
			//allocate
			contouri=(Contour*)xmalloc(sizeof(Contour));
			//retrieve dimension of this contour.
			contouri->nods=(int)mxGetScalar(mxGetField(matlabstructure,i,"nods"));
			//set pointers.
			contouri->x=mxGetPr(mxGetField(matlabstructure,i,"x"));
			contouri->y=mxGetPr(mxGetField(matlabstructure,i,"y"));
			*(contours+i)=contouri;
		}

		/* Debugging of contours :{{{1*/
		/*for(i=0;i<numcontours;i++){
		  printf("\nContour echo: contour number  %i / %i\n",i+1,numcontours);
		  contouri=*(contours+i);
		  printf("   Number of vertices %i\n",contouri->nods);
		  for (j=0;j<contouri->nods;j++){
		  printf("   %lf %lf\n",*(contouri->x+j),*(contouri->y+j));
		  }
		  }*/
		/*}}}*/
	}
	else{
		num_default_values=0;
		default_values=NULL;
		numcontours=0;
		contours=NULL;
	}

	/*some checks*/
	if (verbose) printf("Checking inputs\n");
	if (x_data_rows!=y_data_rows){
		_error_("vectors x and y should have the same length!");
	}
	if (x_interp_rows!=y_interp_rows){
		_error_("vectors x_interp and y_interp should have the same length!");
	}
	if (index_cols!=3){
		_error_("index should have 3 columns (input provided has %i columns)",index_cols);
	}

	/*get number of elements and number of nodes in the data*/
	nods_data=x_data_rows;
	nods_interp=x_interp_rows;

	/* Run core computations: */
	if (verbose) printf("Call core\n");
	InterpFromMeshToMesh2dx(&data_interp,index,x_data,y_data,nods_data,nels_data,data,data_rows,data_cols,x_interp,y_interp,nods_interp,default_values,num_default_values,contours,numcontours);

	/*Write data: */
	WriteMatlabData(DATAINTERP,data_interp,nods_interp,data_cols);

	/*end module: */
	MODULEEND();
}

void InterpFromMeshToMesh2dUsage(void)
{
	_printf_(true,"INTERFROMMESHTOMESH2D - interpolation from a 2d triangular mesh onto a list of point\n");
	_printf_(true,"\n");
	_printf_(true,"   This function is a multi-threaded mex file that interpolates a field\n");
	_printf_(true,"   defined on a triangular mesh onto a list of point\n");
	_printf_(true,"\n");
	_printf_(true,"   Usage:\n");
	_printf_(true,"         data_interp=InterpFromMeshToMesh2d(index,x,y,data,x_interp,y_interp);\n");
	_printf_(true,"      or data_interp=InterpFromMeshToMesh2d(index,x,y,data,x_interp,y_interp,default_value,contourname);\n");
	_printf_(true,"\n");
	_printf_(true,"      index: index of the mesh where data is defined\n");
	_printf_(true,"      x,y: coordinates of the nodes where data is defined\n");
	_printf_(true,"      data: matrix holding the data to be interpolated onto the mesh. (one column per field)\n");
	_printf_(true,"      x_interp,y_interp: coordinates of the points onto which we interpolate.\n");
	_printf_(true,"      if default_value and contourname not specified: linear interpolation will happen on all x_interp,y_interp.\n");
	_printf_(true,"      if (default_value,contourname) specified: linear interpolation will happen on all x_interp,y_interp inside the contour, default value will be adopted on the rest of the mesh.\n");
	_printf_(true,"      note that default_value is either a scalar, or a vector of size  length(x_interp)\n");
	_printf_(true,"      data_interp: vector of mesh interpolated data.\n");
	_printf_(true,"\n");
	_printf_(true,"   Example:\n");
	_printf_(true,"      load('temperature.mat');\n");
	_printf_(true,"      md.temperature=InterpFromMeshToMesh2d(index,x,y,temperature,md.x,md.y);\n");
	_printf_(true,"      md.temperature=InterpFromMeshToMesh2d(index,x,y,temperature,md.x,md.y,253,'Contour.exp');\n");
	_printf_(true,"\n");
}
