%MATERIALS class definition
%
%   Usage:
%      materials=materials();

classdef materials
	properties (SetAccess=public) 
		rho_ice                    = modelfield('default',0,'marshall',true,'format','Double');
		rho_water                  = modelfield('default',0,'marshall',true,'format','Double');
		mu_water                   = modelfield('default',0,'marshall',true,'format','Double');
		heatcapacity               = modelfield('default',0,'marshall',true,'format','Double');
		latentheat                 = modelfield('default',0,'marshall',true,'format','Double');
		thermalconductivity        = modelfield('default',0,'marshall',true,'format','Double');
		meltingpoint               = modelfield('default',0,'marshall',true,'format','Double');
		beta                       = modelfield('default',0,'marshall',true,'format','Double');
		mixed_layer_capacity       = modelfield('default',0,'marshall',true,'format','Double');
		thermal_exchange_velocity  = modelfield('default',0,'marshall',true,'format','Double');
		rheology_B   = modelfield('default',NaN,'marshall',true,'format','DoubleMat','mattype',1);
		rheology_n   = modelfield('default',NaN,'marshall',true,'format','DoubleMat','mattype',2);
		rheology_law = modelfield('default','','marshall',true,'preprocess','StringToEnum','format','Integer');
	end
	methods
		function obj = materials(varargin) % {{{
			switch nargin
				case 0
					obj=setdefaultparameters(obj);
				case 1
					in=varargin{1};
					if (isa(in,'numeric') & in==0),
						% requesting templates do nothing
					else
						error('constructor not supported');
					end 
				otherwise
					error('constructor not supported');
			end
		end % }}}
		function obj = setdefaultparameters(obj) % {{{

			%first, use the defaults provided by the properties definition above. 
			fieldnames=fields(obj);
			for i=1:length(fieldnames),
				fieldname=fieldnames{i};
				obj.(fieldname)=obj.(fieldname).default;
			end

			%ice density (kg/m^3)
			obj.rho_ice=917;

			%water density (kg/m^3)
			obj.rho_water=1023;

			%water viscosity (N.s/m^2)
			obj.mu_water=0.001787;  

			%ice heat capacity cp (J/kg/K)
			obj.heatcapacity=2093;

			%ice latent heat of fusion L (J/kg)
			obj.latentheat=3.34*10^5;

			%ice thermal conductivity (W/m/K)
			obj.thermalconductivity=2.4;

			%the melting point of ice at 1 atmosphere of pressure in K
			obj.meltingpoint=273.15;

			%rate of change of melting point with pressure (K/Pa)
			obj.beta=9.8*10^-8;

			%mixed layer (ice-water interface) heat capacity (J/kg/K)
			obj.mixed_layer_capacity=3974;

			%thermal exchange velocity (ice-water interface) (m/s)
			obj.thermal_exchange_velocity=1.00*10^-4;

			%Rheology law: what is the temperature dependence of B with T
			%available: none, paterson and arrhenius
			obj.rheology_law='Paterson';
		end % }}}
		function checkconsistency(obj,md,solution,analyses) % {{{
			checkfield(md,'materials.rho_ice','>',0);
			checkfield(md,'materials.rho_water','>',0);
			checkfield(md,'materials.mu_water','>',0);
			checkfield(md,'materials.rheology_B','>',0,'size',[md.mesh.numberofvertices 1]);
			checkfield(md,'materials.rheology_n','>',0,'size',[md.mesh.numberofelements 1]);
			checkfield(md,'materials.rheology_law','values',{'None' 'Paterson' 'Arrhenius'});
		end % }}}
		function disp(obj) % {{{
			disp(sprintf('   Materials:\n'));

			fielddisplay(obj,'rho_ice','ice density [kg/m^3]');
			fielddisplay(obj,'rho_water','water density [kg/m^3]');
			fielddisplay(obj,'mu_water','water viscosity [N s/m^2]');
			fielddisplay(obj,'heatcapacity','heat capacity [J/kg/K]');
			fielddisplay(obj,'thermalconductivity','ice thermal conductivity [W/m/K]');
			fielddisplay(obj,'meltingpoint','melting point of ice at 1atm in K');
			fielddisplay(obj,'latentheat','latent heat of fusion [J/m^3]');
			fielddisplay(obj,'beta','rate of change of melting point with pressure [K/Pa]');
			fielddisplay(obj,'mixed_layer_capacity','mixed layer capacity [W/kg/K]');
			fielddisplay(obj,'thermal_exchange_velocity','thermal exchange velocity [m/s]');
			fielddisplay(obj,'rheology_B','flow law parameter [Pa/s^(1/n)]');
			fielddisplay(obj,'rheology_n','Glen''s flow law exponent');
			fielddisplay(obj,'rheology_law','law for the temperature dependance of the rheology: ''None'', ''Paterson'' or ''Arrhenius''');
		end % }}}
	end
end
