%HYDROLOGY class definition
%
%   Usage:
%      hydrology=hydrology();

classdef hydrology
	properties (SetAccess=public) 
		spcwatercolumn = modelfield('default',NaN,'marshall',true,'format','DoubleMat','mattype',1);
		n              = modelfield('default',0,'marshall',true,'format','Double'); 
		CR             = modelfield('default',0,'marshall',true,'format','Double');
		p              = modelfield('default',0,'marshall',true,'format','Double');
		q              = modelfield('default',0,'marshall',true,'format','Double');
		kn             = modelfield('default',0,'marshall',true,'format','Double');
		stabilization  = modelfield('default',0,'marshall',true,'format','Double');
	end
	methods
		function obj = hydrology(varargin) % {{{
			switch nargin
				case 0
					obj=setdefaultparameters(obj);
				case 1
					in=varargin{1};
					if (isa(in,'numeric') & in==0),
						% requesting templates do nothing
					else
						error('constructor not supported');
					end 
				otherwise
					error('constructor not supported');
			end
		end % }}}
		function obj = setdefaultparameters(obj) % {{{

			%first, use the defaults provided by the properties definition above. 
			fieldnames=fields(obj);
			for i=1:length(fieldnames),
				fieldname=fieldnames{i};
				obj.(fieldname)=obj.(fieldname).default;
			end

			%Parameters from Johnson's 2002 thesis, section 3.5.4			 
			obj.n=.02;			 
			obj.CR=.01;
			obj.p=2;		 
			obj.q=1;		 
			obj.kn=0;

			%Type of stabilization to use 0:nothing 1:artificial_diffusivity
			obj.stabilization=1;
		end % }}}
		function checkconsistency(obj,md,solution,analyses) % {{{

			%Early return
			if ~ismember(HydrologyAnalysisEnum,analyses), return; end

			checkfield(md,'hydrology.spcwatercolumn','forcing',1);
			checkfield(md,'hydrology.stabilization','>=',0);
		end % }}}
		function disp(obj) % {{{
			disp(sprintf('   hydrology solution parameters:'));

			fielddisplay(obj,'spcwatercolumn','water thickness constraints (NaN means no constraint)');
			fielddisplay(obj,'n','Manning roughness coefficient');
			fielddisplay(obj,'CR','turtuosity parameter');
			fielddisplay(obj,'p','dimensionless exponent in Manning velocity formula');
			fielddisplay(obj,'q','dimensionless exponent in Manning velocity formula');
			fielddisplay(obj,'kn','parameter in effective pressure formula');
			fielddisplay(obj,'stabilization','artificial diffusivity (default is 1). can be more than 1 to increase diffusivity.');

		end % }}}
	end
end
