#!/bin/bash
#This bash script calls the dailyrun.m matlab file to run our daily test decks. 
#It then processes the results and sends an email to the Ice developpers.

#Hard coded options
NUMCPUS_RUN=7
NROPTIONS=""

#some functions
function timer() #{{{1
{
	if [[ $# -eq 0 ]]; then
		echo $(date '+%s')
	else
		local  stime=$1
		etime=$(date '+%s')

		if [[ -z "$stime" ]]; then stime=$etime; fi

		dt=$((etime - stime))
		ds=$((dt % 60))
		dm=$(((dt / 60) % 60))
		dh=$((dt / 3600))
		printf '%d:%02d:%02d' $dh $dm $ds
	fi
} #}}}
function todaydate() #{{{1
{
	suffix=`date | awk '{printf("%s-%s-%s  %s",$2,$3,$6,$4);}'`			 
	echo $suffix;			 
} #}}}
function host_name() #{{{1
{
	#return host name depending on the OS
	if [ "$1" = "winxp32" ] 
	then
		HOST_NAME=`hostname`;
	else
		HOST_NAME=`hostname -s`;
	fi
	echo $HOST_NAME;
} #}}}

#Get configuration
#Initialize variables {{{1
TODAY=$(todaydate);
HOST_NAME=$(host_name $OS);
START_TIME=$(timer);
ISSM_RELEASE="N/A"
USER=$(whoami);
INIT_PATH=$(pwd);
#}}}

#Prepare run
#Windows hack for startup.m {{{1
#windows environments: ISSM_TIER_WIN variable not correctly picked up when using 
#the cron job. just get startup to take the ISSM_TIER variable as the pwd:
if [ "$OS" = "winxp32" ]
then
	cat startup.m | sed 's/clear status/clear status; ISSM_TIER=pwd;/g' > startup.m.bak
	mv startup.m.bak startup.m
fi
#}}}
#create softlink to startup {{{1
cd $ISSM_TIER/test/NightlyRun/
ln -s $ISSM_TIER/startup.m .
#}}}
#Create dailylog directory and info.log {{{1
#put installation elapsed time in info.log
INSTALL_TIME=$(timer)
ELAPSED_INSTALL=$(timer $START_TIME)
rm -rf $ISSM_TIER/dailylog
mkdir  $ISSM_TIER/dailylog
cat << END > $ISSM_TIER/dailylog/info.log
today:     $(echo $TODAY)
user:      $(echo $USER)
host:      $(echo $HOST_NAME)
OS:        N/A
release:   N/A
init_path: $(echo $INIT_PATH)
elapsed_install: $(echo $ELAPSED_INSTALL)
END
#}}}
#check NUMCPUS_RUN options {{{1
if [ "$NUMCPUS_RUN" = "" ]
then
	echo "NUMCPUS_RUN option not found, defaulting to NUMCPUS_RUN = 1"
	NUMCPUS_RUN=1
fi
#}}}

#Run tests
#Launch all tests on different cpus {{{1
for (( i=1;i<=$NUMCPUS_RUN;i++ ))
do
	#Launch matlab and the daily run script
	cat > $ISSM_TIER/dailylog/matlab_run$i.m << EOF
	warning off %necessary to avoid a info.log of several Go for parallel runs
	try,
	cd $ISSM_TIER/test/NightlyRun
	startup;
	$(if [ "$NROPTIONS" = ""  ]
	then
		echo "runme('output','daily','rank',$i,'numprocs',$NUMCPUS_RUN);"
	else
		echo "runme($NROPTIONS,'output','daily','rank',$i,'numprocs',$NUMCPUS_RUN);"
	fi
	)
	catch me,
		%An error occured, get report and exit
		directory=strsplit(pwd,'/');
		message=getReport(me)
		fid=fopen([ISSM_TIER '/dailylog/matlaberror.log'], 'at');
		fprintf(fid,'\nMatlab error occured in: %s\n\n',directory{end});
		fprintf(fid,'%s',message);
		fclose(fid);
	end
	disp('MATLABEXITEDCORRECTLY');
	exit
EOF

	#Start run from dailylog directory
	cd $ISSM_TIER/dailylog/

	#Start test
	MATLAB_VERSION="7.6" #7.2,7.4,7.6 and 7.8
	/usr/local/pkgs/matlab-$MATLAB_VERSION/bin/matlab  -nojvm -nosplash  -r matlab_run$i -logfile matlab_log$i.log &
done

#wait until matlab closes
wait
#}}}
#concatenate all reports {{{1
cd $ISSM_TIER/dailylog/
mv matlab_log1.log  matlab_log.log
for (( i=2;i<=$NUMCPUS_RUN;i++ ))
do
	cat matlab_log.log matlab_log$i.log > matlab_log.log.bak
	mv matlab_log.log.bak matlab_log.log
done
#}}}
#Complete info.log {{{1
if [ $(cat matlab_log.log | grep "MATLABEXITEDCORRECTLY" | wc -l) -eq $NUMCPUS_RUN ]
then
	MATLABCRASH=0
else
	MATLABCRASH=1
fi
ELAPSED_RUN=$(timer $INSTALL_TIME)
ELAPSED_TOTAL=$(timer $START_TIME)
cat << END >>  $ISSM_TIER/dailylog/info.log
elapsed_run:   $(echo $ELAPSED_RUN)
elapsed_total: $(echo $ELAPSED_TOTAL)
matlab_crash:  $(echo $MATLABCRASH)
END
#}}}

#Send Report
#Build html report {{{1
cd $ISSM_TIER/dailylog/
sh ../scripts/report.sh
ln -s $ISSM_TIER/dailylog/report.html $INIT_PATH
echo "html report located in $ISSM_TIER/dailylog/report.html and $INIT_PATH/report.html"
#}}}
