/*! \file
  
  \brief Abstract class for solving context-sensative dataflow analysis problems on ICFGs.

  \authors Michelle Strout, Barbara Kreaseck
  \version $Id: ICFGDFSolver.hpp,v 1.2 2005/06/10 02:32:04 mstrout Exp $

  Copyright (c) 2002-2005, Rice University <br>
  Copyright (c) 2004-2005, University of Chicago <br>
  Copyright (c) 2006, Contributors <br>
  All rights reserved. <br>
  See ../../../Copyright.txt for details. <br>


  To use this for performing an interprocedural flow sensitive
  but context insenstive dataflow analysis problems by subclassing
  ICFGCSDFProblem and implementing the required call backs.
*/

#ifndef ICFGCSDFSolver_H
#define ICFGCSDFSolver_H

#include <OpenAnalysis/Utils/OA_ptr.hpp>
//#include <OpenAnalysis/DataFlow/DGraphIterativeDFP.hpp>
#include <OpenAnalysis/DataFlow/DGraphSolverDFP.hpp>
#include <OpenAnalysis/DataFlow/DataFlowSet.hpp>
#include <OpenAnalysis/DataFlow/ICFGCSDFProblem.hpp>
#include <OpenAnalysis/DataFlow/CC2DFSMap.hpp>
#include <OpenAnalysis/ICFG/ICFGInterface.hpp>
//#include <OpenAnalysis/IRInterface/CFGIRInterface.hpp>
#include <OpenAnalysis/Alias/CCSetPerProc.hpp>
#include <map>

namespace OA {
  namespace DataFlow {

  
//*********************************************************************
// class ICFGCSDFSolver
//*********************************************************************
class ICFGCSDFSolver  : private DataFlow::DGraphSolverDFP {
public:
  typedef enum { Forward, Backward } DFDirectionType;

private:
  OA_ptr<DataFlowSet> mTop;
  //OA_ptr<DataFlowSet> mBottom;
  DFDirectionType mDirection;

public:
  //--------------------------------------------------------
  // constructor/destructor
  //--------------------------------------------------------
  ICFGCSDFSolver(DFDirectionType pDirection, 
                 ICFGCSDFProblem& prob);
  virtual ~ICFGCSDFSolver() {}

  //! solves data-flow problem, after done in and out nodes will
  //! have correct data-flow values
  void solve(OA_ptr<ICFG::ICFGInterface> icfg, 
             OA_ptr<Alias::CCSetPerProc> ccResults,
             DFPImplement algorithm);

  //---------------------------------
  // access to DGraphIterativeDFP info
  //---------------------------------
  int getNumIter() { return afterSolve_getNumIter(); }

private:
  //========================================================
  // implementation of DGraphIterativeDFP callbacks
  // These are implemented in ICFGCSDFSolver.cpp for a ICFG.
  //========================================================
  
  //--------------------------------------------------------
  // initialization upcall 
  //--------------------------------------------------------
  void initialize(OA_ptr<DGraph::DGraphInterface> dg);
  
  //--------------------------------------------------------
  // solver upcalls
  //--------------------------------------------------------
  bool atDGraphNode(OA_ptr<DGraph::NodeInterface> node, 
                    DGraph::DGraphEdgeDirection pOrient);
  
  //--------------------------------------------------------
  // finalization upcalls
  //--------------------------------------------------------
  void finalizeNode(OA_ptr<DGraph::NodeInterface> node);


  /*! 
     transfer function for a ICFG::NodeInterface
     Will clone in set to block before passing it to this function
     so ok to return a modified in set
  */
  OA_ptr<DataFlowSet> transfer(OA_ptr<DataFlowSet> in, 
                               OA_ptr<DGraph::NodeInterface> n);

  //--------------------------------------------------------
  // debugging
  //--------------------------------------------------------
public:
  void dump(std::ostream& os, OA_ptr<IRHandlesIRInterface> ir);

private:
  OA_ptr<CC2DFSMap> doMeet(OA_ptr<CC2DFSMap> map1, OA_ptr<CC2DFSMap> map2,
                                         OA_ptr<ICFG::NodeInterface> iNode);

protected:
  //std::map<OA::StmtHandle,DataFlowSet*> mStmtInSetMap;
  //std::map<OA::StmtHandle,DataFlowSet*> mStmtOutSetMap;
  std::map<OA_ptr<ICFG::NodeInterface>, OA_ptr<CC2DFSMap> > mNodeInSetMap;
  std::map<OA_ptr<ICFG::NodeInterface>, OA_ptr<CC2DFSMap> > mNodeOutSetMap;

  // mapping of CallHandles to in-coming sets of CallContexts
  std::map<CallHandle, OA_ptr<std::set<OA_ptr<Alias::CallContext> > > > mCall2CallContextInSet;

  // mapping of CallHandles to out-going context-sensitive data flow sets
  std::map<CallHandle, OA_ptr<CC2DFSMap> > mCallOutSetMap;


  // whether or not the node has had the transfer function applied
  // to all statements at least once
  std::map<OA_ptr<ICFG::NodeInterface>,bool> mNodeInitTransApp;

  ICFGCSDFProblem& mCSDFProb;
  OA_ptr<Alias::CCSetPerProc> mCCSetPerProc;

};

  } // end of DataFlow namespace
}  // end of OA namespace

#endif
