/*! \file
  
  \brief Implementation of DataFlow::CCDFSPairSet

  \author Michelle Strout, Barbara Kreaseck
  \version $Id: CCDFSPairSet.cpp,  Exp $

  Copyright (c) 2002-2005, Rice University <br>
  Copyright (c) 2004-2005, University of Chicago <br>
  Copyright (c) 2006, Contributors <br>
  All rights reserved. <br>
  See ../../../Copyright.txt for details. <br>
*/


#include "CCDFSPairSet.hpp"
#include <OpenAnalysis/Utils/Util.hpp>
#include <OpenAnalysis/Utils/OutputBuilder.hpp>

namespace OA {
  namespace DataFlow {

static bool debug = false;


//! ===============================================================
//! CCDFSPairSet methods
//! ===============================================================

CCDFSPairSet::CCDFSPairSet()
{
  OA_DEBUG_CTRL_MACRO("DEBUG_CCDFSPairSet:ALL", debug);
  mSet = new std::set<CCDFSPair>;
}


//! does not clone, makes a copy
CCDFSPairSet::CCDFSPairSet(const CCDFSPairSet& other)
  :  mSet(other.mSet)
{
}



CCDFSPairSet::~CCDFSPairSet() { }



OA_ptr<CCDFSPairSet> CCDFSPairSet::clone() const
{
   OA_ptr<CCDFSPairSet> retval;
   retval = new CCDFSPairSet();

   OA_ptr<CCDFSPairSetIterator> setIter;
   setIter = new CCDFSPairSetIterator(*mSet);
   while ( ; setIter.isValid(); (*setIter)++) {
     CCDFSPair ccdfsp = setIter->current(); // CCDFSPair operator=() copies
     retval->insert(*(ccdfsp->clone()));
   }
   return retval;
}


    bool CCDFSPairSet::containsPairCC(OA_ptr<Alias::CallContext> cc) const
{
  OA_ptr<CCDFSPairSetIterator> setIter;
  setIter = new CCDFSPairSetterator(mSet);
  for (; setIter->isValid(); (*setIter)++) {
    if ( (*((setIter->current()).getContextPtr())) == (*cc) ) {
      return true;
    }
  }
  return false;
}




void CCDFSPairSet::insert(CCDFSPair ccdfsp) { 
  // don't insert CCDFSPairs with a NULL constValBasicInterface
  if ((rc.getConstPtr()).ptrEqual(0)) {
    return;
  }
  
  // this is not a UNION, it is an insert.  We assume that the caller
  // knows what they are doing.
  
  
  if ( containsPairCC(ccdfsp.getContextPtr()) ) {
    assert(false && "ERROR?? CCDFSPairSet::insert(): CC already in set");
  }
  
  // insert ccdfsp into pairset
  mSet->insert(ccdfsp);
  
}


void CCDFSPairSet::remove(CCDFSPair ccdfsp) { 
  
  // erase ccdfsp from set
  mSet->remove(ccdfsp);
  
}






//! operator == for a CCDFSPairSet CAN NOW rely upon the == operator for the
// underlying set, because the == operator of an element of a CCDFSPairSet, 
// namely a CCDFSPair, considers both the CallContext and DataFlowSet
bool CCDFSPairSet::operator==(const CCDFSPairSet &other) const
{
  
  return (mSet == other.mSet);

}



bool CCDFSPairSet::operator !=(const CCDFSPairSet &other) const
{
  return (!(*this==other)); 
}



void CCDFSPairSet::clear() { 
  mSet->clear();
}


int  CCDFSPairSet::size() const { 
  return mSet->size();
}



bool CCDFSPairSet::isEmpty() const { 
  return (mSet->isEmpty()); 
}

CCDFSPairSet& CCDFSPairSet::operator= (const CCDFSPairSet& other)
{
  mSet = (other.mSet)->clone();
  return *this;
}

CCDFSPairSet& CCDFSPairSet::intersectEqu(CCDFSPairSet& other)
{

  // X intersectEqu Y (where X is *this)
  //=============================================
  // X and Y are either Partial or Explicit sets
  OA_ptr<OA::DataFlow::DataFlowSetImpl<CCDFSPair> > pairsetXonlyY;
  OA_ptr<OA::DataFlow::DataFlowSetImpl<CCDFSPair> > pairsetYonlyX;
  OA_ptr<OA::DataFlow::DataFlowSetImplIterator<CCDFSPair> > pIter;

  
  if (this->isPartialSet()) {
    if (other.isPartialSet()) {
      //! Partial X    intersectEqu   Partial Y
      //   ==> tagset:= X.tagset intersect Y.tagset
      //   ==> pairset:= (X.pairset intersect Y.pairset)
      //           union {<t,c>|<t,c> in X.pairset and t in Y.tagset}
      //           union {<t,c>|<t,c> in Y.pairset and t in X.tagset}
      
      // make {<t,c>|<t,c> in X.pairset and t in Y.tagset}
      pairsetXonlyY = mSet->clone().convert<OA::DataFlow::DataFlowSetImpl<CCDFSPair> >();
      pIter = new OA::DataFlow::DataFlowSetImplIterator<CCDFSPair>(pairsetXonlyY);
      for (; pIter->isValid(); (*pIter)++) {
        Alias::AliasTag tag = (pIter->current()).getTag();
        if (!((other.mTagSet)->contains(tag))) {
          pairsetXonlyY->remove(pIter->current());
        }
      }
      
      // make {<t,c>|<t,c> in Y.pairset and t in X.tagset}
      pairsetYonlyX = other.mSet->clone().convert<OA::DataFlow::DataFlowSetImpl<CCDFSPair> >();
      pIter = new OA::DataFlow::DataFlowSetImplIterator<CCDFSPair>(pairsetYonlyX);
      for (; pIter->isValid(); (*pIter)++) {
        Alias::AliasTag tag = (pIter->current()).getTag();
        if (!(mTagSet->contains(tag))) {
          pairsetYonlyX->remove(pIter->current());
        }
      }

      // make resulting pairset in *this
      mSet->intersectEqu(*(other.mSet));
      mSet->unionEqu(*pairsetXonlyY);
      mSet->unionEqu(*pairsetYonlyX);

      // make resulting tagset in *this (mTagSet could become empty)
      mTagSet->intersectEqu(*(other.mTagSet));
      
    } else {
      //! Partial X    intersectEqu   Explicit Y
      //   ==> tagset becomes empty
      //   ==> pairset:= (X.pairset intersect Y.pairset)
      //           union {<t,c>|<t,c> in Y.pairset and t in X.tagset}

      // make {<t,c>|<t,c> in Y.pairset and t in X.tagset}
      pairsetYonlyX = other.mSet->clone().convert<OA::DataFlow::DataFlowSetImpl<CCDFSPair> >();
      pIter = new OA::DataFlow::DataFlowSetImplIterator<CCDFSPair>(pairsetYonlyX);
      for (; pIter->isValid(); (*pIter)++) {
        Alias::AliasTag tag = (pIter->current()).getTag();
        if (!(mTagSet->contains(tag))) {
          pairsetYonlyX->remove(pIter->current());
        }
      }

      // make resulting pairset in *this
      mSet->intersectEqu(*(other.mSet));
      mSet->unionEqu(*pairsetYonlyX);

      // make resulting tagset in *this
      mTagSet->clear();

    }
  } else {
    if (other.isPartialSet()) {
      //! Explicit X   intersectEqu   Partial Y
      //   ==> tagset becomes empty
      //   ==> pairset:= (X.pairset intersect Y.pairset)
      //           union {<t,c>|<t,c> in X.pairset and t in Y.tagset}

      // make {<t,c>|<t,c> in X.pairset and t in Y.tagset}
      pairsetXonlyY = mSet->clone().convert<OA::DataFlow::DataFlowSetImpl<CCDFSPair> >();
      pIter = new OA::DataFlow::DataFlowSetImplIterator<CCDFSPair>(pairsetXonlyY);
      for (; pIter->isValid(); ++(*pIter)) {
        Alias::AliasTag tag = (pIter->current()).getTag();
        if (!((other.mTagSet)->contains(tag))) {
          pairsetXonlyY->remove(pIter->current());
        }
      }
      
      // make resulting pairset in *this
      mSet->intersectEqu(*(other.mSet));
      mSet->unionEqu(*pairsetXonlyY);

      // make resulting tagset in *this
      mTagSet->clear();

    } else {
      //! Explicit X   intersectEqu   Explicit Y
      //    ==> intersection of the underlying pairsets

      mSet->intersectEqu(*(other.mSet));

    }
  }
  
  return *this;

}


CCDFSPairSet& CCDFSPairSet::unionEqu(CCDFSPairSet& other)
{
  if (isUniversalSet()) {
    return *this;
  } else if (other.isUniversalSet()) {
    *this = other;
    return *this;
  }

  // X unionEqu Y (where X is *this)
  //============================================
  // X and Y are either Partial or Explicit sets

  // union the underlying sets (for Explicit, tagsets will be empty, OK)
  mTagSet->unionEqu(*(other.mTagSet));
  mSet->unionEqu(*(other.mSet));

  // remove any pairset in mSet whose tag is in mTagSet
  if (mTagSet->size()!=0 && mSet->size()!=0) {
    OA_ptr<OA::DataFlow::DataFlowSetImplIterator<CCDFSPair> > pIter;
    pIter = new OA::DataFlow::DataFlowSetImplIterator<CCDFSPair>(mSet);
    for (; pIter->isValid(); (*pIter)++) {
      Alias::AliasTag tag = (pIter->current()).getTag();
      if ((mTagSet->contains(tag))) {
        mSet->remove(pIter->current());
      }
    }
  }
  
  return *this;

}


CCDFSPairSet& CCDFSPairSet::minusEqu(CCDFSPairSet& other)
{

  // X minusEqu Universal
  // ==============================
  if (other.isUniversalSet()) {
    clear(); // should make *this empty
    return *this;
  }

  // Universal minusEqu X
  // ==============================
  // --> we shouldn't need this right now.
  // --> also, we have no way of indicating const values that are NOT
  //     in the set
  assert(  !(isUniversalSet())  );


  // X minusEqu Y   (where Y is *this)
  //=========================
  // X and Y are either Partial or Explicit

  // if X is Partial and Y has a non-empty pairset, assert for now
  assert ( !((mTagSet->size() > 0) && ((other.mSet)->size() > 0)) );
  
  // set difference of underlying pairsets
  if ((other.mSet)->size() > 0) {
    mSet->minusEqu(*(other.mSet));
  }

  if ((other.mTagSet)->size() > 0) {

    // set difference of underlying tagsets
    if (mTagSet->size() > 0) {
      mTagSet->minusEqu(*(other.mTagSet));
    }

    // remove <t,*> from mSet when t in other.mTagSet
    if (mSet->size() > 0) {
      OA_ptr<OA::DataFlow::DataFlowSetImplIterator<CCDFSPair> > pIter;
      pIter = new OA::DataFlow::DataFlowSetImplIterator<CCDFSPair>(mSet);
      for (; pIter->isValid(); (*pIter)++) {
        Alias::AliasTag tag = (pIter->current()).getTag();
        if ( (other.mTagSet)->contains(tag) ) {
          mSet->remove(pIter->current());
        }
      }
    }

  }

  return *this;

}


//! ================ Output =========================
void CCDFSPairSet::output(IRHandlesIRInterface& ir) const {
     std::cout << "Please call output with alias::Interface"
               << std::endl;
     assert(0);
}



//! Return a string representing the contents of a CCDFSPairSet

std::string CCDFSPairSet::toString(OA_ptr<IRHandlesIRInterface> pIR){

  std::ostringstream oss;
  oss << "{ ";

  if (mUniversal) {
    oss << "isUniversal ";
  } else {
    if (mTagSet->size() > 0) {
      oss << "isPartialUniversal Tags: [";
      oss << mTagSet->toString(*pIR);
      oss << "] ";
    }
    if (mSet->size() > 0) {
      oss << "Pairs: ";
      OA_ptr<OA::DataFlow::DataFlowSetImplIterator<CCDFSPair> > defIter;
      defIter = new OA::DataFlow::DataFlowSetImplIterator<CCDFSPair>(mSet);
      if (defIter->isValid()) { // handle first CCDFSPair
        oss << (defIter->current()).toString(*pIR);
        (*defIter)++;
      }
      for (; defIter->isValid(); (*defIter)++) { // handle rest
        oss << ", " << (defIter->current()).toString(*pIR);
      }
    }
  }

  oss << " }";
  return oss.str();
}


std::string CCDFSPairSet::toString() {
  std::ostringstream oss;
  oss << "{ ";

  if (mUniversal) {
    oss << "isUniversal ";
  } else {
    if (mTagSet->size() > 0) {
      oss << "isPartialUniversal Tags: [";
      OA_ptr<OA::DataFlow::DataFlowSetImplIterator<Alias::AliasTag> > tagIter;
      tagIter = new OA::DataFlow::DataFlowSetImplIterator<Alias::AliasTag>(mTagSet);
      for (; tagIter->isValid(); (*tagIter)++) {
        //Alias::AliasTag tag = tagIter->current();
        oss << tagIter->current() << ", ";
      }
      oss << "] ";
    }
    if (mSet->size() > 0) {
      oss << "Pairs: ";
      OA_ptr<OA::DataFlow::DataFlowSetImplIterator<CCDFSPair> > defIter;
      defIter = new OA::DataFlow::DataFlowSetImplIterator<CCDFSPair>(mSet);
      if (defIter->isValid()) { // handle first CCDFSPair
        oss << (defIter->current());
        (*defIter)++;
      }
      for (; defIter->isValid(); (*defIter)++) { // handle rest
        oss << ", " << (defIter->current());
      }
    }
  }
  oss << " }";

  return oss.str();
}
  




void CCDFSPairSet::output(OA::IRHandlesIRInterface &ir,
                       Alias::Interface& alias) const
{
  sOutBuild->objStart("CCDFSPairSet");
  
  if (mUniversal) {
    sOutBuild->field("mUniversal","TRUE");
  } else {
    if (mTagSet->size() > 0) {
      //sOutBuild->field("mPartial","TRUE");
      //std::ostringstream oss;
      //mTagSet->dump(oss,ir);
      //sOutBuild->field("mTagSet",oss.str());
      sOutBuild->field("mTagSet",mTagSet->toString(ir));
    }
    sOutBuild->listStart();
    OA_ptr<OA::DataFlow::DataFlowSetImplIterator<CCDFSPair> > defIter;
    defIter = new OA::DataFlow::DataFlowSetImplIterator<CCDFSPair>(mSet);
    for (; defIter->isValid(); (*defIter)++) {
        sOutBuild->listItemStart();
        sOutBuild->fieldStart("CCDFSPair");
        (defIter->current()).output(ir,alias);
        sOutBuild->fieldEnd("CCDFSPair");
        sOutBuild->listItemEnd();
    }
    sOutBuild->listEnd();
  }

  sOutBuild->objEnd("CCDFSPairSet");
}



void CCDFSPairSet::dump(std::ostream &os, OA_ptr<IRHandlesIRInterface> pIR)
{ os << toString(pIR) << std::endl; }




//! ===============================
//! CCDFSPairSet Iterator 
//! ===============================

CCDFSPairSetIterator::CCDFSPairSetIterator (CCDFSPairSet& RCSet) 
  : mSet(RCSet.mSet)
{
  assert(!mSet.ptrEqual(NULL));
  mRCIter = new OA::DataFlow::DataFlowSetImplIterator<CCDFSPair>(mSet);

  mUniversal = RCSet.mUniversal;
  mPartial = RCSet.isPartialSet();
}




CCDFSPairSetIterator::~CCDFSPairSetIterator () {}




void CCDFSPairSetIterator::operator ++ () { 
  if (isValid()) (*mRCIter)++; 
}




//! is the iterator at the end
bool CCDFSPairSetIterator::isValid() 
{ 
  return (mRCIter->isValid()); 
}




//! return copy of current node in iterator
CCDFSPair CCDFSPairSetIterator::current() 
{ 
  assert(isValid());
  return (mRCIter->current()); 
}




//! reset iterator to beginning of set
void CCDFSPairSetIterator::reset() 
{
  mRCIter->reset();
}


//! empty sets and universal sets look the same to the iterator, so check
bool CCDFSPairSetIterator::isUniversalSetIter()
{
  return (mUniversal);
}

//! is CCDFSPairSet base a partial universal set?
bool CCDFSPairSetIterator::isPartialSetIter()
{
  return (mPartial);
}



  } // end of DataFLow namespace
} // end of OA namespace

