/*! \file
  
  \brief The AnnotationManager that generates AliasTagResults
         with CSFIAlias.

  \authors Michelle Strout, Barbara Kreaseck

  Copyright (c) 2002-2005, Rice University <br>
  Copyright (c) 2004-2005, University of Chicago <br>
  Copyright (c) 2006, Contributors <br>
  All rights reserved. <br>
  See ../../../Copyright.txt for details. <br>
*/

#include "ManagerCSFIAliasAliasTag.hpp"
#include <Utils/Util.hpp>


namespace OA {
  namespace Alias {

static bool debug = false;

/*!
*/
ManagerCSFIAliasAliasTag::ManagerCSFIAliasAliasTag(OA_ptr<AliasIRInterface> _ir) 
   : ManagerCSFIAlias(_ir)
{
    OA_DEBUG_CTRL_MACRO("DEBUG_ManagerCSFIAliasAliasTag:ALL", debug);
}

OA_ptr<Alias::Interface> 
ManagerCSFIAliasAliasTag::performAnalysis( OA_ptr<CallGraph::CallGraphInterface> callGraph,
                                           OA_ptr<Alias::CCSetPerProc> ccResults,
                                         FIAliasImplement implement )
{
  mCCResults = ccResults;
  
  // create an empty set of results.
  OA_ptr<CSFIAliasTagResults> retval;
  retval = new CSFIAliasTagResults();
  
  // Invoke the ManagerCSFIAlias routine to create the alias sets
  // within the union-find universe.
  OA_ptr<UnionFindUniverse> ufset = performCSFIAlias(callGraph, implement);
  
  if (debug) {
    std::cout << "\n*** after performCSFIAlias, "
              << " before mapping to AliasTagSets: \n";
  }
  
  // Depending on the implementation of FIAlias, only a 
  // subset of the defined procedures may have been analyzed.
  OA_ptr<IRProcIterator> analyzedProcIter = getAnalyzedProcIter();
  
  //--------------------------------------------------------------
  // Convert the union-find universe to mappings to alias tag sets
  
  // map each ufsetID to a tag id by just using a counter
  std::map<int,AliasTag> ufsetIDToTag;
  // need a map for strictly local named refs as well
  std::map<MREwContext,AliasTag> strictlyLocalToTag;
  
  // for each ufsetID keep track of alias tags for sets
  // and strictly locals that it subsumes
  std::map<int,std::set<AliasTag> > ufsetToSubsumes;
  
  // keeps track of which ufsets have subset ufsets
  // that it subsumes
  std::map<int,bool> ufsetSubsumesSubsets;
  
  // keeps track of the number of fixed refs are in the ufset
  std::map<int,int> ufsetToNumFixed;
  
  int count = 1; // used for unique AliasTag numbering
  
  // visit all of the MREs in the program being analyzed
  // to get the set of valid ufsetIDs.
  if(debug) {
    std::cout << "Assign alias tag numbers to each UF set" << std::endl;
  }
  
  std::set<int> ufsetIDs;
  std::map<MREwContext,int>::iterator mreIter;
  //75
  // loop 1
  for (mreIter=mMREwCToID.begin(); mreIter!=mMREwCToID.end(); mreIter++ ) {
    MREwContext mrewc = mreIter->first;
    OA_ptr<MemRefExpr> mre = mrewc.getMRE();
    OA_ptr<CallContext> cctext = mrewc.getCallContext();
    
    // only map those MREs that do not involve an addressOf operation
    if(mre->isaRefOp()) {
      OA_ptr<RefOp> refop = mre.convert<RefOp>();
      if(refop->isaAddressOf()) { continue; }
    }
    
    // get ufset for MRE
    int ufsetID = ufset->Find(mreIter->second);
    ufsetIDs.insert(ufsetID);
    
    if(debug) {
      std::cout << "\t" << mre->comment(*mIR) 
                << "\n\t\t";
      cctext->output(*mIR);
      std::cout <<" in UFset "<< ufsetID << std::endl;
    }
    // initially assume that ufset does not subsume any ufset
    // due to subsetRef
    ufsetSubsumesSubsets[ufsetID] = false;
    
    // only give this set an alias tag if it is the first
    // time we have seen it
    if (ufsetIDToTag.find(ufsetID)==ufsetIDToTag.end()) {
      if(debug) {
        std::cout << "\t\tUFSet " << ufsetID << " mapped to tag " << count
                  << std::endl;
      }
      
      ufsetIDToTag[ufsetID] = AliasTag(count++);
    }
    
    // new code here as of 7/9/09 as per MMS #171
    // initialize to zero as needed for auto-increment below 
    if (ufsetToNumFixed.find(ufsetID) == ufsetToNumFixed.end()) {
      ufsetToNumFixed[ufsetID] = 0;
    }
    if (mre->isaNamed()) {
      OA_ptr<NamedRef> namedRef = mre.convert<NamedRef>();
      if (namedRef->isStrictlyLocal()) {
        ufsetToNumFixed[ufsetID]++;
      }
    } else if (mre->isaUnnamed()) {
      ufsetToNumFixed[ufsetID]++;
    }
    
    /* //**************************** old code here before 7/9/09 
    // give strictly locals their own alias tag and have
    // the ufset they are in subsume that alias tag
    //113
    if (mre->isaNamed()) {
      OA_ptr<NamedRef> namedRef = mre.convert<NamedRef>();
      //115
      if (namedRef->isStrictlyLocal()) {
        if (strictlyLocalToTag.find(mrewc)==strictlyLocalToTag.end()) {
          strictlyLocalToTag[mrewc] = AliasTag(count++);
        }
        //119
        ufsetToSubsumes[ufsetID].insert(strictlyLocalToTag[mrewc]);
      }
    }
    */ // ***************************

  } // end of loop 1

  // Map each ufset to a set of ufset alias tags that it subsumes 
  // including itself.
  // A ufset subsumes all ufsets that include subsetrefs to the MREs
  // it contains.  Also a ufset subsumes all alias tags given to 
  // strictly local NamedRefs that it contains.
  if(debug) {
    std::cout << "Mapping each UFSet to set of Alias Tags it subsumes due to SubSetRef:" << std::endl;
  }
  // loop through the ufsetIDs 
  std::set<int>::iterator setIter;
  //135
  for (setIter=ufsetIDs.begin(); setIter!=ufsetIDs.end(); setIter++ ) {
    int ufsetID = *setIter;
    
    if(debug) {
      std::cout << "\tConsidering ufset " << ufsetID << std::endl;
    }
    
    // we need to subsume self so that all the MemRefHandles
    // and MREs get mapped appropriately
    ufsetToSubsumes[ufsetID].insert(ufsetIDToTag[ufsetID]);
    if(debug) {
      std::cout << "\tMap self as a subsumed set" << std::endl;
    }
    
    // loop through the justRefop keys
    // for this ufset
    std::map<OA_ptr<MemRefExpr>,int> outerRefwCToUFSetID = mMap[ufsetID];
    std::map<OA_ptr<MemRefExpr>,int>::iterator outerRefwCIter;
    
    for (outerRefwCIter=outerRefwCToUFSetID.begin();
         outerRefwCIter!=outerRefwCToUFSetID.end();
         outerRefwCIter++) 
      {
        //MREwContext outerRefwC = outerRefwCIter->first;
        //OA_ptr<MemRefExpr> outerRef = outerRefwC.getMRE();
        OA_ptr<MemRefExpr> outerRef = outerRefwCIter->first;
        //OA_ptr<CallContext> outerCctext = outerRefwC.getCallContext();
        
        // if the refOp is a SubSetRef of some kind
        if (outerRef->isaSubSetRef()) {
          if(debug) {
            std::cout << "\t\tSubsumes ufset containing "
                      << outerRef->comment(*mIR);
              //           << " with ";
              //outerCctext->output(*mIR);
            std::cout << "\n\t\t\tWhich is "
                      << ufset->Find( outerRefwCIter->second ) << std::endl;
          }
          if (debug) {
            std::cout << "\t\t\touterRefwCIter->second = " << outerRefwCIter->second
                      << "\n\t\t\tufset->Find(outerRefwCIter->second) = "
                      << ufset->Find(outerRefwCIter->second)
                      << "\n\t\t\tufsetIDToTag[" <<  outerRefwCIter->second
                      << "] = " 
                      << ufsetIDToTag[ufset->Find(outerRefwCIter->second )]
                      << "\n\t\t\tinserted into ufsetToSubsumes["
                      << ufsetID << "]\n\n";
          }
          ufsetToSubsumes[ufsetID].insert( ufsetIDToTag[ufset->Find(
                                                outerRefwCIter->second )]);
          ufsetSubsumesSubsets[ufsetID] = true;
        }
      }
  }
  
  // new Loop
  // look for strictly locals that need own tag
  // for each mre
  //181
  for (mreIter=mMREwCToID.begin(); mreIter!=mMREwCToID.end(); mreIter++ ) {
    MREwContext mrewc = mreIter->first;
    OA_ptr<MemRefExpr> mre = mrewc.getMRE();
    OA_ptr<CallContext> cctext = mrewc.getCallContext();
    
    // only map those MREs that do not involve an addressOf operation
    if(mre->isaRefOp()) {
      OA_ptr<RefOp> refop = mre.convert<RefOp>();
      if(refop->isaAddressOf()) { continue; }
    }
    
    if(debug) {
      std::cout << "Checking for StrictlyLocal stuff for: " 
                << mre->comment(*mIR) 
                << " with ";
      cctext->output(*mIR);
      std::cout << std::endl;
    }
    
    // find the ufset for the mre
    int ufsetID = ufset->Find(mreIter->second);
    
    if(debug) {
      std::cout << "\tWhich is in UFSet " << ufsetID << std::endl;
    }
    
    // Check for strictly locals that need own tag
    bool isMust=false;
    OA_ptr<AliasTagSet> tag_set;
    
    // new code as of 7/9/09 as per MMS #171
    if (mre->isaNamed()) {
      OA_ptr<NamedRef> namedRef = mre.convert<NamedRef>();
      if (namedRef->isStrictlyLocal()
          && !ufsetSubsumesSubsets[ufsetID]
          && ufsetToNumFixed[ufsetID] > 1) {
        
        if (debug) {
          std::cout << "\tIS Stricly Local in ufset with multiple Fixed\n";
        }
        
        // give mrewc it's own tag
        if (strictlyLocalToTag.find(mrewc)==strictlyLocalToTag.end()) {
          strictlyLocalToTag[mrewc] = AliasTag(count++);
        }
        ufsetToSubsumes[ufsetID].insert(strictlyLocalToTag[mrewc]);
        
        // old lines 220 - 222
        /*
          isMust = true;
          tag_set = new AliasTagSet(isMust);
          tag_set->insert(strictlyLocalToTag[mrewc]); 
        */
      }
    }
    
  }   
  
  // old Loop 3
  // map each mre to set of alias tags using the above two maps
  // for each mre
  //181
  for (mreIter=mMREwCToID.begin(); mreIter!=mMREwCToID.end(); mreIter++ ) {
    MREwContext mrewc = mreIter->first;
    OA_ptr<MemRefExpr> mre = mrewc.getMRE();
    OA_ptr<CallContext> cctext = mrewc.getCallContext();
    
    // only map those MREs that do not involve an addressOf operation
    if(mre->isaRefOp()) {
      OA_ptr<RefOp> refop = mre.convert<RefOp>();
      if(refop->isaAddressOf()) { continue; }
    }
    
    if(debug) {
      std::cout << "Map alias tags for: " << mre->comment(*mIR) 
                << " with ";
      cctext->output(*mIR);
      std::cout << std::endl;
    }
    
    // find the ufset for the mre
    int ufsetID = ufset->Find(mreIter->second);
    
    if(debug) {
      std::cout << "\tWhich is in UFSet " << ufsetID << std::endl;
    }
    
    // Create AliasTagSet for MRE.
    // All alias tags set should be may EXCEPT for strictly
    // local named refs when the ufset that the strictly local
    // is in only subsumes itself. 
    // The strictly local must also be in a ufset that does NOT
    // subsume any other ufsets due to SubsetRefs.
    //   (now, strictlyLocalToTag is set only for those with 
    //    !ufsetSubsumesSubset already (loop 3 above) )
    // Has to be this way because there is no guarantee that a
    // single strictly local in a ufset subsumes all of the sets
    // involving subset refs for the ufset.
    bool isMust=false;
    OA_ptr<AliasTagSet> tag_set;
    
    // new code here as per MMS #171
    
    if (strictlyLocalToTag.find(mrewc)!=strictlyLocalToTag.end()
        && !ufsetSubsumesSubsets[ufsetID]) {
      // MAYBE already know no subsumesSubsets due to new loop above
      isMust = true;
      tag_set = new AliasTagSet(isMust);
      tag_set->insert(strictlyLocalToTag[mrewc]);
      if (debug) {
        std::cout << "\tStrictly Local with no SubsumesSubsets\n"
                  << "\t\tso, just Strictly Local "
                  << strictlyLocalToTag[mrewc] << "\n\n";
      }
      
    } else { 
      // create set of AliasTag, check for must/may, create AliasTagSet
      isMust = false;
      OA_ptr<std::set<AliasTag> > tags;
      tags = new std::set<AliasTag>;
      tags->insert(ufsetIDToTag[ ufsetID ]);
      if (debug) {
        std::cout << "\t\tinserting tag for UFSet " << ufsetIDToTag[ufsetID]
                  << "\n";
        if (strictlyLocalToTag.find(mrewc) == strictlyLocalToTag.end()) {
          std::cout << "\tNOT using extra Strictly Local mre\n";
        }
        if (ufsetSubsumesSubsets[ufsetID]) {
          std::cout << "\tUFSet " << ufsetID << " SUBSUMES Subsets\n";
        }
      }

      
      // for each subsumed alias tag for MREs ufset 
      std::set<AliasTag>::iterator setIter;
      for (setIter =ufsetToSubsumes[ufsetID].begin();
           setIter!=ufsetToSubsumes[ufsetID].end(); setIter++)
        {
          tags->insert(*setIter);
          if (debug) {
            std::cout << "\t\tinserting tag for subsume " 
                      << *setIter << "\n";
          }
        }
      
      // check for only one fixed location and not SubSetRef
      if (ufsetToNumFixed[ufsetID] <= 1 && !(mre->isaSubSetRef()) ) {
        isMust = true;
      }
      if (debug) {
        std::cout << "\t\tsetting tag_set as isMust==";
        if (isMust) {
          std::cout << "true\n\n";
        } else {
          std::cout << "false\n\n";
        }
      }
      
      tag_set = new AliasTagSet(tags,isMust);
    }
    
    
    
    // ****************************** old code here before 7/9/09 
    /*
    //217
      if (strictlyLocalToTag.find(mrewc)!=strictlyLocalToTag.end()
          && !ufsetSubsumesSubsets[ufsetID])
        {
    //220
          isMust = true;
          tag_set = new AliasTagSet(isMust);
    //222
          tag_set->insert(strictlyLocalToTag[mrewc]); 
          
        // May alias
        } else {
        isMust = false;
        tag_set = new AliasTagSet(isMust);
        tag_set->insert(ufsetIDToTag[ ufsetID ]);
        
        // for each subsumed alias tag for MREs ufset 
        std::set<AliasTag>::iterator setIter;
        for (setIter =ufsetToSubsumes[ufsetID].begin();
             setIter!=ufsetToSubsumes[ufsetID].end(); setIter++)
          {
            tag_set->insert(*setIter);
          }
      }
      */ // *******************************************************
      
    // map MRE and all MemRefHandles for that MRE to 
    // created alias tag set
    retval->mapMemRefToAliasTagSet(mre, cctext, tag_set);
    
    // for each MemRefHandle for mre
    std::set<MemRefHandle>::iterator memrefIter;
    for(memrefIter=mMREwCToMemRefHandles[mrewc].begin();
        memrefIter!=mMREwCToMemRefHandles[mrewc].end();
        memrefIter++)
      {
        retval->mapMemRefToAliasTagSet(*memrefIter, cctext, tag_set); 
      }
  }
  
  
  return retval;
}


  } // end of namespace Alias
} // end of namespace OA
