/*! \file
  
  \brief Implementation of CallContext

  \authors Barbara Kreaseck
  \version $Id$

  Copyright (c) 2002-2005, Rice University <br>
  Copyright (c) 2004-2005, University of Chicago <br>
  Copyright (c) 2006, Contributors <br>
  All rights reserved. <br>
  See ../../../Copyright.txt for details. <br>
*/
#include "CallContext.hpp"

#include <OpenAnalysis/Alias/Interface.hpp>

namespace OA {
  namespace Alias {

// initialization of static class member
unsigned int CallContext::mMaxDegree = 1;

void CallContext::setMaxDegree(unsigned int m)
{
  static int numSetMaxDegreeCalls = 0;
  mMaxDegree = m;
  numSetMaxDegreeCalls++;
  assert(numSetMaxDegreeCalls < 2);
}

unsigned int CallContext::getMaxDegree()
{
  return mMaxDegree;
}

CallContext::CallContext()
{
  // mList is empty
}

CallContext::CallContext(CallHandle call)
{
  mList.push_back(call);
}

CallContext::CallContext(std::list<CallHandle> l)
{
  unsigned int cnt;
  std::list<CallHandle>::iterator iter;
  for (iter = l.begin(), cnt = 0; 
       iter!=l.end() && cnt < mMaxDegree; 
       iter++,cnt++) {
    mList.push_back(*iter);
  }
}

CallContext::CallContext(OA_ptr<CallContext> cc)
{
  unsigned int cnt;
  if (!(cc.ptrEqual(0))) {
    std::list<CallHandle>::iterator iter;
    for (iter = cc->mList.begin(), cnt = 0; 
	 iter!=cc->mList.end() && cnt < mMaxDegree; 
	 iter++,cnt++) {
      mList.push_back(*iter);
    }
  }
}



CallContext::~CallContext()
{

}


int CallContext::size() 
{ 
  return mList.size();
}


bool CallContext::operator ==(const CallContext& other) const
{
  /*
  bool result = true;
  std::list<CallHandle>::iterator iter1;
  std::list<CallHandle>::iterator iter2;
  int cnt;

  if (mList.size() != other.mList.size()) {
    return false;
  }

  iter1 = mList.begin();
  iter2 = other.mList.begin();
  for (cnt = 0;
       iter1 != mList.end(),  iter2 != other.mList.end(), cnt < mMaxDegree;
       iter1++, iter2++, cnt++)
    {
      if (*iter1 != *iter2) {
	return false;
      }
    }

  if ((mList.size() > mMaxDegree)) {
    // all we know is that they agree for the first mMaxDegree CallHandles
    // but not if they agree beyond that ...
    //   Although, not sure how the lists got longer than mMaxDegree ...
    assert(0);
  }
  
  return true;
  */

  if (mList == other.mList) {
    return true; 
  } else {
    return false;
  }
}

//! order operator based upon order operator of CallHandles
//! and list order of CallHandles
bool CallContext::operator <(const CallContext& other) const
{
  if (mList < other.mList) {
    return true;
  } else {
    return false;
  }
}

//!Expand context by given CallHandle (i.e., add it to the end of the list)
//! *this calls 'call'
//! before A->B
//! after  A->B->call
void CallContext::append(CallHandle call)
{
  if (mList.size() >= mMaxDegree) {
    // cannot append ... do we drop ?? from which end ??
    assert (0 && "CallContext::append() exceeds mMaxDegree");
  } else {
    assert(call != CallHandle(0));
    mList.push_back(call);
  }
}


//!Expand context by given CallHandle (i.e., add it to the front of the list)
//! 'call' calls *this
//! before A->B
//! after call->A->B
void CallContext::prepend(CallHandle call)
{
  if (mList.size() >= mMaxDegree) {
    // cannot prepend ... do we drop ?? from which end ??
    assert (0 && "CallContext::prepend() exceeds mMaxDegree");
  } else {
    assert(call!=CallHandle(0));
    mList.push_front(call);
  }
}

//!Expand context by given CallHandle (i.e., add it to the end of the list)
//! *this calls 'call'
//! before A->B
//! after  A->B->call
void CallContext::appendForce(CallHandle call)
{
  if (mList.size() >= mMaxDegree) {
    // remove first on the list to make room for newest call
    mList.pop_front();
  }

  assert(call != CallHandle(0));
  mList.push_back(call);
}


//!Expand context by given CallHandle (i.e., add it to the front of the list)
//! 'call' calls *this
//! before A->B
//! after call->A->B
void CallContext::prependForce(CallHandle call)
{
  if (mList.size() >= mMaxDegree) {
    // remove last on the list to make room for newest pre-call
    mList.pop_back();
  }
  assert(call!=CallHandle(0));
  mList.push_front(call);
}


//!Return a clone
OA_ptr<CallContext> CallContext::clone() const
{
  OA_ptr<CallContext> result;
  result = new CallContext(mList);
  return result;
}

//!assignment
CallContext& CallContext::operator=( CallContext& other)
{
  unsigned int cnt;
  mList.clear();
  std::list<CallHandle>::iterator iter;
  for (iter = other.mList.begin(), cnt = 0; 
       iter!= other.mList.end() && cnt < mMaxDegree; 
       iter++,cnt++) {
    mList.push_back(*iter);
  }
  return *this; 
}


void CallContext::output(OA::IRHandlesIRInterface& ir) const
{
    sOutBuild->objStart("CallContext");
    sOutBuild->field("mMaxDegree",int2string(mMaxDegree));
    sOutBuild->listStart();
    std::list<CallHandle> copyList = mList;
    std::list<CallHandle>::iterator iter;
    for (iter = copyList.begin(); iter != copyList.end(); iter++) {
      sOutBuild->listItem(ir.toString(*iter));
    }
    sOutBuild->listEnd();
    sOutBuild->objEnd("CallContext");
}

void CallContext::output(
    OA::IRHandlesIRInterface& ir,
    const Interface& aliasResults) const
{
  output(ir);
}

void CallContext::dump(
    std::ostream& os,
    IRHandlesIRInterface& ir
    //,Interface& aliasResults
    )
{
    bool first = true;

    os << "<mMaxDegree: " << int2string(mMaxDegree) << ", CallContext: \n";

    // iterate through each CallHandle
    std::list<CallHandle>::iterator iter;
    for (iter = mList.begin(); iter != mList.end(); iter++) {

        if(!first) {
            os << ", \n";
        } else {
            first = false;
        }
	os << "\t";
        os << ir.toString(*iter);
    }

    os << "\n>";
}

std::string 
CallContext::toString(OA::IRHandlesIRInterface& ir)
{
  std::ostringstream oss;
  oss << "(CC: ";
  
  bool first = true;
  // iterate through each CallHandle
  std::list<CallHandle>::iterator iter;
  for (iter = mList.begin(); iter != mList.end(); iter++) {
    
    if(!first) {
      oss << ", ";
    } else {
      first = false;
    }
    oss << ir.toString(*iter);
  }
  
  
  oss << " :CC)";
  
  return oss.str();
}

std::string 
CallContext::toString(OA::IRHandlesIRInterface& ir,
                      const Interface& aliasResults)
{
  return toString(ir);
}


}  // end of Alias namespace
}  // end of OA namespace
