/*! \file
  
  \brief Implementation of Activity::ActiveStandard

  \author Michelle Strout, Barbara Kreaseck
  \version $Id: ActiveStandard.cpp,v 1.8 2005/06/10 02:32:01 mstrout Exp $

  Copyright (c) 2002-2005, Rice University <br>
  Copyright (c) 2004-2005, University of Chicago <br>
  Copyright (c) 2006, Contributors <br>
  All rights reserved. <br>
  See ../../../Copyright.txt for details. <br>
*/

#include <OpenAnalysis/Activity/ActiveStandard.hpp>

namespace OA {
  namespace Activity {

ActiveStandard::ActiveStandard(ProcHandle p) 
{ mActiveStmtSet = new std::set<StmtHandle>;
  mActiveMemRefSet = new std::set<MemRefHandle>;
  mActiveTagSet = new DataFlow::DataFlowSetImpl<Alias::AliasTag>;
  mActiveSymSet = new std::set<SymHandle>;
  mUnknownLocActive = false;

  mHasCSResults = false;
}

//! Indicates whether Context-Sensitive results are available
bool ActiveStandard::hasCSResults()
{
  return mHasCSResults;
}

// ----------- active tags -------------

//! Return an iterator for set of active tags
OA_ptr<OA::DataFlow::DataFlowSetImplIterator<Alias::AliasTag> > 
ActiveStandard::getActiveTagsIterator() 
{ 
  OA_ptr<OA::DataFlow::DataFlowSetImplIterator<Alias::AliasTag> > retval;
  retval = new OA::DataFlow::DataFlowSetImplIterator<Alias::AliasTag>(mActiveTagSet);
  return retval;
}

//! Return an iterator for set of active tags for context
OA_ptr<OA::DataFlow::DataFlowSetImplIterator<Alias::AliasTag> > 
ActiveStandard::getActiveTagsIterator(OA_ptr<Alias::CallContext> cc) 
{ 
  OA_ptr<OA::DataFlow::DataFlowSetImplIterator<Alias::AliasTag> > retval;
  OA_ptr<DataFlow::DataFlowSetImpl<Alias::AliasTag> > aTagSet;
  if (mCCToActiveTagSetMap.find(cc) == mCCToActiveTagSetMap.end()) {
    aTagSet = new DataFlow::DataFlowSetImpl<Alias::AliasTag>;
  } else {
    aTagSet = mCCToActiveTagSetMap[cc];
  }
  retval = new OA::DataFlow::DataFlowSetImplIterator<Alias::AliasTag>(aTagSet);
  return retval;
}


// ----------- active stmts -------------

//! Return an iterator for set of active stmts
OA_ptr<StmtHandleIterator> 
ActiveStandard::getActiveStmtIterator() 
{
  OA_ptr<StmtHandleIterator> retval;
  retval = new ActiveStmtIterator(mActiveStmtSet);
  return retval;
}

//! Return an iterator for set of active stmts for context
OA_ptr<StmtHandleIterator> 
ActiveStandard::getActiveStmtIterator(OA_ptr<Alias::CallContext> cc) 
{
  OA_ptr<StmtHandleIterator> retval;
  OA_ptr<std::set<StmtHandle> > aStmtSet;
  if (mCCToActiveStmtSetMap.find(cc) == mCCToActiveStmtSetMap.end()) {
    aStmtSet = new std::set<StmtHandle>;
  } else {
    aStmtSet = mCCToActiveStmtSetMap[cc];
  }
  retval = new ActiveStmtIterator(aStmtSet);
  return retval;
}


// ----------- active memrefs -------------

//! Return an iterator for set of active memory references
OA_ptr<MemRefHandleIterator> ActiveStandard::getActiveMemRefIterator() 
{
  OA_ptr<MemRefHandleIterator> retval;
  retval = new ActiveMemRefIterator(mActiveMemRefSet);
  return retval;
}

//! Return an iterator for set of active memory references for context
OA_ptr<MemRefHandleIterator> ActiveStandard::getActiveMemRefIterator(OA_ptr<Alias::CallContext> cc) 
{
  OA_ptr<MemRefHandleIterator> retval;
  OA_ptr<std::set<MemRefHandle> > aMemRefSet;
  if (mCCToActiveMemRefSetMap.find(cc) == mCCToActiveMemRefSetMap.end()) {
    aMemRefSet = new std::set<MemRefHandle>;
  } else {
    aMemRefSet = mCCToActiveMemRefSetMap[cc];
  }
  retval = new ActiveMemRefIterator(aMemRefSet);
  return retval;
}


// ----------- active syms -------------

//! Return an iterator for set of active symbols
OA_ptr<SymHandleIterator> ActiveStandard::getActiveSymIterator() 
{
  OA_ptr<SymHandleIterator> retval;
  retval = new ActiveSymIterator(mActiveSymSet);
  return retval;
}

//! Return an iterator for set of active symbols for context
OA_ptr<SymHandleIterator> ActiveStandard::getActiveSymIterator(OA_ptr<Alias::CallContext> cc) 
{
  OA_ptr<SymHandleIterator> retval;
  OA_ptr<std::set<SymHandle> > aSymSet;
  if (mCCToActiveSymSetMap.find(cc) == mCCToActiveSymSetMap.end()) {
    aSymSet = new std::set<SymHandle>;
  } else {
    aSymSet = mCCToActiveSymSetMap[cc];
  }
  retval = new ActiveSymIterator(aSymSet);
  return retval;
}



// ----------- is active tag -------------

//! Indicate whether the given sym is active or not
bool 
ActiveStandard::isActive(SymHandle sym)
{
  // an unknown location is active, therefore all symbols are active
  if (mUnknownLocActive) {
    return true;
  } else if (mActiveSymSet->find(sym) != mActiveSymSet->end()) {
    return true;
  } else {
    return false;
  }  
}

//! Indicate whether the given sym is active or not in this context
bool 
ActiveStandard::isActive(SymHandle sym, OA_ptr<Alias::CallContext> cc)
{
  // an unknown location is active, therefore all symbols are active
  if (mUnknownLocActive) {
    return true;
  } else if (mCCToActiveSymSetMap.find(cc) != mCCToActiveSymSetMap.end()) {
    OA_ptr<std::set<SymHandle> > aSymSet;
    aSymSet = mCCToActiveSymSetMap[cc];
    if (aSymSet->find(sym) != aSymSet->end()) {
      return true;
    } else {
      return false;
    }
  } else {
    return false;
  }  
}



// ----------- is active stmt -------------

//! Indicate whether the given stmt is active or not
bool ActiveStandard::isActive(StmtHandle stmt)
{
  if (mActiveStmtSet->find(stmt) != mActiveStmtSet->end()) {
    return true;
  } else {
    return false;
  }  
}

//! Indicate whether the given stmt is active or not in context
bool ActiveStandard::isActive(StmtHandle stmt, 
                              OA_ptr<Alias::CallContext> cc)
{
  if (mCCToActiveStmtSetMap.find(cc) != mCCToActiveStmtSetMap.end()) {
    OA_ptr<std::set<StmtHandle> > aStmtSet;
    aStmtSet = mCCToActiveStmtSetMap[cc];
    if (aStmtSet->find(stmt) != aStmtSet->end()) {
      return true;
    } else {
      return false;
    }
  } else {
    return false;
  }  
}


// ----------- is active memref -------------

//! Indicate whether the given memref is active or not
bool ActiveStandard::isActive(MemRefHandle memref)
{
  if (mActiveMemRefSet->find(memref) != mActiveMemRefSet->end()) {
    return true;
  } else {
    return false;
  }  
}

//! Indicate whether the given memref is active or not in context
bool ActiveStandard::isActive(MemRefHandle memref, 
                              OA_ptr<Alias::CallContext> cc)
{
  if (mCCToActiveMemRefSetMap.find(cc) != mCCToActiveMemRefSetMap.end()) {
    OA_ptr<std::set<MemRefHandle> > aMemRefSet;
    aMemRefSet = mCCToActiveMemRefSetMap[cc];
    if (aMemRefSet->find(memref) != aMemRefSet->end()) {
      return true;
    } else {
      return false;
    }
  } else {
    return false;
  }  
}

//*****************************************************************
// Construction methods 
//*****************************************************************

//! insert an active location
void ActiveStandard::insertTag(Alias::AliasTag tag) 
{ mActiveTagSet->insert(tag); }

//! insert an active Stmt
void ActiveStandard::insertStmt(StmtHandle stmt) 
{ mActiveStmtSet->insert(stmt); }

//! insert an active MemRef
void ActiveStandard::insertMemRef(MemRefHandle memref) 
{ mActiveMemRefSet->insert(memref); }

void ActiveStandard::insertSym(SymHandle sym)
{ mActiveSymSet->insert(sym); }

//*****************************************************************
// Construction methods for Contexts
//*****************************************************************

//! insert an active location
void ActiveStandard::insertTag(Alias::AliasTag tag, 
                               OA_ptr<Alias::CallContext> cc) 
{ // insert into non-cs results
  insertTag(tag);

  if (mCCToActiveTagSetMap.find(cc) == mCCToActiveTagSetMap.end()) {
    mCCToActiveTagSetMap[cc] = new DataFlow::DataFlowSetImpl<Alias::AliasTag>;
  }
  mCCToActiveTagSetMap[cc]->insert(tag);

  mHasCSResults = true;
}

//! insert an active Stmt
void ActiveStandard::insertStmt(StmtHandle stmt, 
                                OA_ptr<Alias::CallContext> cc) 
{ // insert into non-cs results
  mActiveStmtSet->insert(stmt); 
  
  if (mCCToActiveStmtSetMap.find(cc) == mCCToActiveStmtSetMap.end()) {
    mCCToActiveStmtSetMap[cc] = new std::set<StmtHandle>;
  }
  mCCToActiveStmtSetMap[cc]->insert(stmt);

  mHasCSResults = true;
}

//! insert an active MemRef
void ActiveStandard::insertMemRef(MemRefHandle memref, 
                                  OA_ptr<Alias::CallContext> cc) 
{ // insert into non-cs results
  mActiveMemRefSet->insert(memref);

  if (mCCToActiveMemRefSetMap.find(cc) == mCCToActiveMemRefSetMap.end()) {
    mCCToActiveMemRefSetMap[cc] = new std::set<MemRefHandle>;
  }
  mCCToActiveMemRefSetMap[cc]->insert(memref); 

  mHasCSResults = true;
}

void ActiveStandard::insertSym(SymHandle sym, 
                               OA_ptr<Alias::CallContext> cc)
{ // insert into non-cs results
  mActiveSymSet->insert(sym); 

  if (mCCToActiveSymSetMap.find(cc) == mCCToActiveSymSetMap.end()) {
    mCCToActiveSymSetMap[cc] = new std::set<SymHandle>;
  }
  mCCToActiveSymSetMap[cc]->insert(sym); 

  mHasCSResults = true;
}


//*****************************************************************
// Output
//*****************************************************************

//! incomplete output of info for debugging
void ActiveStandard::dump(std::ostream& os, OA_ptr<IRHandlesIRInterface> ir)
{
    os << "ActiveTagSet = " << std::endl;
    mActiveTagSet->dump(os,ir);

    os << std::endl << "ActiveSymSet = " << std::endl;
    OA_ptr<SymHandleIterator> symIterPtr;
    symIterPtr = new ActiveSymIterator(mActiveSymSet);
    for ( ; symIterPtr->isValid(); (*symIterPtr)++ ) {
        os << ir->toString(symIterPtr->current()) << std::endl;
    }
    os << "mUnknownLocActive = " << mUnknownLocActive << std::endl;

    os << std::endl << "ActiveStmtList = " << std::endl;
    OA_ptr<StmtHandleIterator> stmtIterPtr = getActiveStmtIterator();
    for ( ; stmtIterPtr->isValid(); (*stmtIterPtr)++ ) {
        os << ir->toString(stmtIterPtr->current()) << std::endl;
    }

    os << std::endl << "ActiveMemRefList = " << std::endl;
    OA_ptr<MemRefHandleIterator> memrefIterPtr = getActiveMemRefIterator();
    for ( ; memrefIterPtr->isValid(); (*memrefIterPtr)++ ) {
        os << ir->toString(memrefIterPtr->current()) << std::endl;
    }

    if (mHasCSResults) {

      os << "\nCSActiveTagSets =============== \n\n";
      {
        std::map<OA_ptr<Alias::CallContext>, OA_ptr<DataFlow::DataFlowSetImpl<Alias::AliasTag> > >::iterator mapIter;
        for (mapIter = mCCToActiveTagSetMap.begin();
             mapIter != mCCToActiveTagSetMap.end();
             ++mapIter)
          {
            
            (mapIter->first)->dump(os,*ir); // call context
            os << "\nactiveTagSet ==\n";
            (mapIter->second)->dump(os,ir);
            os << "\n\n";
          }
      }
      
      os << std::endl << "\nCSActiveSymSets =============== \n\n";
      {
        std::map<OA_ptr<Alias::CallContext>, OA_ptr<std::set<SymHandle> > >::iterator mapIter;
        for (mapIter = mCCToActiveSymSetMap.begin();
             mapIter != mCCToActiveSymSetMap.end();
             ++mapIter)
          {
            (mapIter->first)->dump(os,*ir); // call context
            os << "\nactiveSymSet ==\n";
            OA_ptr<SymHandleIterator> symIterPtr;
            symIterPtr = new ActiveSymIterator(mapIter->second);
            for ( ; symIterPtr->isValid(); (*symIterPtr)++ ) {
              os << "\t" << ir->toString(symIterPtr->current()) << std::endl;
            }
            os << "\n\n";
          }
      }

      os << "\n===== mUnknownLocActive = " << mUnknownLocActive << std::endl;
      
      os << std::endl << "\nCSActiveStmtLists =============== \n\n";
      {
        std::map<OA_ptr<Alias::CallContext>, OA_ptr<std::set<StmtHandle> > >::iterator mapIter;
        for (mapIter = mCCToActiveStmtSetMap.begin();
             mapIter != mCCToActiveStmtSetMap.end();
             ++mapIter)
          {
            (mapIter->first)->dump(os,*ir); // call context
            os << "\nactiveStmtSet ==\n";
            OA_ptr<StmtHandleIterator> stmtIterPtr;
            stmtIterPtr = getActiveStmtIterator(mapIter->first);
            for ( ; stmtIterPtr->isValid(); (*stmtIterPtr)++ ) {
              os << "\t" << ir->toString(stmtIterPtr->current()) << std::endl;
            }
            os << "\n\n";
          }
      }
      
      os << std::endl << "\nCSActiveMemRefLists =============== \n\n";
      {
        std::map<OA_ptr<Alias::CallContext>, OA_ptr<std::set<MemRefHandle> > >::iterator mapIter;
        for (mapIter = mCCToActiveMemRefSetMap.begin();
             mapIter != mCCToActiveMemRefSetMap.end();
             ++mapIter)
          {
            (mapIter->first)->dump(os,*ir); // call context
            os << "\nactiveMemRefSet ==\n";
            OA_ptr<MemRefHandleIterator> memRefIterPtr;
            memrefIterPtr = getActiveMemRefIterator(mapIter->first);
            for ( ; memrefIterPtr->isValid(); (*memrefIterPtr)++ ) {
              os << "\t" << ir->toString(memrefIterPtr->current()) << std::endl;
            }
            os << "\n\n";
          }
      }

    }

}


  } // end of Activity namespace
} // end of OA namespace
