/*! \file
  
  \brief Implementation of Activity::ActivePerStmt

  \author Michelle Strout
  \version $Id: InActivePerStmt.cpp,v 1.1 2005/07/01 02:49:56 mstrout Exp $

  Copyright (c) 2002-2005, Rice University <br>
  Copyright (c) 2004-2005, University of Chicago <br>
  Copyright (c) 2006, Contributors <br>
  All rights reserved. <br>
  See ../../../Copyright.txt for details. <br>
*/

#include <OpenAnalysis/Activity/ActivePerStmt.hpp>
#include <OpenAnalysis/Utils/OutputBuilder.hpp>

namespace OA {
  namespace Activity {

ActivePerStmt::ActivePerStmt() 
{ 
  mNumIter = 0;
  mHasCSResults = false;
}


OA_ptr<DataFlow::DFSetBitImpl<Alias::AliasTag> > 
ActivePerStmt::getInActiveSet(StmtHandle stmt)
{
    if (mInActive.find(stmt)!=mInActive.end()) {
        return mInActive[stmt];
    } else {
        OA_ptr<DataFlow::DFSetBitImpl<Alias::AliasTag> > emptySet; 
        emptySet = new DataFlow::DFSetBitImpl<Alias::AliasTag>(1);
        return emptySet;
    }
}

OA_ptr<DataFlow::DFSetBitImpl<Alias::AliasTag> > 
ActivePerStmt::getInActiveSet(StmtHandle stmt, OA_ptr<Alias::CallContext> cc)
{
  assert(mHasCSResults);

  // wrap stmt and cc
  Alias::StmtHwContext shwc(stmt,cc);

  if (mInActiveCS.find(shwc)!=mInActiveCS.end()) {
    return mInActiveCS[shwc];
  } else {
    OA_ptr<DataFlow::DFSetBitImpl<Alias::AliasTag> > emptySet; 
    emptySet = new DataFlow::DFSetBitImpl<Alias::AliasTag>(1);
    return emptySet;
  }
}

OA_ptr<DataFlow::DFSetBitImpl<Alias::AliasTag> > 
ActivePerStmt::getOutActiveSet(StmtHandle stmt)
{
    if (mOutActive.find(stmt)!=mOutActive.end()) {
        return mOutActive[stmt];
    } else {
        OA_ptr<DataFlow::DFSetBitImpl<Alias::AliasTag> > emptySet; 
        emptySet = new DataFlow::DFSetBitImpl<Alias::AliasTag>(1);
        return emptySet;
    }
}

OA_ptr<DataFlow::DFSetBitImpl<Alias::AliasTag> > 
ActivePerStmt::getOutActiveSet(StmtHandle stmt, OA_ptr<Alias::CallContext> cc)
{
  assert(mHasCSResults);

  // wrap stmt and cc
  Alias::StmtHwContext shwc(stmt,cc);

  if (mOutActiveCS.find(shwc)!=mOutActiveCS.end()) {
    return mOutActiveCS[shwc];
  } else {
    OA_ptr<DataFlow::DFSetBitImpl<Alias::AliasTag> > emptySet; 
    emptySet = new DataFlow::DFSetBitImpl<Alias::AliasTag>(1);
    return emptySet;
  }
}


OA_ptr<DataFlow::DFSetBitImpl<Alias::AliasTag> >
ActivePerStmt::getInActiveSet(CallHandle call)
{
    if (mCallInActive.find(call)!=mCallInActive.end()) {
        return mCallInActive[call];
    } else {
        OA_ptr<DataFlow::DFSetBitImpl<Alias::AliasTag> > emptySet;
        emptySet = new DataFlow::DFSetBitImpl<Alias::AliasTag>(1);
        return emptySet;
    }
}

OA_ptr<DataFlow::DFSetBitImpl<Alias::AliasTag> >
ActivePerStmt::getInActiveSet(CallHandle call, OA_ptr<Alias::CallContext> cc)
{
  assert(mHasCSResults);

  // wrap call and cc
  Alias::CallHwContext chwc(call,cc);

  if (mCallInActiveCS.find(chwc)!=mCallInActiveCS.end()) {
    return mCallInActiveCS[chwc];
  } else {
    OA_ptr<DataFlow::DFSetBitImpl<Alias::AliasTag> > emptySet;
    emptySet = new DataFlow::DFSetBitImpl<Alias::AliasTag>(1);
    return emptySet;
  }
}

OA_ptr<DataFlow::DFSetBitImpl<Alias::AliasTag> >
ActivePerStmt::getOutActiveSet(CallHandle call)
{
    if (mCallOutActive.find(call)!=mCallOutActive.end()) {
        return mCallOutActive[call];
    } else {
        OA_ptr<DataFlow::DFSetBitImpl<Alias::AliasTag> > emptySet;
        emptySet = new DataFlow::DFSetBitImpl<Alias::AliasTag>(1);
        return emptySet;
    }
}

OA_ptr<DataFlow::DFSetBitImpl<Alias::AliasTag> >
ActivePerStmt::getOutActiveSet(CallHandle call, OA_ptr<Alias::CallContext> cc)
{
  assert(mHasCSResults);

  // wrap call and cc
  Alias::CallHwContext chwc(call,cc);

  if (mCallOutActiveCS.find(chwc)!=mCallOutActiveCS.end()) {
    return mCallOutActiveCS[chwc];
  } else {
    OA_ptr<DataFlow::DFSetBitImpl<Alias::AliasTag> > emptySet;
    emptySet = new DataFlow::DFSetBitImpl<Alias::AliasTag>(1);
    return emptySet;
  }
}

//*****************************************************************
// Construction methods 
//*****************************************************************

void 
ActivePerStmt::copyIntoInActive(StmtHandle s, 
                                OA_ptr<DataFlow::DFSetBitImpl<Alias::AliasTag> > dfset)
{
    OA_ptr<DataFlow::DataFlowSet> cloneDFset = dfset->clone();
    mInActive[s] = cloneDFset.convert<DataFlow::DFSetBitImpl<Alias::AliasTag> >();
}

void 
ActivePerStmt::copyIntoInActive(StmtHandle s, OA_ptr<Alias::CallContext> cc,
                                OA_ptr<DataFlow::DFSetBitImpl<Alias::AliasTag> > dfset)
{
  // wrap stmt and cc
  Alias::StmtHwContext shwc(s,cc);

  OA_ptr<DataFlow::DataFlowSet> cloneDFset = dfset->clone();
  mInActiveCS[shwc] = cloneDFset.convert<DataFlow::DFSetBitImpl<Alias::AliasTag> >();

  mHasCSResults = true;
}

void
ActivePerStmt::copyIntoOutActive(StmtHandle s, 
                                 OA_ptr<DataFlow::DFSetBitImpl<Alias::AliasTag> > dfset)
{
    OA_ptr<DataFlow::DataFlowSet> cloneDFset = dfset->clone();
    mOutActive[s] = cloneDFset.convert<DataFlow::DFSetBitImpl<Alias::AliasTag> >();
}

void
ActivePerStmt::copyIntoOutActive(StmtHandle s, OA_ptr<Alias::CallContext> cc,
                                 OA_ptr<DataFlow::DFSetBitImpl<Alias::AliasTag> > dfset)
{
  // wrap stmt and cc
  Alias::StmtHwContext shwc(s,cc);

  OA_ptr<DataFlow::DataFlowSet> cloneDFset = dfset->clone();
  mOutActiveCS[shwc] = cloneDFset.convert<DataFlow::DFSetBitImpl<Alias::AliasTag> >();

  mHasCSResults = true;
}

void 
ActivePerStmt::copyIntoInActive(CallHandle c,
                                OA_ptr<DataFlow::DFSetBitImpl<Alias::AliasTag> > dfset)
{
    OA_ptr<DataFlow::DataFlowSet> cloneDFset = dfset->clone();
    mCallInActive[c] = cloneDFset.convert<DataFlow::DFSetBitImpl<Alias::AliasTag> >();
}

void 
ActivePerStmt::copyIntoInActive(CallHandle c, OA_ptr<Alias::CallContext> cc,
                                OA_ptr<DataFlow::DFSetBitImpl<Alias::AliasTag> > dfset)
{
  // wrap call and cc
  Alias::CallHwContext chwc(c,cc);

  OA_ptr<DataFlow::DataFlowSet> cloneDFset = dfset->clone();
  mCallInActiveCS[chwc] = cloneDFset.convert<DataFlow::DFSetBitImpl<Alias::AliasTag> >();

  mHasCSResults = true;
}

void 
ActivePerStmt::copyIntoOutActive(CallHandle c,
                                 OA_ptr<DataFlow::DFSetBitImpl<Alias::AliasTag> > dfset)
{
    OA_ptr<DataFlow::DataFlowSet> cloneDFset = dfset->clone();
    mCallOutActive[c] = cloneDFset.convert<DataFlow::DFSetBitImpl<Alias::AliasTag> >();
}

void 
ActivePerStmt::copyIntoOutActive(CallHandle c, OA_ptr<Alias::CallContext> cc,
                                 OA_ptr<DataFlow::DFSetBitImpl<Alias::AliasTag> > dfset)
{
  // wrap call and cc
  Alias::CallHwContext chwc(c,cc);

  OA_ptr<DataFlow::DataFlowSet> cloneDFset = dfset->clone();
  mCallOutActiveCS[chwc] = cloneDFset.convert<DataFlow::DFSetBitImpl<Alias::AliasTag> >();

  mHasCSResults = true;
}


//*****************************************************************
// Annotation Interface
//*****************************************************************
void ActivePerStmt::output(IRHandlesIRInterface &ir) const
{
  if (!mHasCSResults) {
    sOutBuild->mapStart("StmtActiveSets","StmtHandle","ActiveAliasTagSet"); {
      std::map<StmtHandle, OA_ptr<DataFlow::DFSetBitImpl<Alias::AliasTag> > >::const_iterator mapIter;
      for (mapIter = mInActive.begin(); mapIter != mInActive.end(); mapIter++) {
        StmtHandle stmt = mapIter->first;
        OA_ptr<DataFlow::DFSetBitImpl<Alias::AliasTag> > inActiveSet = mapIter->second;
        const OA_ptr<DataFlow::DFSetBitImpl<Alias::AliasTag> > outActiveSet =
          mOutActive.find(stmt)->second;
        sOutBuild->mapEntryStart(); {
          sOutBuild->mapKeyStart(); {
            sOutBuild->outputIRHandle(stmt,ir);
          } sOutBuild->mapKeyEnd();
          sOutBuild->mapValueStart(); {
            sOutBuild->objStart("InActiveSet"); {
              inActiveSet->output(ir);
            } sOutBuild->objEnd("InActiveSet");
            sOutBuild->objStart("OutActiveSet"); {
              if(!outActiveSet.ptrEqual(0)) {
                outActiveSet->output(ir);
              }
            } sOutBuild->objEnd("OutActiveSet");
          } sOutBuild->mapValueEnd();
        } sOutBuild->mapEntryEnd();
      }
    } sOutBuild->mapEnd("StmtActiveSets");
    sOutBuild->mapStart("CallActiveSets","CallHandle","ActiveAliasTagSet"); {
      std::map<CallHandle, OA_ptr<DataFlow::DFSetBitImpl<Alias::AliasTag> > >::const_iterator mapIter;
      for (mapIter = mCallInActive.begin(); mapIter != mCallInActive.end(); mapIter++) {
        CallHandle call = mapIter->first;
        OA_ptr<DataFlow::DFSetBitImpl<Alias::AliasTag> > inActiveSet = mapIter->second;
        const OA_ptr<DataFlow::DFSetBitImpl<Alias::AliasTag> > outActiveSet =
          mCallOutActive.find(call)->second;
        sOutBuild->mapEntryStart(); {
          sOutBuild->mapKeyStart(); {
            sOutBuild->outputIRHandle(call,ir);
          } sOutBuild->mapKeyEnd();
          sOutBuild->mapValueStart(); {
            sOutBuild->objStart("InActiveSet"); {
              inActiveSet->output(ir);
            } sOutBuild->objEnd("InActiveSet");
            sOutBuild->objStart("OutActiveSet"); {
              if(!outActiveSet.ptrEqual(0)) {
                outActiveSet->output(ir);
              }
            } sOutBuild->objEnd("OutActiveSet");
          } sOutBuild->mapValueEnd();
        } sOutBuild->mapEntryEnd();
      }
    } sOutBuild->mapEnd("CallActiveSets");
  } 
  else {
    sOutBuild->mapStart("CSStmtActiveSets","StmtHwContext","ActiveAliasTagSet"); {
      std::map<Alias::StmtHwContext, OA_ptr<DataFlow::DFSetBitImpl<Alias::AliasTag> > >::const_iterator mapIter;
      for (mapIter = mInActiveCS.begin(); mapIter != mInActiveCS.end(); mapIter++) {
        Alias::StmtHwContext shwc = mapIter->first;
        OA_ptr<DataFlow::DFSetBitImpl<Alias::AliasTag> > inActiveSet = mapIter->second;
        const OA_ptr<DataFlow::DFSetBitImpl<Alias::AliasTag> > outActiveSet =
          mOutActiveCS.find(shwc)->second;
        sOutBuild->mapEntryStart(); {
          sOutBuild->mapKeyStart(); {
            shwc.output(ir);
          } sOutBuild->mapKeyEnd();
          sOutBuild->mapValueStart(); {
            sOutBuild->objStart("CSInActiveSet"); {
              inActiveSet->output(ir);
            } sOutBuild->objEnd("CSInActiveSet");
            sOutBuild->objStart("CSOutActiveSet"); {
              if(!outActiveSet.ptrEqual(0)) {
                outActiveSet->output(ir);
              }
            } sOutBuild->objEnd("CSOutActiveSet");
          } sOutBuild->mapValueEnd();
        } sOutBuild->mapEntryEnd();
      }
    } sOutBuild->mapEnd("CSStmtActiveSets");
    sOutBuild->mapStart("CSCallActiveSets","CallHwContext","ActiveAliasTagSet"); {
      std::map<Alias::CallHwContext, OA_ptr<DataFlow::DFSetBitImpl<Alias::AliasTag> > >::const_iterator mapIter;
      for (mapIter = mCallInActiveCS.begin(); mapIter != mCallInActiveCS.end(); mapIter++) {
        Alias::CallHwContext chwc = mapIter->first;
        OA_ptr<DataFlow::DFSetBitImpl<Alias::AliasTag> > inActiveSet = mapIter->second;
        const OA_ptr<DataFlow::DFSetBitImpl<Alias::AliasTag> > outActiveSet =
          mCallOutActiveCS.find(chwc)->second;
        sOutBuild->mapEntryStart(); {
          sOutBuild->mapKeyStart(); {
            chwc.output(ir);
          } sOutBuild->mapKeyEnd();
          sOutBuild->mapValueStart(); {
            sOutBuild->objStart("CSInActiveSet"); {
              inActiveSet->output(ir);
            } sOutBuild->objEnd("CSInActiveSet");
            sOutBuild->objStart("CSOutActiveSet"); {
              if(!outActiveSet.ptrEqual(0)) {
                outActiveSet->output(ir);
              }
            } sOutBuild->objEnd("CSOutActiveSet");
          } sOutBuild->mapValueEnd();
        } sOutBuild->mapEntryEnd();
      }
    } sOutBuild->mapEnd("CSCallActiveSets");
  } 

}

//*****************************************************************
// Output
//*****************************************************************

//! incomplete output of info for debugging
void ActivePerStmt::dump(std::ostream& os, OA_ptr<IRHandlesIRInterface> ir)
{
  if (!mHasCSResults) {
    os << "============================ ActivePerStmt Stmt Results" << std::endl;

    // iterate over all stmts we know about
    std::map<StmtHandle, 
             OA_ptr<DataFlow::DFSetBitImpl<Alias::AliasTag> > >::iterator mapIter;
    for (mapIter = mInActive.begin(); mapIter != mInActive.end(); mapIter++) {
        StmtHandle s = mapIter->first;
        OA_ptr<DataFlow::DFSetBitImpl<Alias::AliasTag> > inActSet = mapIter->second;
        OA_ptr<DataFlow::DFSetBitImpl<Alias::AliasTag> > outActSet = mOutActive[s];
        if (outActSet.ptrEqual(0)) {
          outActSet = new DataFlow::DFSetBitImpl<Alias::AliasTag>(1);
        }

        os << "StmtHandle(" << ir->toString(s) << ")" 
           << std::endl << "\tInActive: " << std::endl;
        inActSet->dump(os,ir);
        os << "\tOutActive: " << std::endl;
        outActSet->dump(os,ir);
    }
    os << "============================ ActivePerStmt Call Results" << std::endl;

    // iterate over all calls we know about
    std::map<CallHandle, 
             OA_ptr<DataFlow::DFSetBitImpl<Alias::AliasTag> > >::iterator cmapIter;
    for (cmapIter = mCallInActive.begin(); cmapIter != mCallInActive.end(); cmapIter++) {
        CallHandle c = cmapIter->first;
        OA_ptr<DataFlow::DFSetBitImpl<Alias::AliasTag> > inActSet = cmapIter->second;
        OA_ptr<DataFlow::DFSetBitImpl<Alias::AliasTag> > outActSet = mCallOutActive[c];
        if (outActSet.ptrEqual(0)) {
          outActSet = new DataFlow::DFSetBitImpl<Alias::AliasTag>(1);
        }

        os << "CallHandle(" << ir->toString(c) << ")" 
           << std::endl << "\tInActive: " << std::endl;
        inActSet->dump(os,ir);
        os << "\tOutActive: " << std::endl;
        outActSet->dump(os,ir);
    }
  }
  else {
    os << "============================ CSActivePerStmt" << std::endl;

    // iterate over all stmts we know about
    std::map<Alias::StmtHwContext, 
             OA_ptr<DataFlow::DFSetBitImpl<Alias::AliasTag> > >::iterator mapIter;
    for (mapIter = mInActiveCS.begin(); mapIter != mInActiveCS.end(); mapIter++) {
        Alias::StmtHwContext shwc = mapIter->first;
        OA_ptr<DataFlow::DFSetBitImpl<Alias::AliasTag> > inActSet = mapIter->second;
        OA_ptr<DataFlow::DFSetBitImpl<Alias::AliasTag> > outActSet = mOutActiveCS[shwc];
        if (outActSet.ptrEqual(0)) {
          outActSet = new DataFlow::DFSetBitImpl<Alias::AliasTag>(1);
        }

        shwc.dump(os, *ir);
        os <<  "\tCSInActive: " << std::endl;
        inActSet->dump(os,ir);
        os << "\tCSOutActive: " << std::endl;
        outActSet->dump(os,ir);
    }
    os << "============================ CSActivePerStmt Call Results" << std::endl;

    // iterate over all calls we know about
    std::map<Alias::CallHwContext, 
             OA_ptr<DataFlow::DFSetBitImpl<Alias::AliasTag> > >::iterator cmapIter;
    for (cmapIter = mCallInActiveCS.begin(); cmapIter != mCallInActiveCS.end(); cmapIter++) {
        Alias::CallHwContext chwc = cmapIter->first;
        OA_ptr<DataFlow::DFSetBitImpl<Alias::AliasTag> > inActSet = cmapIter->second;
        OA_ptr<DataFlow::DFSetBitImpl<Alias::AliasTag> > outActSet = mCallOutActiveCS[chwc];
        if (outActSet.ptrEqual(0)) {
          outActSet = new DataFlow::DFSetBitImpl<Alias::AliasTag>(1);
        }

        chwc.dump(os, *ir); 
        os << "\tInActive: " << std::endl;
        inActSet->dump(os,ir);
        os << "\tOutActive: " << std::endl;
        outActSet->dump(os,ir);
    }
  }
}



  } // end of Activity namespace
} // end of OA namespace
