
// tps (01/14/2010) : Switching from rose.h to sage3.
#include "sage3basic.h"

#include "checkIsModifiedFlag.h"
#include "AstPDFGeneration.h"
#include "AstDOTGeneration.h"
#include "wholeAST_API.h"

#ifdef _MSC_VER
#include <direct.h>	// getcwd
#endif

// DQ (10/11/2007): This is commented out to avoid use of this mechanism.
// #include <copy_unparser.h>

// DQ (10/14/2010):  This should only be included by source files that require it.
// This fixed a reported bug which caused conflicts with autoconf macros (e.g. PACKAGE_BUGREPORT).
// Interestingly it must be at the top of the list of include files.
#include "rose_config.h"

// DQ (12/31/2005): This is OK if not declared in a header file
using namespace std;

// global variable for turning on and off internal debugging
int ROSE_DEBUG = 0;

#if 1
// ArrayAssignmentUsingTransformationGrammar* globalArrayAssignmentUsingTransformationGrammar = NULL;

// CW: here we should definitly find a better way
// to specify the cache parameters
const int roseTargetCacheSize     = 8192;
const int roseTargetCacheLineSize = 32;
// cacheInfo roseTargetCacheInfo(roseTargetCacheSize,roseTargetCacheLineSize);

// What is this and who put it here?
// unsigned int *uint_global_dbug_ptr;

#endif

// DQ (8/10/2004): This was moved to the SgFile a long time ago and should not be used any more)
// bool ROSE::verbose                 = false;
// DQ (8/11/2004): build a global state here
// int ROSE::roseVerbose = 0;


#define OUTPUT_TO_FILE true
#define DEBUG_COPY_EDIT false


// DQ (4/17/2010): This function must be define if C++ support in ROSE is disabled.
std::string edgVersionString()
{
#ifdef ROSE_BUILD_CXX_LANGUAGE_SUPPORT
  string edg_version = string("edg-") + StringUtility::numberToString(ROSE_EDG_MAJOR_VERSION_NUMBER) + "." + StringUtility::numberToString(ROSE_EDG_MINOR_VERSION_NUMBER);
#else
  string edg_version = "unknown (EDG is disabled)";
#endif
  return edg_version;
}




// DQ (4/17/2010): Added OFP version number support.
// DQ (2/12/2010): When we have a mechanism to get the version number of OFP, put it here.
std::string ofpVersionString()
   {
  // Need to make sure that ROSE can get a version number independent of Fortran support 
  // being installed or include information in the return string when OFP is not installed.
  // return "unknown";
#ifdef ROSE_BUILD_FORTRAN_LANGUAGE_SUPPORT
     string ofp_version = string("ofp-") + StringUtility::numberToString(ROSE_OFP_MAJOR_VERSION_NUMBER) + "." + StringUtility::numberToString(ROSE_OFP_MINOR_VERSION_NUMBER) + "." + StringUtility::numberToString(ROSE_OFP_PATCH_VERSION_NUMBER);
#else
     string ofp_version = "unknown (OFP is disabled)";
#endif

     return ofp_version;
   }

// DQ (11/1/2009): replaced "version()" with separate "version_number()" and "version_message()" functions.
std::string version_message()
   {
  // returns a string with the version message for ROSE.
  // return "\nROSE (pre-release alpha version: " + version_number() + ") \n";

  // DQ (2/12/2010): Added EDG version number to make our versioning more clear.
  // return "ROSE (pre-release beta version: " + version_number() + ")";
     extern string edgVersionString();
     extern string ofpVersionString();
  // return "ROSE (pre-release beta version: " + version_number() + " using EDG C/C++ front-end version " + edgVersionString() + ")";
     return "ROSE (pre-release beta version: " + version_number() + ")"
          "\n   --- using EDG C/C++ front-end version: " + edgVersionString() +
          "\n   --- using OFP Fortran parser version: " + ofpVersionString();
   }

// DQ (11/1/2009): replaced "version()" with separate "version_number()" and "version_message()" functions.
std::string version_number()
   {
  // returns a string containing the current version number
  // the VERSION macro is defined at compile time on the 
  // compile command line by the Makefile generated by automake
     return VERSION;
   }


/*! \brief Call to frontend, processes commandline and generates a SgProject object.

    This function represents a simple interface to the use of ROSE as a library.
    The commandline is processed and the return parameter is the generate SgProject object.
 */

// #include "sageCommonSourceHeader.h"
// extern an_il_header il_header;
//
SgProject*
frontend (int argc, char** argv)
   {
     return frontend(std::vector<std::string>(argv, argv + argc));
   }

SgProject*
frontend (const std::vector<std::string>& argv)
   {
  // DQ (6/14/2007): Added support for timing of high level frontend function.
     TimingPerformance timer ("ROSE frontend():");

  // Syncs C++ and C I/O subsystems!
     ios::sync_with_stdio();

  // make sure that there is some sort of commandline (at least a file specified)
     if (argv.size() == 1)
        {
       // ROSE::usage(1);      // Print usage and exit with exit status == 1
          SgFile::usage(1);      // Print usage and exit with exit status == 1
        }
  // Error code checks and reporting are done in SgProject constructor
  // return new SgProject (argc,argv);
     SgProject* project = new SgProject (argv);
     ROSE_ASSERT (project != NULL);

  // DQ (9/6/2005): I have abandoned this form or prelinking (AT&T C Front style).
  // To be honest I find this level of technology within ROSE to be embarassing...
  // We not handle prelinking by generating all required template instantiations
  // as static functions.  A more global based prelinker will be built at some
  // point and will likely utilize the SGLite database or some other auxiliary file
  // mechansism.
  // DQ (3/31/2004): If there are templates used then we need to modify the *.ti file build by EDG.
  // buildTemplateInstantiationSupportFile ( project );

     /* Make sure the isModified boolean is clear for all newly-parsed nodes. */
     checkIsModifiedFlag(project);

     return project;
   }

/*! \brief Call to build SgProject with empty SgFiles.

    This function represents a simple interface to build a SgProject with all the
    SgFiles in place any initialized properly but containing a SgGlobal with no declarations.
    The purpose of this function is to build a SgProject with SgFile object for each file 
    on the command line, and allow for the frontend to be called separately for each file.
    The commandline is completely processed so that all information is in each SgFile of the 
    SgProject.
    This way specific files can be processed conditionally.  This mechanism can be used to
    build support for exclusion of specific files or for exclusion of files in a specific
    subdirectory.  Or to support specialized handling of files with a specific extension, etc.

 */

SgProject*
frontendShell (int argc, char** argv)
   {
     return frontendShell(std::vector<std::string>(argv, argv + argc));
   }

SgProject*
frontendShell (const std::vector<std::string>& argv)
   {
  // Convert this to a list of strings to simplify editing (adding new option)
     Rose_STL_Container<string> commandLineList = argv;
     printf ("frontendShell (top): argv = \n%s \n",StringUtility::listToString(commandLineList).c_str());

  // Invoke ROSE commandline option to skip internal frontend processing (we will 
  // call the fronend explicitly for selected files, after construction of SgProject).
     commandLineList.push_back("-rose:skip_rose");

  // Build the SgProject, but if the above option was used this will only build empty SgFile nodes 
     SgProject* project = frontend(commandLineList);
     ROSE_ASSERT(project != NULL);

     project->display("In frontendShell(), after frontend()");

     SgFilePtrList::const_iterator i = project->get_fileList().begin();
     while (i != project->get_fileList().end())
        {
       // Get the local command line so that we can remove the "-rose:skip_rose" option
          vector<string> local_argv = (*i)->get_originalCommandLineArgumentList();

       // Note that we have to remove the "-rose:skip_rose" option that was saved
          CommandlineProcessing::removeArgs (local_argv,"-rose:skip_rose");
       // printf ("Remove -rose:skip_rose: argv = \n%s \n",StringUtility::listToString(CommandlineProcessing::generateArgListFromArgcArgv (local_argc,local_argv)).c_str());

          printf ("frontendShell (after): argv = \n%s \n",StringUtility::listToString(commandLineList).c_str());

       // Set the new commandline (without the "-rose:skip_rose" option)
          (*i)->set_originalCommandLineArgumentList(local_argv);

       // Things set by "-rose:skip_rose" option, which must be unset (reset to default valees)!
          (*i)->set_skip_transformation(false);
          (*i)->set_disable_edg_backend(false);

       // Leave this set to true so that the frontend can set it if the frontend is called for this SgFile
       // (*i)->set_skip_unparse(false);

          (*i)->set_useBackendOnly(false);
          (*i)->set_skipfinalCompileStep(false);

       // Skip all processing of comments
          (*i)->set_skip_commentsAndDirectives(false);
       // (*i)->set_collectAllCommentsAndDirectives(false);

       // file->display("After Remove -rose:skip_rose");

       // increment the file list iterator
          i++;
        }

     return project;
   }

/*! \brief Call to backend, generates either object file or executable.

    This function operates in two modes:
        1) If source files were specified on the command line, then it calls 
           unparser and compiles generated file.
        2) If no source files are present then it operates as a linker processing
           all specified object files.
    If no source files or object files are specified then we return a error.

    This function represents a simple interface to the use of ROSE as a library.

     At this point in the control flow we have returned from the processing via the 
     EDG frontend (or skipped it if that option was specified).
     The following has been done or explicitly skipped if such options were specified 
     on the commandline:
        1) The application program has been parsed
        2) All AST's have been build (one for each grammar)
        3) The transformations have been edited into the C++ AST
        4) The C++ AST has been unparsed to form the final output file (all code has 
           been generated into a different filename "rose_<original file name>.C")

    \internal The error code is returned, but it might be appropriate to make
              it more similar to the frontend() function and its handling of 
              the error code.
 */
int
backend ( SgProject* project, UnparseFormatHelp *unparseFormatHelp, UnparseDelegate* unparseDelagate )
   {
  // DQ (7/12/2005): Introduce tracking of performance of ROSE.
     TimingPerformance timer ("AST Object Code Generation (backend):");

     int finalCombinedExitStatus = 0;

     if (project->get_binary_only() == true)
        {
          ROSE_ASSERT(project != NULL);

       // DQ (8/21/2008): Only output a message when we we use verbose option.
          if ( SgProject::get_verbose() >= 1 )
              printf ("Note: Binary executables are unparsed, but not passed to gcc as assembly source code \n");

          project->skipfinalCompileStep(true);
        }

     if ( SgProject::get_verbose() >= BACKEND_VERBOSE_LEVEL )
          printf ("Inside of backend(SgProject*) \n");

  // printf ("   project->get_useBackendOnly() = %s \n",project->get_useBackendOnly() ? "true" : "false");
     if (project->get_useBackendOnly() == false)
        {
       // Add forward references for instantiated template functions and member functions 
       // (which are by default defined at the bottom of the file (but should be declared 
       // at the top once we know what instantiations should be built)).  They must be 
       // defined at the bottom since they could call other functions not yet declared in 
       // the file.  Note that this fixup is required since we have skipped the class template 
       // definitions which would contain the declarations that we are generating.  We might 
       // need that as a solution at some point if this fails to be sufficently robust.
          if ( SgProject::get_verbose() >= BACKEND_VERBOSE_LEVEL )
               printf ("Calling fixupInstantiatedTemplates() \n");

       // DQ (9/6/2005): I think this is handled separately within post processing 
       // (either that or they are just marked for output n the post processing)
       // fixupInstantiatedTemplates(project);

       // generate C++ source code
          if ( SgProject::get_verbose() >= BACKEND_VERBOSE_LEVEL )
               printf ("Calling project->unparse() \n");

          project->unparse(unparseFormatHelp,unparseDelagate);

          if ( SgProject::get_verbose() >= BACKEND_VERBOSE_LEVEL )
               cout << "source file(s) generated. (from AST)" << endl;
        }

  // printf ("Inside of backend(SgProject*): SgProject::get_verbose() = %d \n",SgProject::get_verbose());
  // printf ("Inside of backend(SgProject*): project->numberOfFiles() = %d \n",project->numberOfFiles());

  // DQ (1/25/2010): We have to now test for both numberOfFiles() and numberOfDirectories(),
  // or perhaps define a more simple function to use more directly.
  // if (project->numberOfFiles() > 0)
     if (project->numberOfFiles() > 0 || project->numberOfDirectories() > 0)
        {
       // Compile generated C++ source code with vendor compiler.
       // Generate object file (required for further template processing 
       // if templates exist).
          if ( SgProject::get_verbose() >= BACKEND_VERBOSE_LEVEL )
               printf ("Calling project->compileOutput() \n");

          finalCombinedExitStatus = project->compileOutput();
        }
       else
  // if (project->get_compileOnly() == false)
        {
          if ( SgProject::get_verbose() >= BACKEND_VERBOSE_LEVEL )
               printf ("   project->get_compileOnly() = %s \n",project->get_compileOnly() ? "true" : "false");

       // DQ (5/20/2005): If we have not permitted templates to be instantiated during initial 
       // compilation then we have to do the prelink step (this is however still new and somewhat 
       // problematic (buggy?)).  It relies upon the EDG mechansisms which are not well understood.
          bool callTemplateInstantation = (project->get_template_instantiation_mode() == SgProject::e_none);

          if (callTemplateInstantation == true)
             {
            // DQ (9/6/2005): I think that this is no longer needed
               printf ("I don't think we need to call instantiateTemplates() any more! \n");
               ROSE_ASSERT(false);

            // The instantiation of templates can cause new projects (sets of source files) 
            // to be generated, but since the object files are already processed this is 
            // not an issue here.  A project might, additionally, keep track of the ASTs
            // associated with the different phases of instantions of templates.
               if ( SgProject::get_verbose() >= BACKEND_VERBOSE_LEVEL )
                    printf ("Calling instantiateTemplates() \n");

               printf ("Skipping template support in backend(SgProject*) \n");
            // instantiateTemplates (project);
             }

          if ( SgProject::get_verbose() >= BACKEND_VERBOSE_LEVEL )
               printf ("Calling project->link() \n");

       // DQ (10/15/2005): Trap out case of C programs where we want to make sure that we don't use the C++ compiler to do our linking!
       // This could be done in the 
          if (project->get_C_only() == true)
             {
                printf ("Link using the C language linker (when handling C programs) = %s \n",BACKEND_C_COMPILER_NAME_WITH_PATH);
            // finalCombinedExitStatus = project->link("gcc");
               finalCombinedExitStatus = project->link(BACKEND_C_COMPILER_NAME_WITH_PATH);
             }
            else
             {
            // Use the default name for C++ compiler (defined at configure time)
               if ( SgProject::get_verbose() >= BACKEND_VERBOSE_LEVEL )
                  printf ("Link using the default linker (when handling non-C programs) = %s \n",BACKEND_CXX_COMPILER_NAME_WITH_PATH);
               finalCombinedExitStatus = project->link(BACKEND_CXX_COMPILER_NAME_WITH_PATH);
             }

       // printf ("DONE with link! \n");
        }

  // Message from backend to user.
  // Avoid all I/O to stdout if useBackendOnly == true.
  // if (project->get_useBackendOnly() == false)
     if ( SgProject::get_verbose() >= 1 )
          cout << "source file(s) compiled with vendor compiler. (exit status = " << finalCombinedExitStatus << ").\n" << endl;

  // Set the final error code to be returned to the user.
     project->set_backendErrorCode(finalCombinedExitStatus);
     return project->get_backendErrorCode();
   }


int
backendCompilesUsingOriginalInputFile ( SgProject* project )
   {
  // DQ (8/24/2009):
  // To work with existing makefile systems, we want to force an object file to be generated.
  // So we want to call the backend compiler on the original input file (instead of generating
  // an new file from the AST and running it through the backend).  The whole point is to 
  // gnerate a object file.  The effect is that this does a less agressive test of ROSE
  // but test only the parts required for analysis tools instead of transformation tools.
  // This avoids some programs in the name qualification support that is called by the backend
  // and permits testing of the parts of ROSE relevant for analysis tools (e.g. Compass).
  // Of course we eventually want everything to work, but I want to test the compilation of 
  // ROSE using ROSE as part of test to get Compass running regularly on ROSE.

  // Note that the command generated may have to be fixup later to include more subtle details 
  // required to link libraries, etc.  At present this function only handles the support required
  // to build an object file.
     string commandLineToGenerateObjectFile;

     enum language_enum
        {
          e_none    = 0,
          e_c       = 1, 
          e_cxx     = 2, 
          e_fortran = 3, 
          e_last_language 
        };

     language_enum language = e_none;
     language = project->get_C_only()       ? e_c       : language;
     language = project->get_Cxx_only()     ? e_cxx     : language;
     language = project->get_Fortran_only() ? e_fortran : language;

  // ROSE_ASSERT(language != e_none);
     if (language == e_none)
        {
       // DQ (4/7/2010): Set the default language for ROSE to be C++
       // This will mean that linking fortran object files will not be possible with ROSE.
       // but at least configure will work propoerly since it will have a valid default.
       // If we add state to SgProject, then we could set the default, but also allow it
       // to be overriden using -rose:C or -rose:Cxx or -rose:Fortran options.
          language = e_cxx;
        }

     switch (language)
        {
          case e_c       : commandLineToGenerateObjectFile = BACKEND_C_COMPILER_NAME_WITH_PATH;       break;
          case e_cxx     : commandLineToGenerateObjectFile = BACKEND_CXX_COMPILER_NAME_WITH_PATH;     break;
          case e_fortran : commandLineToGenerateObjectFile = BACKEND_FORTRAN_COMPILER_NAME_WITH_PATH; break;

          default:
             {
            // Note that the default is C++, and that if there are no SgFile objects then there is no place to hold the default
            // since the SgProject does not have any such state to hold this information.  A better idea might be to give the
            // SgProject state so that it can hold if it is used with -rose:C or -rose:Cxx or -rose:Fortran on the command line.

               printf ("Default reached in switch in backendCompilesUsingOriginalInputFile() \n");
               printf ("   Note use options: -rose:C or -rose:Cxx or -rose:Fortran to specify which language backend compiler to link object files. \n");
               ROSE_ASSERT(false);
             }
        }

     int finalCombinedExitStatus = 0;
     if (project->numberOfFiles() > 0)
        {
          SgStringList originalCommandLineArgumentList = project->get_originalCommandLineArgumentList();

       // DQ (2/20/2010): Added filtering of options that should not be passed to the vendor compiler.
          SgFile::stripRoseCommandLineOptions(originalCommandLineArgumentList);
          SgFile::stripEdgCommandLineOptions(originalCommandLineArgumentList);

          SgStringList::iterator it = originalCommandLineArgumentList.begin();

       // Iterate past the name of the compiler being called (arg[0]).
          if (it != originalCommandLineArgumentList.end())
               it++;

       // Make a list of the remaining command line arguments
          for (SgStringList::iterator i = it; i != originalCommandLineArgumentList.end(); i++)
             {
               commandLineToGenerateObjectFile += " " + *i;
             }
       // printf ("From originalCommandLineArgumentList(): commandLineToGenerateObjectFile = %s \n",commandLineToGenerateObjectFile.c_str());

          if ( SgProject::get_verbose() >= 1 )
             {
               printf ("numberOfFiles() = %d commandLineToGenerateObjectFile = \n     %s \n",project->numberOfFiles(),commandLineToGenerateObjectFile.c_str());
             }

       // DQ (12/28/2010): If we specified to NOT compile the input code then don't do so even when it is the 
       // original code. This is important for Fortran 2003 test codes that will not compile with gfortran and 
       // for which the tests/testTokenGeneration.C translator uses this function to generate object files.
       // finalCombinedExitStatus = system (commandLineToGenerateObjectFile.c_str());
          if (project->get_skipfinalCompileStep() == false)
             {
               finalCombinedExitStatus = system (commandLineToGenerateObjectFile.c_str());
             }
        }
       else
        {
       // Note that in general it is not possible to tell whether to use gcc, g++, or gfortran to do the linking.
       // When we just have a list of object files then we can't assume anything (and project->get_C_only() will be false).

       // Note that commandLineToGenerateObjectFile is just the name of the backend compiler to use!
       // finalCombinedExitStatus = project->link(commandLineToGenerateObjectFile);
          finalCombinedExitStatus = project->link(commandLineToGenerateObjectFile);
        }

     return finalCombinedExitStatus;
   }



int
backendGeneratesSourceCodeButCompilesUsingOriginalInputFile ( SgProject* project )
   {
  // DQ (2/6/2010): This function is a step between calling the backend()
  // and calling backendCompilesUsingOriginalInputFile().  It it used
  // the test the generation of the source code, but not the compilation of
  // it using the backend (vendor) compiler.  This is used to test ROSE.

  // Users are likely to either want to use backend() to generate the source 
  // code for there project and it compiled (e.g. for optimization) or call
  // backendCompilesUsingOriginalInputFile() to process the input code and
  // then generate object files or executables from the original code 
  // (e.g for analysis).

  // This instance of complexity is why this needs to be a separate backend function.
  // Note that file->get_skip_unparse() will be false when the "-E" option, and
  // the unparse() function will properly assert that it should be true.
     if (project->get_skip_unparse() == false)
        {
          project->unparse();
        }

     return backendCompilesUsingOriginalInputFile(project);
   }


int
copy_backend( SgProject* project, UnparseFormatHelp *unparseFormatHelp )
   {
  // This is part of the copy-based unparser (from Qing).
  // This function calls the unparseFile function with a 
  // CopyUnparser object (derived from UnparseDelegate)
  // to control the unparsing and substitute text based 
  // copying for code generation from the AST.

  // This function does not presently have the same semantics as the "backend()".
  // The code above could be refactored to permit both backend function to more
  // readily have the same semantics later.

#if 0
  // DQ (10/11/2007): This mechanism has been replaced by a token based mechanism to support exact code generation.

     for (int i=0; i < project->numberOfFiles(); ++i)
        {
          SgFile & file = project->get_file(i);

       // Build this on the stack for each SgFile
          CopyUnparser repl(file);

       // DQ (3/18/2006): Modified to handle formating options
          unparseFile(&file,unparseFormatHelp,&repl);
        }
#else
     printf ("Error: Inside of copy_backend(), the copy backend has been disabled in favor of a token based mechanism for unparsing. \n");
     ROSE_ASSERT(false);
#endif

  // DQ (5/19/2005): I had to make up a return value since one was not previously specified
     return 0;
   }

void
generatePDF ( const SgProject & project )
   {
  // DQ (6/14/2007): Added support for timing of the generatePDF() function.
     TimingPerformance timer ("ROSE generatePDF():");

     if ( SgProject::get_verbose() >= BACKEND_VERBOSE_LEVEL )
          printf ("Inside of generatePDF \n");

  // Output the source code file (as represented by the SAGE AST) as a PDF file (with bookmarks)
     AstPDFGeneration pdftest;
     SgProject & nonconstProject = (SgProject &) project;
     pdftest.generateInputFiles(&nonconstProject);
   }

void
generateDOT ( const SgProject & project, std::string filenamePostfix )
   {
  // DQ (7/4/2008): Added default parameter to support the filenamePostfix 
  // mechanism in AstDOTGeneration

  // DQ (6/14/2007): Added support for timing of the generateDOT() function.
     TimingPerformance timer ("ROSE generateDOT():");

     AstDOTGeneration astdotgen;
     SgProject & nonconstProject = (SgProject &) project;

  // Note that the use of generateInputFiles causes the graph to be generated 
  // for only the input source file and not any included header files. The 
  // result is a much smaller file (and generally a more useful one).
#if 0
  // This used to be the default, but it would output too much data (from include files).
     astdotgen.generate(&nonconstProject);
#else
  // DQ (9/1/2008): This is the default for the last long while, but the SgProject IR nodes 
  // is not being processed (which appears to be a bug). This is because in the implementation
  // of the generateInputFiles the function traverseInputFiles is called.
     astdotgen.generateInputFiles(&nonconstProject,DOTGeneration<SgNode*>::TOPDOWNBOTTOMUP,filenamePostfix);
#endif
   }

void
generateDOT_withIncludes ( const SgProject & project, std::string filenamePostfix )
   {
     TimingPerformance timer ("ROSE generateDOT_withIncludes():");

     AstDOTGeneration astdotgen;
     SgProject & nonconstProject = (SgProject &) project;

  // Note that the use of generateInputFiles causes the graph to be generated 
  // for only the input source file and not any included header files. The 
  // result is a much smaller file (and generally a more useful one).
#if 1
  // This used to be the default, but it would output too much data (from include files).
  // It is particularly useful when handling multiple files on the command line and 
  // traversing the files included from each file.
     astdotgen.generate(&nonconstProject);
#else
  // DQ (9/1/2008): This is the default for the last long while, but the SgProject IR nodes 
  // is not being processed (which appears to be a bug). This is because in the implementation
  // of the generateInputFiles the function traverseInputFiles is called.
     astdotgen.generateInputFiles(&nonconstProject,DOTGeneration<SgNode*>::TOPDOWNBOTTOMUP,filenamePostfix);
#endif
   }

void
generateDOTforMultipleFile ( const SgProject & project, std::string filenamePostfix )
   {
     TimingPerformance timer ("ROSE generateDOT():");

  // This is the best way to handle generation of DOT files where multiple files
  // are specified on the command line.  Later we may be able to filter out the
  // include files (but this is a bit difficult until generateInputFiles() can be
  // implemetned to call the evaluation of inherited and synchizied attributes.
     generateDOT_withIncludes(project,filenamePostfix);
   }

void
generateAstGraph ( const SgProject* project, int maxSize, std::string filenameSuffix )
   {
  // DQ (6/14/2007): Added support for timing of the generateAstGraph() function.
     TimingPerformance timer ("ROSE generateAstGraph():");

  // Generate a name from all the files on the command line
     string filename = SageInterface::generateProjectName(project, /* supressSuffix = */ true );

     filename += "_WholeAST";

     filename += filenameSuffix;

     int numberOfASTnodes = numberOfNodes();

     if ( SgProject::get_verbose() >= 1 )
          printf ("In generateAstGraph(): numberOfASTnodes = %d maxSize = %d filename = %s \n",numberOfASTnodes,maxSize,filename.c_str());

  // Compute the number of IR nodes for the AST
     if (numberOfASTnodes < maxSize)
        {
          generateWholeGraphOfAST(filename);
        }
       else
        {
          if ( SgProject::get_verbose() >= 1 )
               printf ("In generateAstGraph(): WHOLE AST greaph too large to generate. \n");
        }
   }





#if 0

// Include tree traversal for EDG abstract program tree
// #include "../EDG/src/displayTree.h"
#include "/frontend/EDG/EDG_3.3/src/displayTree.h"

//! Generate PDF file representing EDG's Abstract Syntax Tree
void
pdfPrintAbstractSyntaxTreeEDG ( SgFile *file )
   {
     char filename[256];

     printf ("## Dumping the EDG program tree to a PDF file ## \n");

  // Use the PDF file declared in the EDG/src/displayTree.C
     extern PDF* pdfGlobalFile;

     printf ("ROSE::getWorkingDirectory() = %s \n",ROSE::getWorkingDirectory());
     sprintf(filename,"%s/%s.edg.pdf",ROSE::getWorkingDirectory(),ROSE::stripPathFromFileName(ROSE::getFileName(file)));
     printf ("filename = %s \n",filename);

     ifstream sourceFile (ROSE::getFileName(file));
     if (!sourceFile)
          cerr << "ERROR opening sourceFile" << endl;
     ROSE_ASSERT (sourceFile);

     fprintf(stderr, "Creating PDFlib file '%s'!\n", filename);

#if 0
     #define MAX_BUFFER_LENGTH 10000

     char buffer[100][MAX_BUFFER_LENGTH];
     int i = 0;

     while (sourceFile.getline(buffer[i],100,'\n') && i < MAX_BUFFER_LENGTH)
        {
          printf ("string = %s \n",buffer[i]);
          i++;
        }

  // printf ("Exting after printing source file \n");
  // ROSE_ABORT();
#endif

  // Initialize the PDFLib library
     PDF_boot();

  // Build a PDF file
     PDF* pdfFile = PDF_new();
     ROSE_ASSERT (pdfFile != NULL);

  // Open the pdf file for writing
     if (PDF_open_file(pdfFile, filename) == -1)
        {
          printf("Couldn't open PDF file '%s'!\n", filename);
          ROSE_ABORT();
        }

  // Initialize properties stored in PDF file
     PDF_set_info(pdfFile, "Keywords", "Abstract Syntax Tree (AST) EDG");
     PDF_set_info(pdfFile, "Subject", "Display of AST for EDG");
     PDF_set_info(pdfFile, "Title", "AST for Program Code");
     PDF_set_info(pdfFile, "Creator", "ROSE");
     PDF_set_info(pdfFile, "Author", "Daniel Quinlan");

//  unsigned char buf[64], tmp[64];
//  int c, pos;
//  int level1, level2=0, level3=0;
    int font;

    font = PDF_findfont(pdfFile, "Helvetica", "host", 0);

 // Specify the page size (can be much larger than letter size)
 // PDF_begin_page(pdfFile, a4_width, a4_height);
    PDF_begin_page(pdfFile, letter_width, letter_height);

 // Specify a 10 pt font
    int fontSize = 10;

 // setup the font to use in this file
    PDF_setfont(pdfFile, font, fontSize);

 // start at a reasonable position on the page
    int topMargin  = fontSize * 4;
    int leftMargin = fontSize * 4;

 // Translate the orgin of the mapping to the top left corner of the page
 // with a little room for margins
    PDF_translate(pdfFile,0+leftMargin,letter_height-topMargin);

#if 0
    strcpy (buffer[0], "abcdefg");
    PDF_show(pdfFile, buffer[0]);
    for (i=0; i < 20; i++)
         PDF_continue_text(pdfFile, buffer[0]);
#endif

#if 0
     while (sourceFile.getline(buffer[i],100,'\n') && i < MAX_BUFFER_LENGTH)
        {
          PDF_continue_text(pdfFile, buffer[i]);
       // printf ("string = %s \n",buffer[i]);
	  i++;
        }
#endif

  /* private Unicode info entry */
     PDF_set_info(pdfFile, "Revision", "initial version 0.2");

  // initialize the global PDF file
     pdfGlobalFile = pdfFile;

  // Print out the EDG AST
     pdfOutput_complete_il();

  // Need this to avoid a warning from PDFLIB
     PDF_end_page(pdfFile);

  // Now close the PDF file
     PDF_close(pdfFile);
     PDF_delete(pdfFile);

  // finalize the last use of the PDF library
     PDF_shutdown();

     fprintf(stderr, "\nPDFlib EDG AST file '%s' finished!\n", filename);
   }
#endif

#if 0
void
generatePDFofEDG ( const SgProject & project )
   {
#if 0
  // Output the source code file (as represented by the SAGE AST) as a PDF file (with bookmarks)
     int i = 0;
     for (i=0; i < project.numberOfFiles(); i++)
        {
          SgFile & file = project.get_file(i);
          pdfPrintAbstractSyntaxTreeEDG(&file);
        }
#endif
   }
#endif

#if 1
int
ROSE::getLineNumber ( SgLocatedNode* locatedNodePointer )
   {
  // Get the line number from the Sage II statement object
     ROSE_ASSERT (locatedNodePointer != NULL);
     int lineNumber = -1;
  // Sometimes the locatedNode doesn't have a SgFile object 
  // (likely because it is part of a parent statement object)
     if (locatedNodePointer->get_file_info() != NULL)
        {
          ROSE_ASSERT (locatedNodePointer->get_file_info() != NULL);
          ROSE_ASSERT (locatedNodePointer->get_file_info()->get_filename() != NULL);
          Sg_File_Info* fileInfo = locatedNodePointer->get_file_info();
          lineNumber = fileInfo->get_line();
        }

     return lineNumber;
   }
#endif
#if 1
int
ROSE::getColumnNumber ( SgLocatedNode* locatedNodePointer )
   {
  // Get the line number from the Sage II statement object
     ROSE_ASSERT (locatedNodePointer != NULL);
     int columnNumber = -1;
  // Sometimes the locatedNode doesn't have a SgFile object 
  // (likely because it is part of a parent statement object)
     if (locatedNodePointer->get_file_info() != NULL)
        {
          ROSE_ASSERT (locatedNodePointer->get_file_info() != NULL);
          ROSE_ASSERT (locatedNodePointer->get_file_info()->get_filename() != NULL);
          Sg_File_Info* fileInfo = locatedNodePointer->get_file_info();
          columnNumber = fileInfo->get_col();
	}

     return columnNumber;
   }
#endif
#if 1
std::string
ROSE::getFileName ( SgLocatedNode* locatedNodePointer )
   {
  // Get the filename from the Sage II statement object
     ROSE_ASSERT (locatedNodePointer != NULL);
     std::string fileName = "NO NAME FILE";
  // Sometimes the locatedNode doesn't have a SgFile object 
  // (likely because it is part of a parent statement object)
     if (locatedNodePointer->get_file_info() != NULL)
        {
          ROSE_ASSERT (locatedNodePointer->get_file_info() != NULL);
       // printf ("In ROSE::getFileName(): locatedNodePointer->get_file_info() = %p \n",locatedNodePointer->get_file_info());
          Sg_File_Info* fileInfo = locatedNodePointer->get_file_info();
          fileName = fileInfo->get_filenameString();
        }

     return fileName;
   }
#endif
#if 1
bool ROSE:: isPartOfTransformation( SgLocatedNode* locatedNodePointer )
{
  bool result = false;
  Sg_File_Info *fileInfo = locatedNodePointer->get_file_info();
  if (fileInfo != 0)
     result = fileInfo->get_isPartOfTransformation();
  return result;
}
#endif

std::string
ROSE::getFileNameWithoutPath ( SgStatement* statementPointer )
   {
  // Get the filename from the Sage III statement object
     ROSE_ASSERT (statementPointer != NULL);
     ROSE_ASSERT (statementPointer->get_file_info() != NULL);

  // char* fileName = getFileName(statementPointer);
     std::string fileName = statementPointer->get_file_info()->get_filenameString();

     return stripPathFromFileName(fileName);
   }

#if 1
std::string
ROSE::stripPathFromFileName ( const std::string& fileNameWithPath )
   {
     size_t pos = fileNameWithPath.rfind('/');
     if (pos == std::string::npos || pos == fileNameWithPath.size() - 1) {
       return fileNameWithPath;
     } else {
       return fileNameWithPath.substr(pos + 1);
   }
   }
#endif

#if 0
std::string
ROSE::stripFileSuffixFromFileName ( const std::string& fileNameWithSuffix )
   {
  // This function is not sophisticated enough to handle binaries with paths such as:
  //    ROSE/ROSE_CompileTree/svn-LINUX-64bit-4.2.2/tutorial/inputCode_binaryAST_1
#if 1
     size_t lastDotPos = fileNameWithSuffix.rfind('.');
     return fileNameWithSuffix.substr(0, lastDotPos);
#else
  // Handle the case of files where the filename does not have a suffix
     size_t lastSlashPos = fileNameWithSuffix.rfind('/');
     size_t lastDotPos   = fileNameWithSuffix.rfind('.');

     string returnString;
     if (lastDotPos < lastSlashPos)
          returnString = fileNameWithSuffix;
       else
          returnString = fileNameWithSuffix.substr(0, lastDotPos);

     return returnString;
#endif
   } 
#endif

#if 1
// DQ (3/15/2005): New, simpler and better implementation suggested function from Tom, thanks Tom!
string
ROSE::getPathFromFileName ( const string fileName )
   {
     size_t pos = fileName.rfind('/');
     if (pos == std::string::npos) {
       return ".";
     } else {
       return fileName.substr(0, pos);
     }
   } 
#endif

// Later I expect we will move these functions to be SgFile member functions

#if 0
 //! get the source directory (requires an input string currently)
char*
ROSE::getSourceDirectory ( char* fileNameWithPath )
   {
     return getPathFromFileName (fileNameWithPath);
   }
#else
 //! get the source directory (requires an input string currently)
string
ROSE::getSourceDirectory ( string fileNameWithPath )
   {
     return getPathFromFileName (fileNameWithPath);
   }
#endif

#if 0
 //! get the current directory
char*
ROSE::getWorkingDirectory ()
   {
     int i = 0;  // index variable declaration

  // DQ (9/5/2006): Increase the buffer size
  // const int maxPathNameLength = 1024;
     const int maxPathNameLength = 10000;
     char* currentDirectory = new char [maxPathNameLength];
     for (i=0; i < maxPathNameLength; i++)
          currentDirectory[i] = '\0';  // set to NULL string

     char* returnString = getcwd(currentDirectory,maxPathNameLength);
     ROSE_ASSERT (returnString != NULL);

  // The semantics of the getcwd is that these should be the same (see if they are)
  // printf ("In ROSE::getWorkingDirectory: Current directory = %s \n",currentDirectory);
  // printf ("In ROSE::getWorkingDirectory: Current directory = %s \n",returnString);

  // live with the possible memory leak for now
  // delete currentDirectory;
     currentDirectory = NULL;

     return returnString;
   }
#else
 //! get the current directory
string
ROSE::getWorkingDirectory ()
   {
  // DQ (9/5/2006): Increase the buffer size
  // const int maxPathNameLength = 1024;
     const unsigned int maxPathNameLength = 10000;
     char* currentDirectory = new char [maxPathNameLength+1];


  // CH (4/7/2010): In MSVC, the header file "direct.h" contains function 'getcwd'


	 const char* getcwdResult = getcwd(currentDirectory,maxPathNameLength);

     if (!getcwdResult) {
       perror("getcwd: ");
       ROSE_ABORT();
     }
     string returnString = getcwdResult;
     delete currentDirectory;
     currentDirectory = NULL;
     return returnString;
   }
#endif

SgName
ROSE::concatenate ( const SgName & X, const SgName & Y )
   {
     return X + Y;
   }

#if 0
// DQ (9/5/2008): Try to remove this function!
std::string
ROSE::getFileName ( const SgFile* file )
   {
  // Get the filename from the Sage II file object
     ROSE_ASSERT (file != NULL);
  // SgScopeStatement *globalScope = (SgScopeStatement *)(&(file->root()));
     const SgScopeStatement *globalScope = file->get_globalScope();
     ROSE_ASSERT (globalScope != NULL);
     Sg_File_Info* fileInfo = globalScope->get_file_info();
     ROSE_ASSERT (fileInfo != NULL);
     return fileInfo->get_filenameString();
   }
#endif

#if 1
// DQ (9/5/2008): Try to remove this function!
string
ROSE::getFileNameByTraversalBackToFileNode ( const SgNode* astNode )
   {
     string returnString;

     ROSE_ASSERT (astNode != NULL);

  // Make sure this is not a project node (since the SgFile exists below 
  // the project and could not be found by a traversal of the parent list)
     if (isSgProject(astNode) == NULL)
        {
          const SgNode* parent = astNode;
          while ( (parent != NULL) && (isSgFile(parent) == NULL) )
             {
            // printf ("In getFileNameByTraversalBackToFileNode(): parent = %p = %s \n",parent,parent->class_name().c_str());
               parent = parent->get_parent();
             }

	  if (!parent) {
	    const SgLocatedNode* ln = isSgLocatedNode(astNode);
	    ROSE_ASSERT (ln);
	    return ln->get_file_info()->get_filenameString();
	  }
          // ROSE_ASSERT (parent != NULL);
          const SgFile* file = isSgFile(parent);
          ROSE_ASSERT (file != NULL);
          if (file != NULL)
             {
            // returnString = ROSE::getFileName(file);
               returnString = file->getFileName();
             }

       // ROSE_ASSERT (returnString.length() > 0);
          ROSE_ASSERT (returnString.empty() == false);
        }

     return returnString;
   }
#endif

void
ROSE::usage (int status)
   {
     SgFile::usage(status);
  // exit(status);
   }

int 
ROSE::containsString ( const std::string& masterString, const std::string& targetString )
   {
     return masterString.find(targetString) != string::npos;
   }

void
ROSE::filterInputFile ( const string inputFileName, const string outputFileName )
   {
  // This function filters the input file to remove ^M characters and expand tabs etc.
  // Any possible processing of the input file, before being compiled, should be done
  // by this function.

  // This function is implemented in the ROSE/dqDevelopmentDirectory directory.
   }

SgStatement*
ROSE::getNextStatement ( SgStatement *currentStatement )
   {
     ROSE_ASSERT (currentStatement  != NULL);
    //CI (1/3/2007): This sued to be not implemented ,,, here is my try
    //! get next statement will return the next statement in a function or method. if at the end or outside, it WILL return NULL
     
     SgStatement      *nextStatement = NULL;
     SgScopeStatement *scope             = currentStatement->get_scope();
     ROSE_ASSERT (scope != NULL);

  // DQ (9/18/2010): If we try to get the next statement from SgGlobal, then return NULL.
     if (isSgGlobal(currentStatement) != NULL)
          return NULL;

  // Make sure that we didn't get ourselves back from the get_scope() 
  // function (previous bug fixed, but tested here).
     ROSE_ASSERT (scope != currentStatement);

     switch (currentStatement->variantT())
        {
          case V_SgForInitStatement:
          // case V_SgBasicBlock: // Liao 10/20/2010, We should allow users to get a statement immediately AFTER a block.
          case V_SgClassDefinition:
          case V_SgFunctionDefinition:
	  case V_SgStatement:
          case V_SgFunctionParameterList:
					
							 ROSE_ASSERT(0);
							 // not specified
							 break;
          default:
             {
            // We have to handle the cases of a SgStatementPtrList and a 
            // SgDeclarationStatementPtrList separately
							 if (scope->containsOnlyDeclarations() == true)
							 {
								 // Usually a global scope or class declaration scope
								 SgDeclarationStatementPtrList & declarationList = scope->getDeclarationList();
								 Rose_STL_Container<SgDeclarationStatement*>::iterator i;
								 for (i = declarationList.begin();(*i)!=currentStatement;i++) {}
								 // now i == currentStatement
								 i++;
								 if (declarationList.end() == i) nextStatement=NULL;
								 else nextStatement=*i;
							 }
							 else
							 {
								 SgStatementPtrList & statementList = scope->getStatementList();
								 Rose_STL_Container<SgStatement*>::iterator i;
                                                                 // Liao, 11/18/2009, Handle the rare case that current statement is not found
                                                                 // in its scope's statement list
								 //for (i = statementList.begin();(*i)!=currentStatement;i++) 
								 for (i = statementList.begin();(*i)!=currentStatement && i!= statementList.end();i++) 
                                                                 {
                                                                 //  SgStatement* cur_stmt = *i;
                                                                 //  cout<<"Skipping current statement: "<<cur_stmt->class_name()<<endl;
                                                                 //  cout<<cur_stmt->get_file_info()->displayString()<<endl;
                                                                 }
								 // currentStatement is not found in the list
                                                                 if (i ==  statementList.end()) 
                                                                 {
                                                                   cerr<<"fatal error: ROSE::getNextStatement(): current statement is not found within its scope's statement list"<<endl;
                                                                   cerr<<"current statement is "<<currentStatement->class_name()<<endl;
                                                                   cerr<<currentStatement->get_file_info()->displayString()<<endl;
                                                                   cerr<<"Its scope is "<<scope->class_name()<<endl;
                                                                   cerr<<scope->get_file_info()->displayString()<<endl;
                                                                   ROSE_ASSERT (false);
                                                                  }
                                                                 //  now i == currentStatement
                                                                 ROSE_ASSERT (*i == currentStatement);
								 i++;
								 if (statementList.end() == i) nextStatement=NULL;
								 else nextStatement=*i;
							 }

            // If the target statement was the last statement in a scope then 
               if (nextStatement == NULL)
							 {
								 // Someone might think of a better answer than NULL
							 }
             }
						 break;
        }
     //This assertion does not make sense. 
     //Since a trailing statement within a scope can have null next statement, 
     //and  the statement can be not global scope statement, Liao 3/12/2009
     //ROSE_ASSERT (isSgGlobal(currentStatement) != NULL || nextStatement != NULL);

     return nextStatement;
   }
	 
SgStatement*
ROSE::getPreviousStatement ( SgStatement *targetStatement )
   {
     ROSE_ASSERT (targetStatement  != NULL);

     SgStatement      *previousStatement = NULL;
     SgScopeStatement *scope             = targetStatement->get_scope();
     ROSE_ASSERT (scope != NULL);

  // DQ (9/18/2010): If we try to get the previous statement from SgGlobal, then return NULL.
     if (isSgGlobal(targetStatement) != NULL)
          return NULL;

  // Make sure that we didn't get ourselves back from the get_scope() 
  // function (previous bug fixed, but tested here).
     if (scope == targetStatement)
        {
          printf ("Error: targetStatement = %p = %s \n",targetStatement,targetStatement->class_name().c_str());
        }
     ROSE_ASSERT (scope != targetStatement);

#if 0
     printf ("@@@@@ In ROSE::getPreviousStatement(): targetStatement = %s \n",targetStatement->sage_class_name());
     printf ("@@@@@ In ROSE::getPreviousStatement(): targetStatement->unparseToString() = %s \n",targetStatement->unparseToString().c_str());
     printf ("@@@@@ In ROSE::getPreviousStatement(): scope = %s \n",scope->sage_class_name());
     printf ("@@@@@ In ROSE::getPreviousStatement(): scope->unparseToString() = %s \n",scope->unparseToString().c_str());
#endif

     switch (targetStatement->variantT())
        {
          case V_SgFunctionParameterList:
             {
            // We define the previous statement in this case to be the function declaration
               previousStatement = isSgStatement(targetStatement->get_parent());
               ROSE_ASSERT (isSgFunctionDeclaration(previousStatement) != NULL);
               break;
             }

          case V_SgForInitStatement:
             {
               previousStatement = isSgStatement(targetStatement->get_parent());
               ROSE_ASSERT (isSgForStatement(previousStatement) != NULL);
               break;
             }

          case V_SgBasicBlock:
             {
               previousStatement = isSgStatement(targetStatement->get_parent());
               ROSE_ASSERT (previousStatement != NULL);
               break;
             }

          case V_SgClassDefinition:
          case V_SgFunctionDefinition:
             {
            // In the case of a definition we define the previous statement
            // to be to one appearing before the associated declaration.
               previousStatement = isSgStatement(targetStatement->get_parent());
               ROSE_ASSERT (previousStatement != NULL);
               break;
             }
          default:
             {
            // We have to handle the cases of a SgStatementPtrList and a 
            // SgDeclarationStatementPtrList separately
               if (scope->containsOnlyDeclarations() == true)
                  {
                 // Usually a global scope or class declaration scope
                    SgDeclarationStatementPtrList & declarationList = scope->getDeclarationList();
                    Rose_STL_Container<SgDeclarationStatement*>::iterator i = declarationList.begin();
                    Rose_STL_Container<SgDeclarationStatement*>::iterator previousStatementIterator =  declarationList.end();
                    while ( ( i != declarationList.end() ) && ( (*i) != targetStatement ) )
                       {
                         previousStatementIterator = i++;
                       }

                    if ( previousStatementIterator != declarationList.end() )
                       {
                         previousStatement = *previousStatementIterator;
                       }
                  }
                  // Liao 5/10/2010, special case when a true/false body of a if statement is not a basic block
                  // since getStatementList() is not defined for a if statement. 
                  // We define the previous statement of the true/false body to be the if statement
                  // This is consistent with the later handling that when a statement is the first in a parent, 
                  // treat the parent as the previous statement
                 else if (isSgIfStmt(scope))
                  {
                    previousStatement = isSgStatement(targetStatement->get_parent());
                    ROSE_ASSERT (isSgIfStmt(previousStatement) != NULL);
                  }
                 else
                  {
                    SgStatementPtrList & statementList = scope->getStatementList();
                    Rose_STL_Container<SgStatement*>::iterator i = statementList.begin();
                    Rose_STL_Container<SgStatement*>::iterator previousStatementIterator =  statementList.end();
                    while ( ( i != statementList.end() ) && ( (*i) != targetStatement ) )
                       {
                         previousStatementIterator = i++;
                       }

                    if ( previousStatementIterator != statementList.end() )
                       {
                         previousStatement = *previousStatementIterator;
                       }
                  }

            // If the target statement was the first statement in a scope then 
               if (previousStatement == NULL)
                  {
                 // Then set the previous statement to be the scope containing the current statement
                 // printf ("previousStatement == NULL: previous scope = %s \n",scope->unparseToString().c_str());
                    switch (scope->variantT())
                       {
                         case V_SgBasicBlock:
                            {
                              previousStatement = getPreviousStatement(scope);
                              break;
                            }

                         default:
                            {
                              previousStatement = scope;
                              break;
                            }
                       }
                  }
               break;
             }
        }

#if 0
     printf ("@@@@@ previousStatement = %p \n",previousStatement);
     if (previousStatement != NULL)
        printf ("@@@@@ previousStatement->unparseToString() = %s \n",previousStatement->unparseToString().c_str());
#endif

     ROSE_ASSERT (isSgGlobal(targetStatement) != NULL || previousStatement != NULL);

     return previousStatement;
   }







