#!/bin/csh -f
#
# csh script to checkout Dakota and packages from other repositories
#
# --- external arguments/variables used ---
# $1          target directory for checkout or export, e.g.
#             $TEST_ROOT/test_linux/packages
# $2          packages (for packages only), trunk, branches/Version_4_1_Branch
# $3          optional (Dakota checkout only): svn export (default) or co
# $4          optional (Dakota checkout only): options to pass to svn
# $TEST_SQA   script directory (passed to top-level scripts)
# $DEV_SVNROOT path to development subversion root
# -----------------------------------------

# -------------------------------
# required command-line arguments
# -------------------------------
if ( $#argv < 2 ) then
  echo "Usage: checkout_script.csh CHECKOUT_DIR DAKOTA_VER [SVN_ACTION] [SVN_OPTIONS]"
  echo "       CHECKOUT_DIR base directory to receive checkouts"
  echo "       DAKOTA_VER   packages, trunk, branches/Version_4_1_Branch, etc."
  echo "       SVN_ACTION   optional: export (default) or checkout"
  echo "       SVN_OPTIONS  optional: specific switches for Dakota svn command"
  echo "                              such as revision number"
  echo "Examples:"
  echo "  checkout_script.csh /usr/sharelan/dakota/test_linux packages"
  echo "  checkout_script.csh /usr/sharelan/dakota/test_linux trunk export '-r 43'"
  exit  
endif

echo "Running $0 on `uname -n` at `date`"

set CHECKOUT_DIR = $1
set DAKOTA_VER = $2

# --------------------------------
# command-line options
# (used only for DAKOTA checkouts)
# --------------------------------

# in nightly testing, probably only need export, offer checkout for developers
if ( $#argv > 2 ) then
  set SVN_ACTION = $3 # checkout (was base), export (was votd)
else 
  set SVN_ACTION = export
endif

if ( $#argv > 3 ) then
  set SVN_OPTIONS = "$4"
else
  set SVN_OPTIONS = ''
endif


# if checkout directory does not exist, create
if ( -e $CHECKOUT_DIR ) then
  echo "Checkout directory is $CHECKOUT_DIR"
else
  mkdir -p $CHECKOUT_DIR
  echo "Creating checkout directory $CHECKOUT_DIR"
endif

cd $CHECKOUT_DIR
# cleanup any existing directories for packages or Dakota
if ( $DAKOTA_VER == 'packages' ) then
  foreach p ( `cat $TEST_SQA/package-list-test` )
    set pdir = `echo $p | awk -F^ '{print $NF}'`
    # if a previous package directory exists, remove it
    if ( -e $pdir ) then 
      echo "Removing package directory $pdir"
      \rm -rf $pdir
    endif
  end
else 
  # if Dakota exists, remove it
  if ( -e Dakota ) then
    echo "Removing directory Dakota"
    \rm -rf Dakota
  endif
endif

# proceed with clean checkouts for packages or Dakota
if ( $DAKOTA_VER == 'packages' ) then 
  echo "Bypass Dakota checkout (package tests only)"
  # get the correct version of each package using constructed checkout command
  foreach p ( `cat $TEST_SQA/package-list-test` )
    set pv = `echo $p | awk -F^ '{print $NF}'`
    set pc = "`echo $p | tr '^' ' '`"
    echo "Getting package $pv"
    $pc
  end
else
  # export or checkout requested Dakota version
  echo "Performing $SVN_ACTION of Dakota/$DAKOTA_VER $SVN_OPTIONS to Dakota"
  svn --non-interactive $SVN_ACTION $DEV_SVNROOT/Dakota/$DAKOTA_VER $SVN_OPTIONS Dakota > $SVN_ACTION.log
  # if export, create Dakota/src/.build-info with DAKOTA repository version
  if ( $SVN_ACTION == 'export' ) then
    echo 'DAKOTA BUILD INFORMATION' >! Dakota/src/.build-info
    echo '' >> Dakota/src/.build-info
    echo 'SUBVERSION REPOSITORY REVISIONS:' >> Dakota/src/.build-info
    set dakver=`awk -F. 'END{n=split($1,f," ");print f[n]}' export.log`
    echo "Dakota^${dakver}" >> Dakota/src/.build-info
  endif
  # perform autoreconf (may want to remove from this script)
  cd Dakota
  autoreconf --install --verbose
  cd $CHECKOUT_DIR
endif

echo "Checkout process complete"
