/*  _______________________________________________________________________

    DAKOTA: Design Analysis Kit for Optimization and Terascale Applications
    Copyright (c) 2001, Sandia National Laboratories.
    This software is distributed under the GNU General Public License.
    For more information, see the README file in the top Dakota directory.
    _______________________________________________________________________ */

//- Class:	 NonDSparseGrid
//- Description: Wrapper class for C++ code from packages/quadrature/sparse_grid
//- Owner:       Mike Eldred
//- Revised by:  
//- Version:

#ifndef NOND_SPARSE_GRID_H
#define NOND_SPARSE_GRID_H

#include "data_types.h"
#include "NonDIntegration.H"

namespace Dakota {

typedef void ( *FPType ) ( int order, double alpha, double beta, double* data );


/// Derived nondeterministic class that generates N-dimensional
/// Smolyak sparse grids for numerical evaluation of expectation
/// integrals over independent standard random variables.

/** This class is used by NonDPolynomialChaos and
    NonDStochCollocation, but could also be used for general numerical
    integration of moments.  It employs 1-D Clenshaw-Curtis and Gaussian
    quadrature rules within Smolyak sparse grids. */

class NonDSparseGrid: public NonDIntegration
{
public:

  //
  //- Heading: Constructors and destructor
  //

  // alternate constructor for instantiations "on the fly"
  NonDSparseGrid(Model& model, const UShortArray& levels);

  //
  //- Heading: Member functions
  //

  /// set rules based on u_types
  void initialize_rules(const Pecos::ShortArray& u_types, IntArray& rules,
			Array<FPType>& compute_1d_pts,
			Array<FPType>& compute_1d_wts, RealArray& poly_alphas,
			RealArray& poly_betas);

  /// return sparseGridLevel
  const UShortArray& sparse_grid_level() const;
  /// return integrationRules
  const IntArray& integration_rules() const;
  /// return duplicateTol
  const Real& duplicate_tolerance() const;

  // return sparseGridIndexMap
  //const IntArraySizetMap& sparse_grid_index_map() const;
  /// return uniqueIndexMapping
  const IntArray& unique_index_mapping() const;

  /// return ProbabilityTransformation::ranVarTypesU
  const Pecos::ShortArray& integrated_variable_types() const;

  /// converts sparse grid level to integration order for closed rules
  /// with exponential growth
  void level_to_order_closed_exponential(const unsigned short& level,
					 unsigned short& order) const;
  /// converts sparse grid level to integration order for open rules
  /// with exponential growth
  void level_to_order_open_exponential(const unsigned short& level,
				       unsigned short& order) const;
  /// converts sparse grid level to integration order for open rules
  /// with linear growth
  void level_to_order_open_linear(const unsigned short& level,
				  unsigned short& order) const;
  /// converts an array of sparse grid levels to an array of
  /// integration orders based on integrationRules
  void level_to_order(const UShortArray& levels, UShortArray& orders) const;

protected:

  //
  //- Heading: Constructors and destructor
  //

  NonDSparseGrid(Model& model); ///< constructor
  ~NonDSparseGrid();            ///< destructor

  //
  //- Heading: Virtual function redefinitions
  //

  void get_parameter_sets(const Model& model);

  void check_input();

  void sampling_reset(int min_samples, int rec_samples, bool all_data_flag,
		      bool stats_flag);

private:

  //
  //- Heading: Convenience functions
  //

  //
  //- Heading: Data
  //

  /// the user specification for the sparse grid levels
  UShortArray sparseGridLevelSpec;
  /// the actual sparse grid levels, as modified by external requirements
  /// communicated through sampling_reset()
  UShortArray sparseGridLevel;

  /// integer codes for sparse_grid_mixed routine integration rule options
  IntArray integrationRules;
  /// array of alpha polynomial parameters for input to sparse_grid_mixed
  /// routines (corresponds to set of variables defined by integrationRules)
  RealArray alphaPolyParams;
  /// array of beta polynomial parameters for input to sparse_grid_mixed
  /// routines (corresponds to set of variables defined by integrationRules)
  RealArray betaPolyParams;
  /// duplication tolerance used in sparse_grid_mixed routines
  Real duplicateTol;

  // maps indices and bases from sparse_grid_mixed_index() to
  // collocation point index
  //IntArraySizetMap sparseGridIndexMap;

  /// output from sparse_grid_mixed_unique_index()
  IntArray uniqueIndexMapping;

  Array<FPType> compute1DPoints;
  Array<FPType> compute1DWeights;
};


inline const UShortArray& NonDSparseGrid::sparse_grid_level() const
{ return sparseGridLevel; }


inline const IntArray& NonDSparseGrid::integration_rules() const
{ return integrationRules; }


inline const Real& NonDSparseGrid::duplicate_tolerance() const
{ return duplicateTol; }


//inline const IntArraySizetMap& NonDSparseGrid::sparse_grid_index_map() const
//{ return sparseGridIndexMap; }


inline const IntArray& NonDSparseGrid::unique_index_mapping() const
{ return uniqueIndexMapping; }


inline const Pecos::ShortArray& NonDSparseGrid::
integrated_variable_types() const
{ return natafTransform.u_types(); }


// NOTE: these mappings are intended for 0-based level indices,
// i.e. the j index set

/** Adapted from webbur::level_to_order_default() for DAKOTA data types. */
inline void NonDSparseGrid::
level_to_order_closed_exponential(const unsigned short& level,
				  unsigned short& order) const
{ order = (level) ? (unsigned short)pow(2., (int)level) + 1 : 1; }


/** Adapted from webbur::level_to_order_default() for DAKOTA data types. */
inline void NonDSparseGrid::
level_to_order_open_exponential(const unsigned short& level,
				unsigned short& order) const
{ order = (unsigned short)pow(2., (int)level + 1) - 1; }


/** Adapted from webbur::level_to_order_default() for DAKOTA data types. */
inline void NonDSparseGrid::
level_to_order_open_linear(const unsigned short& level,
			   unsigned short& order) const
{ order = 2 * level + 1; }


inline void NonDSparseGrid::
level_to_order(const UShortArray& levels, UShortArray& orders) const
{
  size_t i, num_levels = levels.length();
  if (orders.length() != num_levels)
    orders.reshape(num_levels);
  for (i=0; i<num_levels; i++)
    switch (integrationRules[i]) {
    case 1:  // Clenshaw-Curtis
      level_to_order_closed_exponential(levels[i], orders[i]); break;
    case 2: case 3: // Fejer Type 2 and Gauss Patterson
      level_to_order_open_exponential(levels[i], orders[i]);   break;
    default: // all Gauss rules
      level_to_order_open_linear(levels[i], orders[i]);        break;
    }
}

} // namespace Dakota

#endif
