/*  _______________________________________________________________________

    DAKOTA: Design Analysis Kit for Optimization and Terascale Applications
    Copyright (c) 2006, Sandia National Laboratories.
    This software is distributed under the GNU General Public License.
    For more information, see the README file in the top Dakota directory.
    _______________________________________________________________________ */

//- Class:	 NonDIntervalEst
//- Description: Class for interval bound estimation for epistemic UQ
//- Owner:       Laura Swiler
//- Checked by:
//- Version:

#include "system_defs.h"
#include "data_io.h"
#include "NonDIntervalEst.H"
#include "NonDLHSSampling.H"
#include "DakotaModel.H"
#include "DakotaIterator.H"
#include "RecastModel.H"
#include "DataFitSurrModel.H"
#include "DakotaApproximation.H"
#include "ProblemDescDB.H"
#ifdef DAKOTA_NCSU
#include "NCSUOptimizer.H"
#endif
#include "pecos_stat_util.hpp"

//#define DEBUG

namespace Dakota {

// initialization of statics
NonDIntervalEst* NonDIntervalEst::nondIntervalEstInstance(NULL);


NonDIntervalEst::NonDIntervalEst(Model& model): NonD(model)  
{
  // initialize finalStatistics using non-default definition.
  initialize_final_statistics();

  // Use a hardwired minimal initial samples
  int samples  = (numContinuousVars+1)*(numContinuousVars+2)/2,
      lhs_seed = probDescDB.get_int("method.random_seed");
  String approx_type = "global_gaussian", corr_type, sample_reuse = "none";
  UShortArray approx_order; 
  short corr_order = -1;

  // instantiate the Gaussian Process DataFit recursions

  // The following uses on the fly derived ctor:
  daceIterator.assign_rep(new
    NonDLHSSampling(iteratedModel, samples, lhs_seed, ACTIVE_UNIFORM), false);

  // Construct fHatModel using a GP approximation over the active/uncertain
  // vars (same view as iteratedModel: not the typical All view for DACE).
 
  // Construct f-hat using a GP approximation for each response function over
  // the active/design vars (same view as iteratedModel: not the typical All
  // view for DACE).
  const Variables& curr_vars = iteratedModel.current_variables();
  fHatModel.assign_rep(new DataFitSurrModel(daceIterator,
    iteratedModel, curr_vars.view(), curr_vars.variables_components(),
    iteratedModel.current_response().active_set(), approx_type,
    approx_order, corr_type, corr_order, sample_reuse), false);

  // eifModel.init_communicators() recursion is currently sufficient for
  // fHatModel.  An additional fHatModel.init_communicators() call would be
  // motivated by special parallel usage of fHatModel below that is not
  // otherwise covered by the recursion.
  //fHatMaxConcurrency = maxConcurrency; // local derivative concurrency
  //fHatModel.init_communicators(fHatMaxConcurrency);

  // Following this ctor, Strategy::init_iterator() initializes the parallel
  // configuration for NonDIntervalEst + iteratedModel using NonDIntervalEst's
  // maxConcurrency.  During fHatModel construction above, DataFitSurrModel::
  // derived_init_communicators() initializes the parallel configuration for
  // daceIterator + iteratedModel using daceIterator's maxConcurrency.  The
  // only iteratedModel concurrency currently exercised is that used by
  // daceIterator within the initial GP construction, but the NonDIntervalEst
  // maxConcurrency must still be set so as to avoid parallel configuration
  // errors resulting from avail_procs > max_concurrency within Strategy::
  // init_iterator().  A max of the local derivative concurrency and the DACE
  // concurrency is used for this purpose.
  maxConcurrency = max(maxConcurrency, daceIterator.maximum_concurrency());

  // Configure a RecastModel with one objective and no constraints using the
  // alternate minimalist constructor: the recast fn pointers are reset for
  // each level within the run fn.
  eifModel.assign_rep(new RecastModel(fHatModel, 1, 0, 0), false);

  // instantiate the optimizer used to improve the GP
  int max_iter = 1000, max_eval = 10000;
#ifdef DAKOTA_NCSU  
  gpOptimizer.assign_rep(new
    NCSUOptimizer(eifModel, max_iter, max_eval), false);
  eifModel.init_communicators(gpOptimizer.maximum_concurrency());
#else
  Cerr << "NCSU DIRECT Optimizer is not available to use to find the" 
       << " interval bounds from the GP model." << endl;
  abort_handler(-1);
#endif // DAKOTA_NCSU
}


NonDIntervalEst::~NonDIntervalEst()
{  
  // deallocate communicators for DIRECT on mppModel
  eifModel.free_communicators(gpOptimizer.maximum_concurrency());

  // eifModel.free_communicators() recursion is currently sufficient for
  // fHatModel.  An additional fHatModel.free_communicators() call would be
  // motivated by special parallel usage of fHatModel below that is not
  // otherwise covered by the recursion.
  //fHatModel.free_communicators(fHatMaxConcurrency);
}


void NonDIntervalEst::quantify_uncertainty()
{
  // set the object instance pointer for use within static member functions
  NonDIntervalEst*  prev_instance = nondIntervalEstInstance;
  nondIntervalEstInstance = this;

  // Optimize the GP
  //if (globalFlag)
    nongradient_global_estimation();
  //else
  //  gradient_local_estimation();

  // restore in case of recursion
  nondIntervalEstInstance = prev_instance;
}


void NonDIntervalEst::nongradient_global_estimation()
{
  eifModel.update_from_subordinate_model();
  // Build initial GP once for all response functions
  fHatModel.build_approximation();

  size_t stat_cntr = 0;
  Sizet2DArray vars_map, primary_resp_map(1), secondary_resp_map;
  primary_resp_map[0].reshape(1);
  BoolDequeArray nonlinear_resp_map(1);

  convergenceTol = 1.e-12;
  maxIterations  = 25*numContinuousVars;

  for (respFnCntr=0; respFnCntr<numFunctions; ++respFnCntr) {

    primary_resp_map[0][0] = respFnCntr;
    nonlinear_resp_map[0] = BoolDeque(numFunctions, false);
    nonlinear_resp_map[0][respFnCntr] = true;

    // Iterate until EGO converges
    size_t convergence_cntr = 0, sb_iter_num = 0;
    bool approxConverged = false;
    while (!approxConverged) {

      ++sb_iter_num;

      // initialize EIF recast model
      RecastModel* eif_model_rep = (RecastModel*)eifModel.model_rep();
      eif_model_rep->initialize(vars_map, false, NULL, NULL, primary_resp_map,
				secondary_resp_map, nonlinear_resp_map,
				EIF_objective_min, NULL);

      // determine approxFnStar from minimum among sample data
      get_best_sample(false, true);

      // Execute GLOBAL search and retrieve results
      Cout << "\n>>>>> Initiating global minimization\n";
      gpOptimizer.run_iterator(); // quiet mode
      const Variables& vars_star = gpOptimizer.variables_results();
      const RealDenseVector& c_vars_star = vars_star.continuous_variables();
      const Response& resp_star_approx = gpOptimizer.response_results();
      Real eif_star = -resp_star_approx.function_values()[0];
      Cout << "\nResults of EGO minimization:\nFinal point =\n";
      write_data(Cout, c_vars_star);
      Cout << "Expected Improvement    =\n                     "
	   << setw(write_precision+7) << eif_star << "\n";

      // Check for convergence based on max EIF
      if ( eif_star < convergenceTol )
	++convergence_cntr;
      // If DIRECT failed to find a point with EIF>0, it returns the
      //   center point as the optimal solution. EGO may have converged,
      //   but DIRECT may have just failed to find a point with a good
      //   EIF value. Adding this midpoint can alter the GPs enough to
      //   to allow DIRECT to find something useful, so we force 
      //   max(EIF)<tol twice to make sure. Note that we cannot make
      //   this check more than 2 because it would cause EGO to add
      //   the center point more than once, which will damage the GPs.
      //   Unfortunately, when it happens the second time, it may still
      //   be that DIRECT failed and not that EGO converged.
      if ( convergence_cntr == 2 || sb_iter_num >= maxIterations )
	approxConverged = true;
      else {
	// Evaluate response_star_truth
	//fHatModel.component_parallel_mode(TRUTH_MODEL);
	iteratedModel.continuous_variables(c_vars_star);
	ActiveSet set = iteratedModel.current_response().active_set();
	set.request_values(0); set.request_value(respFnCntr, 1);
	iteratedModel.compute_response(set);
	const Response& resp_star_truth = iteratedModel.current_response();
    
	// Update the GP approximation
	fHatModel.append_approximation(vars_star, resp_star_truth, true);
      }

    } // end approx convergence while loop

    get_best_sample(false, false); // pull truthFnStar from sample data
    finalStatistics.function_value(truthFnStar, stat_cntr++);  

    // Iterate until EGO converges
    convergence_cntr = 0, sb_iter_num = 0;
    approxConverged = false;
    while (!approxConverged) {

      ++sb_iter_num;

      // initialize EIF recast model
      RecastModel* eif_model_rep2 = (RecastModel*)eifModel.model_rep();
      eif_model_rep2->initialize(vars_map, false, NULL, NULL, primary_resp_map,
				 secondary_resp_map, nonlinear_resp_map,
				 EIF_objective_max, NULL);

      // determine approxFnStar from maximum among sample data
      get_best_sample(true, true);

      // Execute GLOBAL search and retrieve results
      Cout << "\n>>>>> Initiating global maximization\n";
      gpOptimizer.run_iterator(); // quiet mode
      const Variables& vars_star = gpOptimizer.variables_results();
      const RealDenseVector& c_vars_star = vars_star.continuous_variables();
      const Response& resp_star_approx = gpOptimizer.response_results();
      Real eif_star = -resp_star_approx.function_values()[0];
      Cout << "\nResults of EGO maximization:\nFinal point =\n";
      write_data(Cout, c_vars_star);
      Cout << "Expected Improvement    =\n                     "
	   << setw(write_precision+7) << eif_star << "\n";

      // Check for convergence based on max EIF
      if ( eif_star < convergenceTol )
	++convergence_cntr;
      // If DIRECT failed to find a point with EIF>0, it returns the
      //   center point as the optimal solution. EGO may have converged,
      //   but DIRECT may have just failed to find a point with a good
      //   EIF value. Adding this midpoint can alter the GPs enough to
      //   to allow DIRECT to find something useful, so we force 
      //   max(EIF)<tol twice to make sure. Note that we cannot make
      //   this check more than 2 because it would cause EGO to add
      //   the center point more than once, which will damage the GPs.
      //   Unfortunately, when it happens the second time, it may still
      //   be that DIRECT failed and not that EGO converged.
      if ( convergence_cntr == 2 || sb_iter_num >= maxIterations )
	approxConverged = true;
      else {
	// Evaluate response_star_truth
	//fHatModel.component_parallel_mode(TRUTH_MODEL);
	iteratedModel.continuous_variables(c_vars_star);
	ActiveSet set = iteratedModel.current_response().active_set();
	set.request_values(0); set.request_value(respFnCntr, 1);
	iteratedModel.compute_response(set);
	const Response& resp_star_truth = iteratedModel.current_response();
    
	// Update the GP approximation
	fHatModel.append_approximation(vars_star, resp_star_truth, true);
      }

    } // end approx convergence while loop

    get_best_sample(true, false); // pull truthFnStar from sample data
    finalStatistics.function_value(truthFnStar, stat_cntr++);  
  }
}


void NonDIntervalEst::
EIF_objective_min(const Variables& sub_model_vars, const Variables& recast_vars,
		  const Response& sub_model_response, Response& recast_response)
{
  // Means are passed in, but must retrieve variance from the GP
  const RealDenseVector& means = sub_model_response.function_values();
  const RealVector& variances
    = nondIntervalEstInstance->fHatModel.approximation_variances(
      recast_vars.continuous_variables());

  const ShortArray& recast_asv = recast_response.active_set_request_vector();
  if (recast_asv[0] & 1) { // return -EI since we are maximizing
    const Real& mean = means[nondIntervalEstInstance->respFnCntr];
    Real stdv = sqrt(variances[nondIntervalEstInstance->respFnCntr]);
    const Real& approx_fn_star = nondIntervalEstInstance->approxFnStar;
    // Calculate the expected improvement
    Real  snv = (approx_fn_star - mean)/stdv, // standard normal variate
      Phi_snv = Pecos::Phi(snv), phi_snv = Pecos::phi(snv),
      ei      = (approx_fn_star - mean)*Phi_snv + stdv*phi_snv;
    recast_response.function_value(-ei, 0);
  }
}


void NonDIntervalEst::
EIF_objective_max(const Variables& sub_model_vars, const Variables& recast_vars,
		  const Response& sub_model_response, Response& recast_response)
{
  // Means are passed in, but must retrieve variance from the GP
  const RealDenseVector& means = sub_model_response.function_values();
  const RealVector& variances
    = nondIntervalEstInstance->fHatModel.approximation_variances(
      recast_vars.continuous_variables());

  const ShortArray& recast_asv = recast_response.active_set_request_vector();
  if (recast_asv[0] & 1) { // return -EI to the minimizer
    Real mean = -means[nondIntervalEstInstance->respFnCntr],
         stdv = sqrt(variances[nondIntervalEstInstance->respFnCntr]);
    const Real& approx_fn_star = nondIntervalEstInstance->approxFnStar;
    // Calculate the expected improvement
    Real  snv = (-approx_fn_star-mean)/stdv, // standard normal variate   
      Phi_snv = Pecos::Phi(snv), phi_snv = Pecos::phi(snv),
      ei      = (-approx_fn_star-mean)*Phi_snv + stdv*phi_snv;
    // minimize -EIF -> maximize EIF
    recast_response.function_value(-ei, 0);
  }
}


void NonDIntervalEst::get_best_sample(bool find_max, bool eval_approx)
{
  // Pull the samples and responses from data used to build latest GP
  // to determine fnStar for use in the expected improvement function

  const List<SurrogateDataPoint>& gp_data
    = fHatModel.approximation_data(respFnCntr);
  List<SurrogateDataPoint>::const_iterator cit, cit_star;

  truthFnStar = (find_max) ? -DBL_MAX : DBL_MAX;
  for (cit = gp_data.begin(); cit != gp_data.end(); ++cit) {
    const Real& truth_fn = cit->response_function();
    if ( (  find_max && truth_fn > truthFnStar ) ||
	 ( !find_max && truth_fn < truthFnStar ) ) {
      cit_star    = cit;
      truthFnStar = truth_fn;
    }
  }

  if (eval_approx) {
    fHatModel.continuous_variables(cit_star->continuous_variables());
    ActiveSet set = fHatModel.current_response().active_set();
    set.request_values(0); set.request_value(respFnCntr, 1);
    fHatModel.compute_response(set);
    approxFnStar = fHatModel.current_response().function_values()[respFnCntr];
  }
}


void NonDIntervalEst::initialize_final_statistics()
{
  size_t num_final_stats = 2*numFunctions;
  ActiveSet stats_set(num_final_stats, numUncertainVars);
  finalStatistics = Response(stats_set);

  // Assign meaningful fn labels to final stats (appear in NestedModel output)
  size_t i, j, num_levels, cntr = 0;
  StringArray stats_labels(num_final_stats);
  const StringArray& fn_labels = iteratedModel.response_labels();
  for (i=0; i<numFunctions; ++i) {
    stats_labels[cntr++] = fn_labels[i] + String("_min");
    stats_labels[cntr++] = fn_labels[i] + String("_max");
  }
  finalStatistics.function_labels(stats_labels);
}


void NonDIntervalEst::print_results(ostream& s)
{
  const StringArray& fn_labels = iteratedModel.response_labels();
  const RealDenseVector& final_stats = finalStatistics.function_values();

  s << "------------------------------------------------------------------\n";

  // output CBF/CCBF and CPF/CCPF response/probabilities pairs
  s.setf(ios::scientific);
  s << setprecision(write_precision)
    << "Min and Max estimated values for each response function:\n";
  size_t i, stat_cntr = 0;
  for (i=0; i<numFunctions; ++i) {
    s << fn_labels[i] << ":  Min = " << final_stats[stat_cntr]; ++stat_cntr;
    s <<  "  Max = " << final_stats[stat_cntr] << '\n';         ++stat_cntr;
  }

  s << "-----------------------------------------------------------------"
    << endl;
}

} // namespace Dakota
