/*  _______________________________________________________________________

    DAKOTA: Design Analysis Kit for Optimization and Terascale Applications
    Copyright (c) 2001, Sandia National Laboratories.
    This software is distributed under the GNU General Public License.
    For more information, see the README file in the top Dakota directory.
    _______________________________________________________________________ */

//- Class:	 NonDIntegration
//- Description: Base class for multidimensional integration using
//-              tensor-product quadrature or Smolyak sparse grids.
//- Owner:       Mike Eldred
//- Revised by:  
//- Version:

#ifndef NOND_INTEGRATION_H
#define NOND_INTEGRATION_H

#include "data_types.h"
#include "DakotaNonD.H"

namespace Dakota {


/// Derived nondeterministic class that generates N-dimensional
/// numerical integration points for evaluation of expectation integrals

/** This class provides a base class for shared code among
    NonDQuadrature and NonDSparseGrid. */

class NonDIntegration: public NonD
{
public:

  //
  //- Heading: Member functions
  //

  /// return weightProducts
  const RealVector& weight_products() const;

  /// return gaussPts1D
  const RealVector2DArray& gauss_points_array()  const;
  /// return gaussWts1D
  const RealVector2DArray& gauss_weights_array() const;

protected:

  //
  //- Heading: Constructors and destructor
  //

  NonDIntegration(Model& model); ///< constructor

  /// alternate constructor for instantiations "on the fly"
  NonDIntegration(NoDBBaseConstructor, Model& model);

  ~NonDIntegration();            ///< destructor

  //
  //- Heading: Virtual functions
  //

  /// verify self-consistency of data
  virtual void check_input() = 0;

  //
  //- Heading: Virtual function redefinitions
  //

  void quantify_uncertainty();

  //
  //- Heading: Data
  //

  /// set of products of (Gauss) weights for each point in the
  /// n-dimensional stencil
  RealVector weightProducts;

  /// numContinuousVars x num_levels_per_var sets of 1D Gauss points
  RealVector2DArray gaussPts1D;
  /// numContinuousVars x num_levels_per_var sets of 1D Gauss weights
  RealVector2DArray gaussWts1D;

private:

  //
  //- Heading: Data
  //

  /// counter for number of integration executions for this object
  size_t numIntegrations;
};


inline const RealVector& NonDIntegration::weight_products() const
{ return weightProducts; }


inline const RealVector2DArray& NonDIntegration::gauss_points_array() const
{ return gaussPts1D; }


inline const RealVector2DArray& NonDIntegration::gauss_weights_array() const
{ return gaussWts1D; }

} // namespace Dakota

#endif
