/*  _______________________________________________________________________

    DAKOTA: Design Analysis Kit for Optimization and Terascale Applications
    Copyright (c) 2006, Sandia National Laboratories.
    This software is distributed under the GNU General Public License.
    For more information, see the README file in the top Dakota directory.
    _______________________________________________________________________ */

//- Class:       DakotaVector<T>
//- Description: An vector class template that provides additional functionality
//-              that is specific to Dakota's needs.  Inheritance is used to
//-              build upon the capabilities provided by a third party array 
//-              class template.
//- Owner:       Mario Alleva
//- Modified by: 
//- Checked by:
//- Version: $Id

#ifndef DAKOTA_VECTOR_H
#define DAKOTA_VECTOR_H

#include "global_defs.h"
#include "DakotaBaseVector.H"
#include "DakotaString.H"
#include "DakotaBinStream.H"
#include "MPIPackBuffer.H"


namespace Dakota {

/// Template class for the Dakota numerical vector

/** The Dakota::Vector class is the numeric vector class. It inherits
    from the common vector class Dakota::BaseVector which provides the
    same interface for both the STL and RW vector classes. If the STL
    version of BaseVector is based on the valarray class then some
    basic vector operations such as + , * are available.  This class
    adds functionality to read/write vectors in a variety of ways */

template<class T>
class Vector : public BaseVector<T>
{
public:

  //
  //- Heading: Constructors and destructor
  //

  /// Default constructor
  Vector();
  /// Constructor which takes an initial length
  Vector(size_t len);
  /// Constructor which takes an initial length and an initial value
  Vector(size_t len, const T& initial_val);
  /// Copy constructor
  Vector(const Vector<T>& a);
  /// Constructor which copies len entries from T*
  Vector(const T* p, size_t len);
  /// Destructor
  ~Vector();

  //
  //- Heading: Assignment operators
  //

  /// Normal const assignment operator.
  Vector<T>& operator=(const Vector<T>& a);

  /// Sets all elements in self to the value ival.
  Vector<T>& operator=(const T& ival);

  //
  //- Heading: Conversion operators
  //

  /// Converts the Vector to a standard C-style array.
  /// Use with care!
  operator T*() const;

  //
  //- Heading: Member functions
  //

  /// Reads a Vector from an input stream
  void read(istream& s);
  /// Reads a Vector and associated label array from an input stream
  void read(istream& s, Array<String>& label_array);

  /// Reads part of a Vector from an input stream
  void read_partial(istream& s, size_t start_index, size_t num_items);
  /// Reads part of a Vector and the corresponding labels from an input stream
  void read_partial(istream& s, size_t start_index, size_t num_items, 
		    Array<String>& label_array);

  /// Reads a Vector from a tabular text input file
  void read_tabular(istream& s);

  /// Reads a Vector and associated label array in annotated from an
  /// input stream
  void read_annotated(istream& s, Array<String>& label_array);

  /// Writes a Vector to an output stream
  void write(ostream& s) const;
  /// Writes a Vector and associated label array to an output stream
  void write(ostream& s, const Array<String>& label_array) const;

  /// Writes part of a Vector to an output stream
  void write_partial(ostream& s, size_t start_index, size_t num_items) const;
  /// Writes part of a Vector and the corresponding labels to an
  /// output stream
  void write_partial(ostream& s, size_t start_index, size_t num_items, 
		     const Array<String>& label_array) const;

  /// Writes a Vector and associated label array to an output stream
  /// in aprepro format
  void write_aprepro(ostream& s, const Array<String>& label_array) const;

  /// Writes part of a Vector and the corresponding labels to an
  /// output stream in aprepro format
  void write_partial_aprepro(ostream& s, size_t start_index, size_t num_items,
			     const Array<String>& label_array) const;

  /// Writes a Vector and associated label array in annotated form
  /// to an output stream
  void write_annotated(ostream& s, const Array<String>& label_array) const;

  /// Writes a Vector in tabular form to an output stream
  void write_tabular(ostream& s) const;

  /// Writes part of a Vector in tabular form to an output stream
  void write_partial_tabular(ostream& s, size_t start_index, size_t num_items)
    const;

  /// Reads a Vector and associated label array from a binary input stream
  void read(BiStream& s, Array<String>& label_array);

  /// Writes a Vector and associated label array to a binary output stream
  void write(BoStream& s, const Array<String>& label_array) const;

  /// Reads a Vector from a buffer after an MPI receive
  void read(MPIUnpackBuffer& s);
  /// Reads a Vector and associated label array from a buffer after an
  /// MPI receive
  void read(MPIUnpackBuffer& s, Array<String>& label_array);

  /// Writes a Vector to a buffer prior to an MPI send
  void write(MPIPackBuffer& s) const;
  /// Writes a Vector and associated label array to a buffer prior to
  /// an MPI send
  void write(MPIPackBuffer& s, const Array<String>& label_array) const;

private:

  //
  //- Heading: Data
  //
};


/* Default constructor, defaults to base class constructor */
template <class T>
inline Vector<T>::Vector()
{ }


template <class T>
inline Vector<T>::Vector(size_t len): BaseVector<T>(len)
{ }


template <class T> 
inline Vector<T>::Vector(size_t len, const T& initial_val):
  BaseVector<T>(len, initial_val)
{ }


template <class T> 
inline Vector<T>::Vector(const Vector<T>& a): BaseVector<T>(a)
{ }


/** Assigns size values from p into array. */
template <class T>
inline Vector<T>::Vector(const T* p, size_t len): BaseVector<T>(len)
{
  for (size_t i=0; i<len; i++)
    BaseVector<T>::operator[](i) = p[i];
}


template <class T>
inline Vector<T>::~Vector()
{ }


template <class T>
inline Vector<T>& Vector<T>::operator=(const Vector<T>& a)
{
  if (this != &a) // verify that these are different objects
    BaseVector<T>::operator=(a); // base class assignment operator
  return *this;
}


/** Assigns all values of array to ival. If STL, uses the vector assign method
    because there is no operator=(ival). */
template <class T>
inline Vector<T>& Vector<T>::operator=(const T& ival)
{
  // set the values in the array using the base class assign() method
  assign(DAKOTA_BASE_VECTOR<T>::size(), ival);
  return *this;
}


// I would like to use the const version but linking with the FORTRAN
// and passing the array prevents const!
//template<class T> Vector<T>::operator const T*() const
//{ return BaseVector<T>::data(); }
template <class T>
inline Vector<T>::operator T*() const
{ return BaseVector<T>::array(); }


template <class T>
void Vector<T>::read(istream& s)
{
  size_t len = DAKOTA_BASE_VECTOR<T>::size();
  for (register size_t i=0; i<len; i++)
    s >> BaseVector<T>::operator[](i);
}


template <class T>
void Vector<T>::read(istream& s, Array<String>& label_array)
{
  size_t len = DAKOTA_BASE_VECTOR<T>::size();
  if (label_array.size() != len) {
    Cerr << "Error: size of label_array in Vector<T>::read(istream) does not "
	 << "equal length of Vector." << endl;
    abort_handler(-1);
  }
  for (register size_t i=0; i<len; i++)
    s >> BaseVector<T>::operator[](i) >> label_array[i];
}


template <class T>
void Vector<T>::read_partial(istream& s, size_t start_index, size_t num_items)
{
  size_t end = start_index + num_items;
  if (end > DAKOTA_BASE_VECTOR<T>::size()) { // start_index >= 0 since size_t
    Cerr << "Error: indexing in Vector<T>::read_partial(istream) exceeds "
	 << "length of Vector." << endl;
    abort_handler(-1);
  }
  for (register size_t i=start_index; i<end; i++)
    s >> BaseVector<T>::operator[](i);
}


template <class T>
void Vector<T>::read_partial(istream& s, size_t start_index, size_t num_items, 
			     Array<String>& label_array)
{
  size_t end = start_index + num_items, len = DAKOTA_BASE_VECTOR<T>::size();
  if (end > len) { // start_index >= 0 since size_t
    Cerr << "Error: indexing in Vector<T>::read_partial(istream) exceeds "
	 << "length of Vector." << endl;
    abort_handler(-1);
  }
  if (label_array.size() != len) {
    Cerr << "Error: size of label_array in Vector<T>::read_partial(istream) "
	 << "does not equal length of Vector." << endl;
    abort_handler(-1);
  }
  for (register size_t i=start_index; i<end; i++)
    s >> BaseVector<T>::operator[](i) >> label_array[i];
}


template <class T>
void Vector<T>::read_tabular(istream& s)
{
  // differs from read(istream& s) only in exception handling
  size_t len = DAKOTA_BASE_VECTOR<T>::size();
  for (register size_t i=0; i<len; i++) {
    if (s)
      s >> BaseVector<T>::operator[](i);
    else {
      char err[80];
      sprintf(err, "At EOF: insufficient tabular data for vector[%d]", i);
      throw String(err);
    }
  }
}


template <class T>
void Vector<T>::read_annotated(istream& s, Array<String>& label_array)
{
  size_t len;
  s >> len;

  if( len != DAKOTA_BASE_VECTOR<T>::size() )
    DAKOTA_BASE_VECTOR<T>::resize(len);

  if( len != label_array.size() )
    label_array.resize(len);

  for (register size_t i=0; i<len; i++)
    s >> BaseVector<T>::operator[](i) >> label_array[i];
}


template <class T>
void Vector<T>::write(ostream& s) const
{
  size_t len = DAKOTA_BASE_VECTOR<T>::size();
  s.setf(ios::scientific);
  s << setprecision(write_precision);
  for (register size_t i=0; i<len; i++)
    s << "                     " << setw(write_precision+7) 
      << BaseVector<T>::operator[](i) << '\n';
}


template <class T>
void Vector<T>::write(ostream& s, const Array<String>& label_array) const
{
  size_t len = DAKOTA_BASE_VECTOR<T>::size();
  if (label_array.size() != len) {
    Cerr << "Error: size of label_array in Vector<T>::write(ostream) does not "
	 << "equal length of Vector." << endl;
    abort_handler(-1);
  }
  s.setf(ios::scientific);
  s << setprecision(write_precision);
  for (register size_t i=0; i<len; i++)
    s << "                     " << setw(write_precision+7) 
      << BaseVector<T>::operator[](i) << ' ' << label_array[i] << '\n';
}


template <class T>
void Vector<T>::
write_aprepro(ostream& s, const Array<String>& label_array) const
{
  size_t len = DAKOTA_BASE_VECTOR<T>::size();
  if (label_array.size() != len) {
    Cerr << "Error: size of label_array in Vector<T>::write_aprepro(ostream) "
	 << "does not equal length of Vector." << endl;
    abort_handler(-1);
  }
  s.setf(ios::scientific);
  s << setprecision(write_precision);
  for (register size_t i=0; i<len; i++)
    s << "                    { " << setw(15) << setiosflags(ios::left) 
      << label_array[i].data() << resetiosflags(ios::adjustfield) << " = " 
      << setw(write_precision+7) << BaseVector<T>::operator[](i) <<" }\n";
}


template <class T>
void Vector<T>::
write_partial(ostream& s, size_t start_index, size_t num_items) const
{
  size_t end = start_index + num_items;
  if (end > DAKOTA_BASE_VECTOR<T>::size()) { // start_index >= 0 since size_t
    Cerr << "Error: indexing in Vector<T>::write_partial(ostream) exceeds "
	 << "length of Vector." << endl;
    abort_handler(-1);
  }
  s.setf(ios::scientific);
  s << setprecision(write_precision);
  for (register size_t i=start_index; i<end; i++)
    s << "                     " << setw(write_precision+7) 
      << BaseVector<T>::operator[](i) << '\n';
}


template <class T>
void Vector<T>::
write_partial(ostream& s, size_t start_index, size_t num_items, 
              const Array<String>& label_array) const
{
  size_t end = start_index + num_items, len = DAKOTA_BASE_VECTOR<T>::size();
  if (end > len) { // start_index >= 0 since size_t
    Cerr << "Error: indexing in Vector<T>::write_partial(ostream) exceeds "
	 << "length of Vector." << endl;
    abort_handler(-1);
  }
  if (label_array.size() != len) {
    Cerr << "Error: size of label_array in Vector<T>::write_partial(ostream) "
	 << "does not equal length of Vector." << endl;
    abort_handler(-1);
  }
  s.setf(ios::scientific);
  s << setprecision(write_precision);
  for (register size_t i=start_index; i<end; i++)
    s << "                     " << setw(write_precision+7) 
      << BaseVector<T>::operator[](i) << ' ' << label_array[i] << '\n';
}


template <class T>
void Vector<T>::
write_partial_aprepro(ostream& s, size_t start_index, size_t num_items, 
                      const Array<String>& label_array) const
{
  size_t end = start_index + num_items, len = DAKOTA_BASE_VECTOR<T>::size();
  if (end > len) { // start_index >= 0 since size_t
    Cerr << "Error: indexing in Vector<T>::write_partial_aprepro(ostream) "
	 << "exceeds length of Vector." << endl;
    abort_handler(-1);
  }
  if (label_array.size() != len) {
    Cerr << "Error: size of label_array in Vector<T>::write_partial_aprepro"
	 << "(ostream) does not equal length of Vector." << endl;
    abort_handler(-1);
  }
  s.setf(ios::scientific);
  s << setprecision(write_precision);
  for (register size_t i=start_index; i<end; i++)
    s << "                    { " << setw(15) << setiosflags(ios::left) 
      << label_array[i].data() << resetiosflags(ios::adjustfield) << " = " 
      << setw(write_precision+7) << BaseVector<T>::operator[](i) <<" }\n";
}


template <class T>
void Vector<T>::
write_annotated(ostream& s, const Array<String>& label_array) const
{
  size_t len = DAKOTA_BASE_VECTOR<T>::size();
  if (label_array.size() != len) {
    Cerr << "Error: size of label_array in Vector<T>::write_annotated(ostream) "
	 << "does not equal length of Vector." << endl;
    abort_handler(-1);
  }
  s.setf(ios::scientific);
  s << len << ' ' << setprecision(write_precision);
  for (register size_t i=0; i<len; i++)
    s << BaseVector<T>::operator[](i) << ' ' << label_array[i] << ' ';
}


template <class T>
void Vector<T>::write_tabular(ostream& s) const
{
  size_t len = DAKOTA_BASE_VECTOR<T>::size();
  s << setprecision(10) << resetiosflags(ios::floatfield);
  for (register size_t i=0; i<len; i++)
    s << setw(14) << BaseVector<T>::operator[](i) << ' ';
}


template <class T>
void Vector<T>::
write_partial_tabular(ostream& s, size_t start_index, size_t num_items) const
{
  size_t end = start_index + num_items;
  if (end > DAKOTA_BASE_VECTOR<T>::size()) { // start_index >= 0 since size_t
    Cerr << "Error: indexing in Vector<T>::write_partial_tabular(ostream) "
	 << "exceeds length of Vector." << endl;
    abort_handler(-1);
  }
  s << setprecision(10) << resetiosflags(ios::floatfield);
  for (register size_t i=start_index; i<end; i++)
    s << setw(14) << BaseVector<T>::operator[](i) << ' ';
}


/// global istream extraction operator for Vector
template <class T>
inline istream& operator>>(istream& s, Vector<T>& data)
{ data.read(s); return s; }


/// global ostream insertion operator for Vector
template <class T>
inline ostream& operator<<(ostream& s, const Vector<T>& data)
{ data.write(s); return s; }


template <class T>
void Vector<T>::read(BiStream& s, Array<String>& label_array)
{
  // Explicitly read a size_t to prevent buffer alignment problem
  // caused by writing a size_t and reading an int.
  size_t len;
  s >> len;

  if( len != DAKOTA_BASE_VECTOR<T>::size() )
    DAKOTA_BASE_VECTOR<T>::resize(len);

  if( len != label_array.size() )
    label_array.resize(len);

  for (register size_t i=0; i<len; i++)
    s >> BaseVector<T>::operator[](i) >> label_array[i];
}


template <class T>
void Vector<T>::write(BoStream& s, const Array<String>& label_array) const
{
  // Explicitly write a size_t to prevent buffer alignment problem
  // caused by writing a size_t and reading an int.
  size_t len = DAKOTA_BASE_VECTOR<T>::size();
  if (label_array.size() != len) {
    Cerr << "Error: size of label_array in Vector<T>::write(BoStream) does not "
	 << "equal length of Vector." << endl;
    abort_handler(-1);
  }

  s << len;
  for (register size_t i=0; i<len; i++)
    s << BaseVector<T>::operator[](i) << label_array[i];
}


template <class T>
void Vector<T>::read(MPIUnpackBuffer& s)
{
  // Slaves receive parameter vector and tags from master processor.

  // Explicitly read a size_t to prevent buffer alignment problem
  // caused by writing a size_t and reading an int.
  size_t len;
  s >> len;

  if( len != DAKOTA_BASE_VECTOR<T>::size() )
    DAKOTA_BASE_VECTOR<T>::resize(len);

  for (register size_t i=0; i<len; i++)
    s >> BaseVector<T>::operator[](i);
}


template <class T>
void Vector<T>::read(MPIUnpackBuffer& s, Array<String>& label_array)
{
  // Slaves receive parameter vector and tags from master processor.

  // Explicitly read a size_t to prevent buffer alignment problem
  // caused by writing a size_t and reading an int.
  size_t len;
  s >> len;

  if( len != DAKOTA_BASE_VECTOR<T>::size() )
    DAKOTA_BASE_VECTOR<T>::resize(len);

  if( len != label_array.size() )
    label_array.resize(len);

  for (register size_t i=0; i<len; i++)
    s >> BaseVector<T>::operator[](i) >> label_array[i];
}


template <class T>
void Vector<T>::write(MPIPackBuffer& s) const
{
  // Master processor sends parameter vector and tags to slaves.

  // Explicitly write a size_t to prevent buffer alignment problem
  // caused by writing a size_t and reading an int.
  size_t len = DAKOTA_BASE_VECTOR<T>::size();
  s << len;

  for (register size_t i=0; i<len; i++)
    s << BaseVector<T>::operator[](i);
}


template <class T>
void Vector<T>::write(MPIPackBuffer& s, const Array<String>& label_array) const
{
  // Master processor sends parameter vector and tags to slaves.

  // Explicitly write a size_t to prevent buffer alignment problem
  // caused by writing a size_t and reading an int.
  size_t len = DAKOTA_BASE_VECTOR<T>::size();
  if (label_array.size() != len) {
    Cerr << "Error: size of label_array in Vector<T>::write(MPIPackBuffer) "
	 << "does not equal length of Vector." << endl;
    abort_handler(-1);
  }

  s << len;
  for (register size_t i=0; i<len; i++)
    s << BaseVector<T>::operator[](i) << label_array[i];
}


/// global MPIUnpackBuffer extraction operator for Vector
template <class T>
inline MPIUnpackBuffer& operator>>(MPIUnpackBuffer& s, Vector<T>& data)
{ data.read(s); return s; }


/// global MPIPackBuffer insertion operator for Vector
template <class T>
inline MPIPackBuffer& operator<<(MPIPackBuffer& s, const Vector<T>& data)
{ data.write(s); return s; }

} // namespace Dakota

#endif  // DAKOTA_VECTOR_H
