// $Id: APPSPACK_Citizen_NAPPSListener.cpp,v 1.2 2008/05/02 00:57:23 tgkolda Exp $ 
// $Source: /usr/local/cvsroot/hopspack/src/APPSPACK_Citizen_NAPPSListener.cpp,v $ 

//@HEADER
// ************************************************************************
// 
//         HOPSPACK: Hybrid Opitmization Parallel Search Package
//               Copyright (2008) Sandia Corporation
// 
// Under terms of Contract DE-AC04-94AL85000, there is a non-exclusive
// license for use of this work by or on behalf of the U.S. Government.
// 
// This library is free software; you can redistribute it and/or modify
// it under the terms of the GNU Lesser General Public License as
// published by the Free Software Foundation; either version 2.1 of the
// License, or (at your option) any later version.
//  
// This library is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.
//                                                                                 
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
// USA.                                                                           .
// 
// Questions? Contact Tammy Kolda (tgkolda@sandia.gov) 
// 
// ************************************************************************
//@HEADER

/*!
  \file APPSPACK_Citizen_NAPPSListener.cpp
  \brief Implements APPSPACK::Citizen::NAPPSListener
*/

#include "APPSPACK_Citizen_NAPPSListener.hpp"
#include "APPSPACK_Print.hpp"
#include <time.h>
#include <cassert>

APPSPACK::Citizen::NAPPSListener::NAPPSListener(APPSPACK::Parameter::List& params_in,
                                      const APPSPACK::Constraints::Linear& constraints_in,
                                      APPSPACK::Combiner::Generic& combiner_in,
                                      const string name_in) :
  name(name_in),
  params(params_in),
  resultsFile(params.sublist(name_in).getParameter("Results File", "listener.txt")),
  evalCount(0),
  etaStar(params.sublist("Nonlinear").getParameter("Constraint Tolerance", 1e-5)),
  resubmitBest(params.sublist(name_in).getParameter("Resubmit Best", true)),
  nlec(params.sublist("Nonlinear").getParameter("Number Equality", 0)),
  nlic(params.sublist("Nonlinear").getParameter("Number Inequality", 0))
{
  time(&startTime);
}

APPSPACK::Citizen::NAPPSListener::~NAPPSListener()
{
}

void APPSPACK::Citizen::NAPPSListener::preprocess()
{
  resultsStream.open(resultsFile.c_str());
}

void APPSPACK::Citizen::NAPPSListener::postprocess()
{
  // Record time immediately.
  time_t endTime;
  time(&endTime);
  double difTime=difftime(endTime,startTime);

  resultsStream.close();
  cout.precision(8);
  string success="f";
  if (infnorm(fcBest) <= etaStar)
    success="s";
  
  cout << "Listener: " << evalCount << ", " << fcBest[0] << ", " 
       << infnorm(fcBest) << ", "  << success << ", " << xBest.size()
       << ", " << nlec << ", " << nlic << ", f=[ ";
  cout.precision(8);
  for (int i=0; i<fcBest.size(); i++)
    cout << fcBest[i] << " ";
  cout << "], x=[";
  for (int i=0; i<xBest.size(); i++)
    cout << xBest[i] <<  " ";
  cout << "], time = [" << difTime << "]" << endl;
}

void APPSPACK::Citizen::NAPPSListener::exchange(const APPSPACK::ConveyorList& R,
                                           APPSPACK::ConveyorList& Wnew,
                                           const map<string, vector<int> >& ownermap)
{
  // Record time immediately.
  time_t endTime;
  time(&endTime);
  double difTime=difftime(endTime,startTime);

  // First invert ownermap.
  map<int,string> tag2name;
  map<string, vector<int> >::const_iterator iter;
  for (iter=ownermap.begin(); iter != ownermap.end(); ++iter) 
  {
    const vector<int>& tlist=iter->second;
    for (int i=0; i<tlist.size(); i++)
      tag2name[tlist[i]] = iter->first;    
  }

  // Now loop through evaluated points.
  vector<int> rtags;
  R.getTagList(rtags);
  list<APPSPACK::Point*>::const_iterator itr;
  for (itr=R.begin(); itr!=R.end(); itr++)
  {
    string nameEval="";
    Vector fc = (*itr)->getVecF();
    Vector x = (*itr)->getX();
    Point::State state = (*itr)->getState();
    int tag = (*itr)->getTag();
    double step = (*itr)->getStep();

    // Points coming from mediator should only be evaluated or cached.
    if ( (state == Point::Infeasible) || (state == Point::Unevaluated) )
    {
      cerr << "Error: Unexpected state.." << endl;
      cerr << "This point's state is " << state << endl;
      cerr << "Owned by " << tag2name[tag] << endl;
      throw "APPSPACK Error";
    }

    if (tag2name.find(tag) == tag2name.end())
    {
      cerr << "Error: failed to find point tag" << endl;
      throw "APPSPACK Error";
    }

    // If point is not Cached eval, process.
    if ( ((*itr)->getState() == Point::EvaluatedInsufficientDecrease) ||
         ((*itr)->getState() == Point::Evaluated) )
    {
      evalCount++;
      nameEval=tag2name[tag];

      // The first evaluated point will always be best.
      if (fcBest.size() == 0)
      {
	nameEval+=" newbest";
	xBest = x;
	fcBest = fc;
      }
      
      double fcnorm=infnorm(fc);
      double fcnormTol=infnorm(fcBest);
      
      // Case I.  Both fc and fcBestTol satisfy tolerance.    
      if ((fcnorm < etaStar) && (fcnormTol < etaStar))
      {
	if (fc[0] < fcBest[0])
	{
	  nameEval+=" newbest";
	  xBest=x;
	  fcBest=fc;
	}
      }
      else if (fcnorm < fcnormTol)
      {
	// Neither satisfies tolerance, but x is closer to feasible
        nameEval+=" newbest";
	xBest=x;
	fcBest=fc;
      }
      
      resultsStream.precision(8);
      resultsStream << evalCount << ", " 
                    << APPSPACK::Print::formatDouble(infnorm(fc)) 
                    << ", " << nameEval  
                    << ", tag=" << tag << ", f=[ ";
      resultsStream.precision(8);
      for (int i=0; i<fc.size(); i++)
        resultsStream << fc[i] << " ";
      resultsStream << "], x=[";
      for (int i=0; i<x.size(); i++)
        resultsStream << x[i] <<  " ";
      resultsStream << "], time=[" << difTime << "]" 
                    << ", step=[" << step << "]" << endl;
    }
  }
  resultsStream << "======================================" << endl;

  if ((fcBest.size() > 0) && (resubmitBest == true))
  {
    //Now submitted dummy version of best point found so far:
    Vector tmp;
    string msg="Listener";
    Point* p = new Point(xBest, tmp, .33, 0.0, combiner, msg);  
    bestTag=p->getTag();
    Wnew.push(p);
  }

}

void APPSPACK::Citizen::NAPPSListener::vote(const APPSPACK::ConveyorList& W,
                                       const map<string, vector<int> >& ownermap,
                                       vector<int>& tagOrder)
{
  if ((fcBest.size() > 0) && (resubmitBest == true))
    tagOrder.push_back(bestTag);
}

bool APPSPACK::Citizen::NAPPSListener::worker()
{
  // Should never be called.
  cerr << "APPSPACK::Citizen::NAPPSListener::worker() should not be called" << endl;
  throw "APPSPACK Error";
}

APPSPACK::Citizen::State APPSPACK::Citizen::NAPPSListener::getState()
{
  // NAPPSListener wishes to stop when everyone is ready.
  return APPSPACK::Citizen::StopRequest;
}

const string& APPSPACK::Citizen::NAPPSListener::getName() const
{
  return name;
}

double APPSPACK::Citizen::NAPPSListener::infnorm(APPSPACK::Vector fc)
{
  assert( (fc.size() == (nlic+nlec+1)) && "fc has incorrect size!");

  // Equality constraints (recall fc[0] = f(x)):
  // fc[1], ... , fc[nlec] 
  // Inequality constraints. (note that we want fc[i] <= 0):
  // fc[nlec+1], ... , fc[fc.size() - 1]

  double norm=0;
  for (int i=1; i<fc.size(); i++)
  {
    if (i <= nlec)
      norm = max(fabs(fc[i]), norm);
    else
      norm = max(fc[i], norm);
  }
  
  return norm;
}
