//@HEADER
// ************************************************************************
// 
//         HOPSPACK: Hybrid Opitmization Parallel Search Package
//               Copyright (2008) Sandia Corporation
// 
// Under terms of Contract DE-AC04-94AL85000, there is a non-exclusive
// license for use of this work by or on behalf of the U.S. Government.
// 
// This library is free software; you can redistribute it and/or modify
// it under the terms of the GNU Lesser General Public License as
// published by the Free Software Foundation; either version 2.1 of the
// License, or (at your option) any later version.
//  
// This library is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.
//                                                                                 
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
// USA.                                                                           .
// 
// Questions? Contact Tammy Kolda (tgkolda@sandia.gov) 
// 
// ************************************************************************
//@HEADER

/*! 
  \file NAPPSPACK_Method_Smooth.cpp
  \brief Implementation of NAPPSPACK::Method::Smooth
*/

#include <math.h> // for fmax with Sun's stlport4
#include "NAPPSPACK_Method_Smooth.hpp"
#include "APPSPACK_Common.hpp"
#include "APPSPACK_Parameter_List.hpp"
#include "APPSPACK_Float.hpp"
#include "APPSPACK_Cache_Point.hpp"

NAPPSPACK::Method::Smooth::Smooth(APPSPACK::Parameter::List& params_in) :
  params(params_in),
  methodName(params.sublist("Nonlinear").getParameter("Method","Merit Two Smooth")),
  combiner(params.sublist("Nonlinear").getParameter("Method","Merit Two Smooth"))
{  
  // Get number of nonlinear constraints.
  int nlec = params.sublist("Nonlinear").getParameter("Number Equality", 0);
  int nlic = params.sublist("Nonlinear").getParameter("Number Inequality", 0);

  // Give combiner number of nonlinear inequality and equality constraints.
  combiner.setConstraintSize(nlec, nlic);

  double rhoInit=params.sublist("Nonlinear").getParameter("Initial Penalty Value", 1.0);
  combiner.setRho(rhoInit);
  double alphaInit=params.sublist("Nonlinear").getParameter("Initial Smoothing Value", 1.0);
  combiner.setAlpha(alphaInit);
  combiner.setDebug(params.sublist("Nonlinear").getParameter("Debug",0));

  // Desired optimality.
  etaStar = params.sublist("Nonlinear").getParameter("Constraint Tolerance", 1e-5);

  // Stopping step size for algorithm.
  deltaStar = params.sublist("Nonlinear").getParameter("Final Step Tolerance", 1e-5);
  // Current step tolerance for GSS subproblem.
  deltak = params.sublist("Nonlinear").getParameter("Initial Step Tolerance", 1e-2);
  // Minimum value for deltak.
  deltaMin = params.sublist("Nonlinear").getParameter("Minimum Step Tolerance", 1e-3*deltaStar);
  // Amount by which delta is scaled and hence reduced after each major.
  deltaDecrease = params.sublist("Nonlinear").getParameter("Step Tolerance Decrease", .5);

  // TwoNorm value for rho;
  rhoMax = params.sublist("Nonlinear").getParameter("Maximum Penalty Value", 1e8);
  // Amount by which penalty parameter is increased after each major.
  rhoIncrease = params.sublist("Nonlinear").getParameter("Penalty Value Increase", 2.0);

  // Minimum value for alpha.
  alphaMin = params.sublist("Nonlinear").getParameter("Minimum Smoothing Value", 1e-5);
  // Amount by which delta is scaled and hence reduced after each major.
  alphaDecrease = params.sublist("Nonlinear").getParameter("Smoothing Value Decrease", .5);
  // Set inital value of cache tolerance.
  double newCache=min(APPSPACK::Cache::Point::tolerance, deltak/2);
  APPSPACK::Cache::Point::setStaticTolerance(newCache);
  if (debug > 3)
  {
    params.sublist("Solver").setParameter("Debug", debug);
    cout << "\nStatic Cache One=" << APPSPACK::Cache::Point::tolerance<< endl;
  }
}

void NAPPSPACK::Method::Smooth::updateAPPSPACKParams(double tol)
{
  // Set cache.  Don't allow increase, ever.
  double newCache=min(APPSPACK::Cache::Point::tolerance, tol/2);
  APPSPACK::Cache::Point::setStaticTolerance(newCache);
  if (debug > 3)
    cout << "\nStatic Cache One=" << APPSPACK::Cache::Point::tolerance<< endl;

  params.sublist("Solver").setParameter("Bounds Tolerance", tol/2);
  params.sublist("Solver").setParameter("Step Tolerance", tol);
  params.sublist("Solver").setParameter("Minimum Step", 2*tol);
  params.sublist("Solver").setParameter("Cache Comparison Tolerance", tol/2);
  //  cout << "Cache tolerance:" <<  APPSPACK::Cache::Point::tolerance << endl;
}

void NAPPSPACK::Method::Smooth::update(APPSPACK::Vector& x, APPSPACK::Vector& fc,
				       const APPSPACK::Solver::State& state)
{ 
  // Increase penalty parameter.
  if (combiner.infnorm(fc) > fmax(etaStar, .1*combiner.getErrorTol(fc)) )
  {
    double rhoNew = rhoIncrease*combiner.getRho();
    combiner.setRho(min(rhoMax, rhoNew));
  }
  
  // Decrease smoothing parameter.
  double alphaNew = alphaDecrease*combiner.getAlpha();
  combiner.setAlpha(max(alphaMin, alphaNew));
  
  // Update stop tolerance for GSS subproblem.
  deltak = max(deltaDecrease*deltak, deltaMin);
  updateAPPSPACKParams(deltak);
}

bool NAPPSPACK::Method::Smooth::isOptimal(const APPSPACK::Vector& fc) const
{
  // Return optimal if deltak is be below stopping step tolerance
  // and current point is sufficiently feasible.
  if ((deltak <= deltaStar) && (combiner.infnorm(fc) <= etaStar))
    return true;
  
  return false;
}

void NAPPSPACK::Method::Smooth::printMajor(const APPSPACK::Vector& x, 
                                            const APPSPACK::Vector& fc) const
{
  APPSPACK::Vector violation;
  combiner.getInfVector(fc, violation);
  cout << "f(x)=" <<  APPSPACK::Print::formatDouble(fc[0]) 
       << ", cinf=" 
       << APPSPACK::Print::formatDouble(combiner.infnorm(fc))
       << ", c+(x)=" << violation << ", x=" << x << ", rho=" 
       << combiner.getRho() << ", r=" << combiner.getAlpha() << ", dk=" << deltak 
       << ", et=" << combiner.getErrorTol(fc);
}

APPSPACK::Combiner::Smooth& NAPPSPACK::Method::Smooth::getCombiner()
{
  return combiner;
}

const string& NAPPSPACK::Method::Smooth::getName()
{
  return methodName;
}

double  NAPPSPACK::Method::Smooth::infnorm(const APPSPACK::Vector& fc) const
{
  return combiner.infnorm(fc);
}
