// $Id: APPSPACK_Conveyor.hpp,v 1.2 2008/05/02 00:57:22 tgkolda Exp $ 
// $Source: /usr/local/cvsroot/hopspack/src-conveyor/APPSPACK_Conveyor.hpp,v $ 

//@HEADER
// ************************************************************************
// 
//         HOPSPACK: Hybrid Opitmization Parallel Search Package
//               Copyright (2008) Sandia Corporation
// 
// Under terms of Contract DE-AC04-94AL85000, there is a non-exclusive
// license for use of this work by or on behalf of the U.S. Government.
// 
// This library is free software; you can redistribute it and/or modify
// it under the terms of the GNU Lesser General Public License as
// published by the Free Software Foundation; either version 2.1 of the
// License, or (at your option) any later version.
//  
// This library is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.
//                                                                                 
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
// USA.                                                                           .
// 
// Questions? Contact Tammy Kolda (tgkolda@sandia.gov) 
// 
// ************************************************************************
//@HEADER

/*!
  \file APPSPACK_Conveyor.hpp
  \brief Class definition for APPSPACK::Conveyor
*/
#ifndef APPSPACK_CONVEYOR_HPP
#define APPSPACK_CONVEYOR_HPP

#include "APPSPACK_ConveyorList.hpp"
#include "APPSPACK_Cache_Manager.hpp"
#include "APPSPACK_Executor_Interface.hpp"
#include "APPSPACK_Parameter_List.hpp"
#include "APPSPACK_Counter.hpp"

namespace APPSPACK
{

//! Conveys trial points through the process of being queued, cached, and then evaluated.
/*!  The conveyor's main interface is the exchange() function. It
  takes a (possibly empty) list of unevaluated trial points and
  returns a (non-empty) list of evaluated trial points with
  corresponding function values.

  The conveyor owns a APPSPACK::Cache::Manager to manage the cache;
  see #cache.

  The conveyor owns a user-provided APPSPACK::Executor object for the
  function evaluations; see #executor.  Points that are currently in
  the process of being evaluated by #executor are stored in the
  #pendingList, an instance of APPSPACK::ConveyorList.  

  When a point \f$p_k\f$ submitted to the conveyor a check is first
  made to determine if this point has already been evaluated and is stored
  within the #cache.  If \f$p_k\f$ is cached, then it is assigned the
  appropriate objective value.  If \f$p_k\f$ is not cached, a
  subsequent check is made to determine if the point is currently being
  evaluated.  If \f$p_k\f$ equals a point \f$p_j\f$ in #pendingList,
  then the tag corresponding to \f$p_j\f$ and the point \f$p_k\f$ are
  stored together the multimap object, #pendingPrime, i.e
  
  \f[ (tag(p_j), pointer(p_k)) \f] 
  
  is stored in a mulitmap.  This prevents the unnecessary evaluation
  of duplicate points.  
  
  When evaluated points return, a check is made to
  determine if the evaluated point's is tag contained within #pendingPrime.  If so,
  all points in #pendingPrime corresponding to returning point's tag, are assigned
  the returning point's objective value and added to the exchange list.  

  \see APPSPACK::Conveyor::exchange()

  \note The test used for equality is identical to that used by APPSPACK::Cache::Point.
*/
class Conveyor
{
public:


  //! Constructor 
  Conveyor(Parameter::List& params, const Vector& scaling_in, 
	   Executor::Interface& executor_in);

  //! Constructor 
  Conveyor(Parameter::List& params, const Vector& scaling_in, 
	   Executor::Interface& executor_in,
	   Cache::Manager& cache);

  //! Destructor 
  ~Conveyor();

  /*! 

  \brief Exchange a list of unevaluated trial points for a list of
    evaluated trial points.

  \param queueList (input/output) Conveyor will begin evaluations in
  FIFO.  Once a point is popped from the queueList, ownership is
  extended to either #pendingPrime, #pendingList, or evalList. On
  exit, the queueList will retain ownership points that the Conveyor has not yet
  looked at.

  \param evalList (output) On exit, list of newly evaluated points are return.

  <ul>

  <li> Collect points to be returned (in exchangeList). So long as the
       return list is smaller than the minimum return size
       (#minReturn) and there are still points to be evaluated (i.e.,
       either the queue is non-empty and/or the pending list is
       non-empty):

       <ul>
       <li> While the queue is non-empty and the executor is not full:
  
            <ul>
	    <li> Pop the next trial point off the queue.

	    <li> Check to see if that trial point is cached. If it is
	    cached, increment the cached value count, fill in the
	    function value, and add this trial point to the return
	    list.
	    
            <li> Check to see if the point is cache equivalent to a point
            in #pendingList, the list of points currently being evaluated.
            If so, this point and the corresponding tag of the point in
            #pendingList are added to the multimap #pendingPrime.  For example,
            if point \f$ p_k \f$ is cache equivalent to point \f$ p_j \f$
            (which has tag \f$ t_j \f$) in #pendingList.  Then
            the pair \f$(t_j, p_k)\f$ are added to #pendingPrime.
            Later, when \f$p_j\f$ returns from being evaluated, tag \f$ t_j\f$
            can be used to map the objective value of \f$p_j\f$ to \f$ p_k\f$.

	    <li> Otherwise, spawn an evaluation of the trial point
	    using the executor and push the trial point onto the
	    #pendingList.
	    </ul>

	    \dot
	    digraph G{
	    bgcolor=lightgrey
	    fontcolor=blue
	    label="Processing an unevaluted point"
	    node[shape=ellipse, peripheries=2, color=blue]
	    edge[color=red]
	    rankdir = LR
	    
	    cache[label="Is point cached?"]
	    yesCache[label="Assign objective value,\nadd point to exchangeList"]
	    noCache[label="Is point pending?"]  
	    
	    cache -> noCache [label="yes"]
	    cache -> yesCache [label="no"]
	    yesPending[label="Add to pendingPrime"]
	    noPending[label="Spawn evaluation,\nadd to pendingList"]
	    noCache -> yesPending [label="yes"]
	    noCache -> noPending [label="no"]
	    }
	    \enddot
	    
       <li> While the executor has a function value to return and the
            exchange list is not full:

	    <ul> 
	    <li> Increment the number of function evaluations.

	    <li> Pop the appropriate point off the pending list and fill in the function value.
	    
	    <li> Insert the new information into the queue for future looks.

	    <li> Add the trial point onto the return list.
     
            <li> Check to see if the tag of the trial point is in #pendingPrime.
            If so, all points stored in #pendingPrime with this tag
            are assigned the corresponding objective value of the returned trial point.

	    \dot
	    digraph G{
	    bgcolor=lightgray
	    fontcolor=blue
	    label="Processing an evaluted point pj"
	    node[shape=ellipse, peripheries=2, color=blue]
	    edge[color=red]
	    rankdir = LR
	    
	    tagq[label="Is tag(pj) in pendingPrime?"]
	    yestag[label="Assign all points in pendingPrime\n corresponding to tag(pj) the \n objective value of pj"]
	    notag[label="Add new evaluations \nto evaluationList"]
	    tagq -> yestag [label="yes"]
	    tagq -> notag [label="no"]
	    yestag -> notag
	    }
	    \enddot
  
	    </ul>
       </ul>
    </ul>
  */
  void exchange(ConveyorList& queueList, ConveyorList& evalList);

  //! Returns counts of all the function evaluations, etc.
  const Counter& getCounter() const;

  //! Returns number of points currently being evaluated.
  int getNumPending() const;

  //! Calls the print function of the #executor
  void print() const;

private:

  //! Object that is used to evaluate the function
  /*! This object is passed in to the constructor */
  Executor::Interface& executor;

  //! The default function value cache
  Cache::Manager cache;

  //! Reference to function value cache (may not be default)
  Cache::Manager& cacheRef;

  //! Trial points that are in the processed of being evaluated
  ConveyorList pendingList;

  /*! \brief Provides record of duplicate points sent to Conveyor
    that have not yet been evaluated, but are already deemed 
    equal (using "Cache Comparison Tolerance" \see Cache::Point)
    to points contained in #pendingList.
  */
  multimap<int, Point*> pendingPrime;

  /*! \brief Be synchronous (set according to "Synchronous") */
  bool doSync;

  /*! \brief The minimum number of items that should be returned on a
    call to exchange() (set according to "Minimum Exchange Return") */
  int minReturn;

  /*! \brief The maximum number of items that should be returned on a
    call to exchange() (set according to "Maximum Exchange Return") */
  int maxReturn;

  //! Counter for the number of evaluations, etc.
  Counter counter;

};

}

#endif
