/*  _________________________________________________________________________
 *
 *  Coliny: A Library of COLIN optimizers
 *  Copyright (c) 2003, Sandia National Laboratories.
 *  This software is distributed under the GNU Lesser General Public License.
 *  For more information, see the README.html file in the top Coliny directory.
 *  _________________________________________________________________________
 */

//
// Cobyla.cpp
//

#include <acro_config.h>
#ifdef ACRO_USING_COBYLA

#include <coliny/Factory.h>
#include <utilib/DoubleMatrix.h>
#include <coliny/Cobyla2.h>
#include <colin/AppResponseAnalysis.h>
#include "cobyla.h"

using namespace std;

namespace coliny {

real Cobyla::calcfc_value;
BasicArray<double> Cobyla::calcfc_point;
BasicArray<real> Cobyla::calcfc_cvec;
BasicArray<real> Cobyla::clower;
BasicArray<real> Cobyla::cupper;
BasicArray<real> Cobyla::blower;
BasicArray<real> Cobyla::bupper;


colin::OptProblem<BasicArray<double>,colin::AppResponse_Utilib>* Cobyla::calcfc_problem;


extern "C" int calc_func(int n, int m, double* x, double* f, double* con, void* state)
{ Cobyla::calcfc(n,m,x,f,con); return 0; }


void Cobyla::calcfc(int n, int m, double* x, double* f, double* con)
{
calcfc_point.set_data(n,((double*)x));
calcfc_problem->EvalF(calcfc_point,calcfc_value);

*f = calcfc_value;

int ndx=0;
if (calcfc_problem->enforcing_bounds()) {
   for (int i=0; i<n; i++) {
     if (finite(blower[i])) con[ndx++] = -(blower[i] - x[i]);
     if (finite(bupper[i])) con[ndx++] = -(x[i] - bupper[i]);
     }
   }
if (calcfc_cvec.size() > 0) {
   calcfc_problem->EvalCF(calcfc_point,calcfc_cvec);
   for (size_type i=0; i<calcfc_cvec.size(); i++) {
     if (finite(clower[i])) con[ndx++] = -(clower[i] - calcfc_cvec[i]);
     if (finite(cupper[i])) con[ndx++] = -(calcfc_cvec[i] - cupper[i]);
     }
   }

//for (int i=0; i<m; i++)
  //cerr << "CON " << i << " " << con[i] << endl;
}


Cobyla::Cobyla()
 : step_tolerance(1e-4),
   initial_step(1.0)
{
opt_name="Cobyla";

disable_parameter("max_iters");
disable_parameter("max_time");
disable_parameter("ftol");
disable_parameter("constraint_tolerance");
disable_parameter("output_frequency");
disable_parameter("output_flush");
disable_parameter("output_final");
disable_parameter("output_dynamic");
disable_parameter("debug_opt_stats");
disable_parameter("debug_all");
disable_parameter("debug_best_point");
disable_parameter("debug_num_evaluations");
disable_parameter("debug_iteration_info");
disable_parameter("debug_best");
disable_parameter("debug_time");

create_parameter("step_tolerance",step_tolerance,
        "<double>","1e-4",
        "Convergence tolerance step length");
create_parameter("initial_step",initial_step,
        "<double>","1.0",
        "Initial step length");
}


void Cobyla::reset()
{
if (!problem)
   return;

colin::StdOptSolver<BasicArray<double>,colin::AppResponse_Utilib>::reset();

if (problem.numEqConstraints() > 0)
   EXCEPTION_MNGR(runtime_error, "Cobyla::reset - " << problem.numEqConstraints() << " equality constraints cannot be handled by cobyla!");

if (problem.enforcing_bounds())
   problem.get_real_bounds(blower,bupper);
if (problem.numConstraints() > 0)
   problem.get_constraint_bounds(clower,cupper);

x.resize(problem.num_real_params());
calcfc_point.resize(problem.num_real_params());
calcfc_cvec.resize(problem.numConstraints());
calcfc_problem = &problem;
}


void Cobyla::minimize()
{
opt_init();

int n=problem.num_real_params();
if (n == 0) {
   best().termination_info = "No-Real-Params";
   return;
   }

x << best().point;	// get initial point

//
// Compute the number of constraints, including bound constraints
//
int ncon=problem.numConstraints();
int m=0;
if (problem.enforcing_bounds()) {
   if (problem.has_all_bounds())
      m += 2*n;
   else {
      for (int i=0; i<n; i++) {
	if (finite(blower[i])) m++;
	if (finite(bupper[i])) m++;
        }
      }
   }
if (ncon > 0) {
   for (int i=0; i<ncon; i++) {
     if (finite(clower[i])) m++;
     if (finite(cupper[i])) m++;
     }
   }

int iprint=(debug>3 ? 3 : debug);

int tmp_neval;
if (max_neval > 0) {
   tmp_neval = max(max_neval-problem.neval(),0);
   if (max_neval_curr != 0) 
      tmp_neval = min(tmp_neval, max_neval_curr);
   }
else
   tmp_neval = max_neval_curr;

int istatus = cobyla(n, m, &(x[0]), initial_step, step_tolerance, iprint, &tmp_neval, accuracy, calc_func,0);

switch (istatus) {
  case COBYLA_ENOMEM:
	best().termination_info="Memory-Errors";
	break;
  case COBYLA_NORMAL:
	best().termination_info="Step-Length";
	break;
  case COBYLA_MAXFUN:
	best().termination_info="Max-Num-Evals";
	break;
  case COBYLA_ROUNDING:
	best().termination_info="Roundoff-Errors";
	break;
  case COBYLA_FUNCVAL:
	best().termination_info="Accuracy";
	break;
  default:
	best().termination_info="Unknown";
	break;
  };

best().point << x;
if (problem.numConstraints() > 0)
   problem.Eval(x, best().response, colin::mode_f | colin::mode_cf);
else
   problem.Eval(x, best().response, colin::mode_f);
compute_response_info(best().response,problem.state->constraint_lower_bounds,  problem.state->constraint_upper_bounds, best().value(),best().constraint_violation);
}

}

typedef colin::OptSolver<utilib::BasicArray<double>,colin::             AppResponse_Utilib> base_t;
FACTORY_REGISTER(cobyla, base_t*, return new coliny::Cobyla();, "An alias to Cobyla")
FACTORY_REGISTER(Cobyla, base_t*, return new coliny::Cobyla();, "The Cobyla2 derivative-free optimizer")

#endif
