/*  _________________________________________________________________________
 *
 *  COLIN: A Common Optimization Library INterface
 *  Copyright (c) 2003, Sandia National Laboratories.
 *  This software is distributed under the GNU Lesser General Public License.
 *  For more information, see the README.html file in the top COLIN directory.
 *  _________________________________________________________________________
 */

/**
 * \file OptProblemFrag_LinearConstraints.h
 *
 * Defines the colin::OptProblemFragment_LinearConstraints class.
 */

#ifndef colin_OptProblemFrag_LinearConstraints_h
#define colin_OptProblemFrag_LinearConstraints_h

#include <acro_config.h>
#include <colin/OptProblemFrag_Base.h>

namespace colin {

//============================================================================
//============================================================================
// Class OptProblem_LinearConstraints
//============================================================================
//============================================================================

/**
 *  Defines the elements of an OptProblem that pertain to linear
 *  constraints.
 */
template <class DomainT, class ResponseT>
class OptProblemFragment_LinearConstraints : 
      virtual public OptProblemFragment_Base<DomainT,ResponseT>
{
public:

  ///
  virtual ~OptProblemFragment_LinearConstraints() {}

  /// Returns the number of linear equality constraints
  int numLinearEqConstraints() const
		{return this->state->numLinearEqConstraints;}

  /// Returns the number of linear inequality constraints
  int numLinearIneqConstraints() const
		{return this->state->numLinearIneqConstraints;}

  /// Returns the number of linear constraints
  int numLinearConstraints() const
		{return this->state->numLinearEqConstraints + 
			this->state->numLinearIneqConstraints;}

  /// Set linear constraints (for all variables) - sparse format
  int set_linear_constraints(const typename ResponseT::matrix_t& mat, const typename ResponseT::vector_t& rhs, const int num_equality=0)
		{return this->handle->set_linear_constraints(mat,rhs,num_equality);}

  /// Return linear constraints
  const typename ResponseT::matrix_t& get_linear_constr_matrix() const
		{return this->handle->linear_constraint_mat();}

  /// Return linear constraint right hand side
  const typename ResponseT::matrix_t& get_linear_constr_rhs() const
		{return this->handle->linear_constraint_rhs();}
};


/// A subclass of OptProblemFragment_LinearConstraints that includes
/// a method to access the linear constraint information.
template <class DomainT, class ResponseT, bool IsUsed>
class OptProblemInterface_LinearConstraints : public OptProblemFragment_LinearConstraints<DomainT,ResponseT>
{
public:

  /// Returns information about the linear constraints.
  OptProblemFragment_LinearConstraints<DomainT,ResponseT>& domain_linearconstraints()
                {return *this;}

};


/// Specialiation of OptProblemInterface_LinearConstraints for the case where no
/// linear constraints are used.
template <class DomainT, class ResponseT>
class OptProblemInterface_LinearConstraints<DomainT,ResponseT,false> : virtual public OptProblemFragment_Base<DomainT,ResponseT>
{
public:

  /// Returns an error.
  OptProblemFragment_LinearConstraints<DomainT,ResponseT>& domain_linearconstraints()
                {
                EXCEPTION_MNGR(std::runtime_error,"Error attempting to access the LinearConstraints domain information.");
                return *static_cast<OptProblemFragment_LinearConstraints<DomainT,ResponseT>*>(0);
                }

};


}
#endif
