/*  _________________________________________________________________________
 *
 *  COLIN: A Common Optimization Library INterface
 *  Copyright (c) 2003, Sandia National Laboratories.
 *  This software is distributed under the GNU Lesser General Public License.
 *  For more information, see the README.html file in the top COLIN directory.
 *  _________________________________________________________________________
 */

/**
 * \file OptProblemFrag_Base.h
 *
 * Defines the colin::OptProblemBaseFragment class.
 */

#ifndef colin_OptProblemFrag_Base_h
#define colin_OptProblemFrag_Base_h

#include <acro_config.h>
#include <utilib/std_headers.h>
#include <utilib/CommonIO.h>
#include <utilib/stl_auxillary.h>
#include <colin/OptApplication.h>
#include <colin/OptProblemAppHandle.h>

namespace colin {

using utilib::SmartPtr;


template <class DomainT, class ResponseT>
class OptProblem;


/** 
 * Defines core fragments of an colin::OptProblem object.  These items
 * are used by colin::OptProblem_BoundsFragment and other 'intermediate' classes 
 * to define the colin::OptProblem object.
 */
template <class DomainT, class ResponseT>
class OptProblemFragment_Base
{
public:

  /// Constructor
  OptProblemFragment_Base()
		: handle(0), app(0), state(0) {}

  /// Virtual destructor
  virtual ~OptProblemFragment_Base() {}

  /// The number of real parameters
  virtual unsigned int num_real_params()
		{return 0;}
  
  /// The number of integer parameters
  virtual unsigned int num_int_params()
		{return 0;}

  /// The number of binary parameters
  virtual unsigned int num_binary_params()
		{return 0;}

  /// The number of evaluation servers (defined by the colin::OptApplication)
  unsigned int num_evaluation_servers()
  		{ return app->num_evaluation_servers(); }

  /// The current point being evaluated
  void get_point(DomainT& point)
		{ point &= handle->get_point(); }

  /// The current point being evaluated
  DomainT& get_point()
		{ return handle->get_point(); }

  /// A smart pointer to a colin::OptProblemAppHandle object
  SmartPtr< OptProblemAppHandle<DomainT,ResponseT> > handle;

  /// Get a pointer to the base of the current application
  OptApplicationBase* get_application()
	  	{ return app; }

  /// The number of objectives
  unsigned int numObjectives() const
		{return state->numFunctions;}

  /**@name Constraint Information */
  //@{
  /// The number of nonlinear equality constraints.
  unsigned int numNonlinearEqConstraints() const
		{return state->numNonlinearEqConstraints;}

  /// The number of nonlinear inequality constraints.
  unsigned int numNonlinearIneqConstraints() const
		{return state->numNonlinearIneqConstraints;}

  /// The number of equality constraints.
  unsigned int numEqConstraints() const
		{return state->numNonlinearEqConstraints + 
			state->numLinearEqConstraints;}

  /// The number of inequality constraints.
  unsigned int numIneqConstraints() const
		{return state->numNonlinearIneqConstraints +
			state->numLinearIneqConstraints;}

  /// The number of nonlinear constraints.
  unsigned int numNonlinearConstraints() const
		{return state->numNonlinearIneqConstraints +
			state->numNonlinearEqConstraints;}

  /// The total number of constraints (except for bound constraints).
  unsigned int numConstraints() const
		{return state->numConstraints;}
  
  /// Set constraint bounds - dense format
#ifdef ACRO_HAVE_TEMPLATES_AS_TEMPLATE_ARGUMENTS
  template <template <class TYPE> class LArrayT>
  void set_constraint_bounds(const LArrayT<real>& lower, const LArrayT<real>& upper)
#else
  template <class ArrayT>
  void set_constraint_bounds(const ArrayT& lower, const ArrayT& upper)
#endif
		{
		std::vector<real> lower_,upper_;
		lower_ << lower;
		upper_ << upper;
		state->set_constraint_bounds(lower_,upper_);
		}

  /// Set constraint bounds - dense format
#ifdef ACRO_HAVE_TEMPLATES_AS_TEMPLATE_ARGUMENTS
  template <template <class TYPE> class LArrayT>
  void set_constraint_bounds(const LArrayT<double>& lower, const LArrayT<double>& upper)
		{
		std::vector<real> lower_(lower.size()),upper_(lower.size());
		for (size_type i=0; i<lower.size(); i++) {
		  lower_[i]=lower[i]; upper_[i]=upper[i]; }
		state->set_constraint_bounds(lower_,upper_);
		}
#endif

  /// Get constraint bounds - dense format
#ifdef ACRO_HAVE_TEMPLATES_AS_TEMPLATE_ARGUMENTS
  template <template <class TYPE> class LArrayT>
  void get_constraint_bounds(LArrayT<real>& lower, LArrayT<real>& upper)
#else
  template <class ArrayT>
  void get_constraint_bounds(ArrayT& lower, ArrayT& upper)
#endif
		{
		std::vector<real> lower_,upper_;
		state->get_constraint_bounds(lower_,upper_);
		lower << lower_;
		upper << upper_;
		}

  /// Get constraint bounds - dense format
#ifdef ACRO_HAVE_TEMPLATES_AS_TEMPLATE_ARGUMENTS
  template <template <class TYPE> class LArrayT>
  void get_constraint_bounds(LArrayT<double>& lower, LArrayT<double>& upper)
		{
		std::vector<real> lower_,upper_;
		state->get_constraint_bounds(lower_,upper_);
		lower.resize(lower_.size());
		upper.resize(upper_.size());
		for (size_type i=0; i<lower.size(); i++) {
		  lower[i]=lower_[i]; upper[i]=upper_[i]; }
		}
#endif

  //}

  /// This is the same as handle->app(), but we store it for efficiency 
  OptApplicationBase* app;

  /// This is the same as handle->state(), but we store it for efficiency 
  OptProblemStateBase* state;

protected:

  /// Initializes the state and app data 
  void init_appinfo()
		{
		app   = handle->app();
		state = handle->state();
		state->numFunctions = app->num_objectives;
		state->set_constraint_ctrs(0,0,app->num_eq_constr,
				app->num_ineq_constr);
		state->reset();
		}
};

}

#endif
