/*  _________________________________________________________________________
 *
 *  COLIN: A Common Optimization Library INterface
 *  Copyright (c) 2003, Sandia National Laboratories.
 *  This software is distributed under the GNU Lesser General Public License.
 *  For more information, see the README.html file in the top COLIN directory.
 *  _________________________________________________________________________
 */

/**
 * \file AppResponseValues.h
 *
 * Defines the colin::AppResponseValues class.
 */

#ifndef colin_AppResponseValues_h
#define colin_AppResponseValues_h

#include <acro_config.h>
#include <utilib/traits.h>
#include <colin/real.h>
#include <colin/AppResponseInfo.h>

namespace colin {

template <class TypesT>
class AppResponseValues;

}


///
/// Setting is_cache_allocated traits for AppResponseValues
/// This needs to be setup _outside_ of the colin namespace
///
namespace utilib {

/// A trait indicating if caching is allowed.
template <class TypesT>
struct is_cache_allocated < typename colin::AppResponseValues<TypesT> > {
   public:
   /// The value of this trait.
   static const bool value = true;
};

}

namespace colin {

/** \class AppResponseValues
  *
  * This class includes the results of an application evaluation, which may
  * include the values of one or more functions and the values of 
  * constraints.  Additionally, this class includes values for gradients and
  * Hessians for the functions and constraints.  Although these are not used
  * in all contexts, it convenient to define optimization problems using a 
  * container class for the optimization response values.
  */
template <class TypesT>
class AppResponseValues :
	public utilib::CachedAllocatorObject< AppResponseValues<TypesT> >
{
public:

  /// The constructor.
  AppResponseValues() 
	: num_real_params(0), num_functions(0), num_constraints(0)
	{
	l2_cviolationPtr = &l2_cviolation;
	functionValuePtr = 0;
	aug_functionValuePtr = 0;
	functionGradientPtr = 0;
	functionHessianPtr = 0;
	functionValuesPtr = &functionValues;
	aug_functionValuesPtr = &aug_functionValues;
	functionGradientsPtr = &functionGradients;
	functionHessiansPtr = &functionHessians;
	constraintValuesPtr = &constraintValues;
	constraintGradientsPtr = &constraintGradients;
	constraintHessiansPtr = &constraintHessians;
	}

  /// Destructor.
  virtual ~AppResponseValues() {}

  /// A method to resize the buffers in a AppResponseValues object.
  void resize(unsigned int num_functions, unsigned int num_constraints,
			unsigned int num_real_params,
			bool using_gradients=false, bool using_hessians=false);

  /// Resize the gradient buffers
  void resize_gradients();

  /// Resize the Hessian buffers
  void resize_hessians();

  /// Copy information from another AppResponseValues object.
  void copy(const AppResponseValues<TypesT>& info)
 	{
	l2_cviolation = info.l2_cviolation;
	functionValues << info.functionValues;
	aug_functionValues << info.aug_functionValues;
	functionGradients << info.functionGradients;
	functionHessians << info.functionHessians;
	constraintValues << info.constraintValues;
	constraintGradients << info.constraintGradients;
	constraintHessians << info.constraintHessians;
	num_real_params = info.num_real_params;
	num_functions = info.num_functions;
	num_constraints = info.num_constraints;
	//aug_functionValuePtr = info.aug_functionValuePtr;
	//functionGradientPtr = info.functionGradientPtr;
	//functionHessianPtr = info.functionHessianPtr;
	}

  /// The L2 constraint violation
  real l2_cviolation;

  /// The set of function values
  typename TypesT::realarray_t	functionValues;

  /// The set of augmented function values
  typename TypesT::realarray_t	aug_functionValues;

  /// The set of function gradients
  typename TypesT::vectorarray_t	functionGradients;

  /// The set of function Hessians
  typename TypesT::matrixarray_t	functionHessians;

  /// The set of constraint values
  typename TypesT::vector_t	constraintValues;

  /// The set of constraint gradients
  typename TypesT::vectorarray_t	constraintGradients;

  /// The set of constraint Hessians
  typename TypesT::matrixarray_t	constraintHessians;
 
  /// A pointer to l2 constraint violation info
  real*	l2_cviolationPtr;

  /// A pointer to function value info
  real*	functionValuePtr;

  /// A pointer to augmented function value info
  real*	aug_functionValuePtr;

  /// A pointer to gradient value info
  typename TypesT::vector_t* functionGradientPtr;

  /// A pointer to Hessian value info
  typename TypesT::matrix_t* functionHessianPtr;

  /// A pointer to an array of function values
  typename TypesT::realarray_t*	functionValuesPtr;

  /// A pointer to an array of augmented function values
  typename TypesT::realarray_t*	aug_functionValuesPtr;

  /// A pointer to an array of gradient values
  typename TypesT::vectorarray_t* functionGradientsPtr;

  /// A pointer to an array of Hessian values
  typename TypesT::matrixarray_t* functionHessiansPtr;

  /// A pointer to an array of constraint values
  typename TypesT::vector_t*	constraintValuesPtr;

  /// A pointer to an array of constraint gradient values
  typename TypesT::vectorarray_t*	constraintGradientsPtr;

  /// A pointer to an array of constraint Hessian values
  typename TypesT::matrixarray_t*	constraintHessiansPtr;
 
  /// The number of real-valued parameters
  unsigned int num_real_params;

  /// The number of functions
  unsigned int num_functions;

  /// The number of constraints
  unsigned int num_constraints;

  /// Initialize with colin::AppResponseInfo data.
  virtual void init(AppResponseInfo* ) {}

  /// Deallocate with cached information.
  void deallocate_derived()
        { utilib::CachedAllocator< AppResponseValues<TypesT> > :: deallocate_derived(this); }

};

template <class TypesT>
void AppResponseValues<TypesT>::resize(
			unsigned int num_functions_, 
			unsigned int num_constraints_,
			unsigned int num_real_params_, 
			bool using_gradients,
			bool using_hessians)
{
if (num_functions != num_functions_) {
   num_functions = num_functions_;
   functionValues.resize(num_functions);
   aug_functionValues.resize(num_functions);
   }
if (num_constraints != num_constraints_) {
   num_constraints = num_constraints_;
   constraintValues.resize(num_constraints);
   }
if (num_real_params != num_real_params_)
   num_real_params = num_real_params_;

if (using_gradients)
   resize_gradients();
if (using_hessians)
   resize_hessians();
}


template <class TypesT>
void AppResponseValues<TypesT>::resize_gradients()
{
functionGradients.resize(num_functions);
for (unsigned int i=0; i<num_functions; i++)
  functionGradients[i].resize(num_real_params);

constraintGradients.resize(num_constraints);
for (unsigned int i=0; i<num_constraints; i++)
  constraintGradients[i].resize(num_real_params);
}



template <class TypesT>
void AppResponseValues<TypesT>::resize_hessians()
{
#if 0
//// TODO
functionHessians.resize(num_functions);
for (unsigned int i=0; i<num_functions; i++)
  functionHessians[i].resize(num_constraints,num_real_params);

constraintHessians.resize(num_constraints);
for (unsigned int i=0; i<num_constraints; i++)
  constraintHessians[i].resize(num_constraints,num_real_params);
#endif
}




/// Copy AppResponseValues information from one objec to another
/// using an AppResponseInfo object to guide the copy process.
template <class TypesT, class LTypesT>
void map_AppResponseValues(AppResponseValues<TypesT>* to,
			AppResponseValues<LTypesT>* from,
			AppResponseInfo* info)
{
if (info->mode & mode_f) {
   if (from->functionValuePtr && to->functionValuePtr) {
      if (info->responseASV[0] | mode_f)
         *(to->functionValuePtr) = *(from->functionValuePtr);
      }

   else if (from->functionValuePtr) {
      if (info->responseASV[0] | mode_f)
         (*(to->functionValuesPtr))[0] = *(from->functionValuePtr);
      }

   else if (to->functionValuePtr) {
      if (info->responseASV[0] | mode_f)
         (*(to->functionValuePtr)) = (*(from->functionValuesPtr))[0];
      }

   else {
      for (unsigned int i=0; i<info->num_functions; i++)
        if (info->responseASV[i] | mode_f)
           (*(to->functionValuesPtr))[i] = (*(from->functionValuesPtr))[i];
      }
   }

if (info->mode & mode_f) {
   if (from->aug_functionValuePtr && to->aug_functionValuePtr) {
      if (info->responseASV[0] | mode_f)
         *(to->aug_functionValuePtr) = *(from->aug_functionValuePtr);
      }

   else if (from->aug_functionValuePtr) {
      if (info->responseASV[0] | mode_f)
         (*(to->aug_functionValuesPtr))[0] = *(from->aug_functionValuePtr);
      }

   else if (to->aug_functionValuePtr) {
      if (info->responseASV[0] | mode_f)
         (*(to->aug_functionValuePtr)) = (*(from->aug_functionValuesPtr))[0];
      }

   else {
      for (unsigned int i=0; i<info->num_functions; i++)
        if (info->responseASV[i] | mode_f)
           (*(to->aug_functionValuesPtr))[i] = (*(from->aug_functionValuesPtr))[i];
      }
   }

if (info->mode & mode_g) {
   if (from->functionGradientPtr && to->functionGradientPtr) {
      if (info->responseASV[0] | mode_g)
         *(to->functionGradientPtr) << *(from->functionGradientPtr);
      }

   if (from->functionGradientPtr) {
      if (info->responseASV[0] | mode_g)
         (*(to->functionGradientsPtr))[0] << *(from->functionGradientPtr);
      }

   if (to->functionGradientPtr) {
      if (info->responseASV[0] | mode_g)
         (*(to->functionGradientPtr)) << (*(from->functionGradientsPtr))[0];
      }

   else 
      map_jacobian(*(to->functionGradientsPtr),*(from->functionGradientsPtr));
   }

if (info->mode & mode_h) {
   if (from->functionHessianPtr && to->functionHessianPtr) {
      if (info->responseASV[0] | mode_h)
         map_matrix(*(to->functionHessianPtr),*(from->functionHessianPtr));
      }

   if (from->functionHessianPtr) {
      if (info->responseASV[0] | mode_h)
         map_matrix((*(to->functionHessianPtr))[0],*(from->functionHessianPtr));
      }

   if (to->functionHessianPtr) {
      if (info->responseASV[0] | mode_h)
         map_matrix(*(to->functionHessianPtr),(*(from->functionHessianPtr))[0]);
      }

   else {
      for (unsigned int i=0; i<info->num_functions; i++) {
        if (info->responseASV[i] | mode_h)
           map_matrix((*(to->functionHessiansPtr))[i],(*(from->functionHessiansPtr))[i]);
        }
      }
   }

if (info->mode & mode_cf) {
   if (from->constraintValuesPtr && to->constraintValuesPtr) {
      *(to->constraintValuesPtr) << *(from->constraintValuesPtr);
      *(to->l2_cviolationPtr) = *(from->l2_cviolationPtr);
      }
   }

if (info->mode & mode_cg) {
   if (from->constraintGradientsPtr && to->constraintGradientsPtr)
      map_jacobian(*(to->constraintGradientsPtr),*(from->constraintGradientsPtr));
   }

if (info->mode & mode_ch) {
   if (from->constraintHessiansPtr && to->constraintHessiansPtr) {
      for (unsigned int i=0; i<info->num_constraints; i++)
        map_matrix((*(to->constraintHessiansPtr))[i],(*(from->constraintHessiansPtr))[i]);
      }
   }
}

} // namespace colin


/// Copy operator.
template <class TypesT>
inline colin::AppResponseValues<TypesT>& operator<<( colin::AppResponseValues<TypesT>& x, const  colin::AppResponseValues<TypesT>& info)
{ x.copy(info); return x; }

#endif
