/*
================================================================================
    PROJECT:

        John Eddy's Genetic Algorithms (JEGA)

    CONTENTS:

        Implementation of class Design.

    NOTES:

        See notes of Design.hpp.

    PROGRAMMERS:

        John Eddy (jpeddy@sandia.gov) (JE)

    ORGANIZATION:

        Sandia National Laboratories

    COPYRIGHT:

        See the LICENSE file in the top level JEGA directory.

    VERSION:

        1.0.0

    CHANGES:

        Thu May 15 11:04:05 2003 - Original Version (JE)

================================================================================
*/




/*
================================================================================
Document This File
================================================================================
*/
/** \file
 * \brief Contains the implementation of the Design class.
 */




/*
================================================================================
Includes
================================================================================
*/
// JEGAConfig.hpp should be the first include in all JEGA files.
#include <../Utilities/include/JEGAConfig.hpp>

#include <algorithm>
#include <../Utilities/include/Design.hpp>
#include <../Utilities/include/Logging.hpp>
#include <../Utilities/include/DesignTarget.hpp>
#include <utilities/include/EDDY_DebugScope.hpp>
#include <../Utilities/include/DesignVariableInfo.hpp>



/*
================================================================================
Namespace Using Directives
================================================================================
*/
using namespace std;
using namespace JEGA::Logging;




/*
================================================================================
Begin Namespace
================================================================================
*/
namespace JEGA {
    namespace Utilities {



/*
================================================================================
Static Member Data Definitions
================================================================================
*/








/*
================================================================================
Mutators
================================================================================
*/
void
Design::SetObjective(
    std::size_t num,
    double val
    )
{
    EDDY_FUNC_DEBUGSCOPE
    EDDY_ASSERT(this->_objectives != 0x0)
    EDDY_ASSERT(num < this->_target.GetNOF())

    JEGAIFLOG_CF_II_G_F(this->_objectives == 0x0, this,
        ostream_entry(lfatal(), "Design: Attempt to set objective at index ")
            << num << " but the objectives array has not yet been allocated."
        )

    JEGAIFLOG_CF_II_G_F(num >= this->_target.GetNOF(), this,
        ostream_entry(lfatal(), "Design: Attempt to set objective at index ")
            << num << " which is beyond the last index of "
            << (this->_target.GetNOF()-1)
        )

    this->_objectives[num] = val;
}

void
Design::SetConstraint(
    std::size_t num,
    double val
    )
{
    EDDY_FUNC_DEBUGSCOPE
    EDDY_ASSERT(this->_constraints != 0x0)
    EDDY_ASSERT(num < this->_target.GetNCN())

    JEGAIFLOG_CF_II_G_F(this->_constraints == 0x0, this,
        ostream_entry(lfatal(), "Design: Attempt to set constraint at index ")
            << num << " but the constraints array has not yet been allocated."
        )

    JEGAIFLOG_CF_II_G_F(num >= this->_target.GetNCN(), this,
        ostream_entry(lfatal(), "Design: Attempt to set constraint at index ")
            << num << " which is beyond the last index of "
            << (this->_target.GetNCN()-1)
        )

    this->_constraints[num] = val;
}

void
Design::SetVariableRep(
    std::size_t num,
    double rep
    )
{
    EDDY_FUNC_DEBUGSCOPE
    EDDY_ASSERT(this->_variables != 0x0)
    EDDY_ASSERT(num < this->_target.GetNDV())

    JEGAIFLOG_CF_II_G_F(this->_variables == 0x0, this,
        ostream_entry(lfatal(), "Design: Attempt to set variable at index ")
            << num << " but the variables array has not yet been allocated."
        )

    JEGAIFLOG_CF_II_G_F(num >= this->_target.GetNDV(), this,
        ostream_entry(lfatal(), "Design: Attempt to set variable at index ")
            << num << " which is beyond the last index of "
            << (this->_target.GetNDV()-1)
        )

    this->_variables[num] = rep;
}








/*
================================================================================
Accessors
================================================================================
*/
double
Design::GetVariableRep(
    std::size_t num
    ) const
{
    EDDY_FUNC_DEBUGSCOPE
    EDDY_ASSERT(num < this->_target.GetNDV())

    JEGAIFLOG_CF_II_G_F(num >= this->_target.GetNDV(), this,
        ostream_entry(lfatal(), "Design: Attempt to get variable at index ")
            << num << " which is beyond the last index of "
            << (this->_target.GetNDV()-1)
        )

    return this->_variables[num];
}

double
Design::GetObjective(
    std::size_t num
    ) const
{
    EDDY_FUNC_DEBUGSCOPE
    EDDY_ASSERT(num < this->_target.GetNOF())

    JEGAIFLOG_CF_II_G_F(num >= this->_target.GetNOF(), this,
        ostream_entry(lfatal(), "Design: Attempt to get objective at index ")
            << num << " which is beyond the last index of "
            << (this->_target.GetNOF()-1)
        )

    return this->_objectives[num];
}

double
Design::GetConstraint(
    std::size_t num
    ) const
{
    EDDY_FUNC_DEBUGSCOPE
    EDDY_ASSERT(num < this->_target.GetNCN())

    JEGAIFLOG_CF_II_G_F(num >= this->_target.GetNCN(), this,
        ostream_entry(lfatal(), "Design: Attempt to get constraint at index ")
            << num << " which is beyond the last index of "
            << (this->_target.GetNCN()-1)
        )

    return this->_constraints[num];
}









/*
================================================================================
Public Methods
================================================================================
*/
const Design&
Design::operator = (
    const Design& right
    )
{
    EDDY_FUNC_DEBUGSCOPE

    if(&right == this) return *this;

    // The targets must be the same for two Designs to equate
    EDDY_ASSERT(&this->_target == &right._target);

    // Allocation should already have taken place so
    // equate this to rhs.

    // equate the variables and responses.
    ::memcpy(
        this->_variables, right._variables,
        this->_target.GetNDV() * sizeof(double)
        );
    ::memcpy(
        this->_objectives, right._objectives,
        this->_target.GetNOF() * sizeof(double)
        );
    ::memcpy(
        this->_constraints, right._constraints,
        this->_target.GetNCN() * sizeof(double)
        );

    // equate the attributes.
    this->_attributes = right._attributes;

    // Tag the two Designs as clones.
    Design::TagAsClones(*this, const_cast<Design&>(right));

    return *this;
}

eddy::utilities::uint8_t
Design::TagAsClones(
    Design& des1,
    Design& des2
    )
{
    EDDY_FUNC_DEBUGSCOPE
    EDDY_ASSERT(&des1 != &des2);

    // Check for trivial abort conditions.
    if(&des1 == &des2) return 0;

    // These pointers will be used to iterate the clone lists.
    Design* pdes1 = &des1;
    Design* pdes2 = &des2;

    if(des1.IsCloned())
    {
        if(des2.IsCloned())
        {
            // both are clones already, check to see
            // if they are already known to be clones
            // of one another.  If not, tag'em.
            if(!des1.HasInCloneList(des2))
            {
                // find beginning of des1's list
                for(; pdes1->_pClone!=0x0; pdes1=pdes1->_pClone);
                // find end of des2's list
                for(; pdes2->_nClone!=0x0; pdes2=pdes2->_nClone);
                //  merge the lists.
                pdes2->_nClone = pdes1;
                pdes1->_pClone = pdes2;
            }

            return 0;
        }
        else // if(!des2.IsCloned())
        {
            // find the end of des1's clone list.
            for(; pdes1->_nClone!=0x0; pdes1=pdes1->_nClone);
            // put des2 at the end.
            pdes1->_nClone = pdes2;
            pdes2->_pClone = pdes1;
            return 1;
        }
    }

    if(des2.IsCloned())
    {
        // find the end of des2's list.
        for(; pdes2->_nClone!=0x0; pdes2=pdes2->_nClone);
        // put des1 at the end.
        pdes2->_nClone = pdes1;
        pdes1->_pClone = pdes2;
        return 1;
    }

    // neither of these guys were clones before, start a new list.
    des1._nClone = pdes2;
    des2._pClone = pdes1;
    return 2;
}

void
Design::RemoveAsClone(
    )
{
    EDDY_FUNC_DEBUGSCOPE

    // typical linked list removal.
    if(this->_pClone != 0x0)
    {
        if(this->_nClone != 0x0) this->_nClone->_pClone = this->_pClone;
        this->_pClone->_nClone = this->_nClone;
    }
    else if(this->_nClone != 0x0) this->_nClone->_pClone = 0x0;

    this->_pClone = this->_nClone = 0x0;
}

bool
Design::HasInCloneList(
    const Design& des
    ) const
{
    EDDY_FUNC_DEBUGSCOPE

    for(const Design* cdes = this->_pClone; cdes!=0x0; cdes=cdes->_pClone)
        if(cdes == &des) return true;

    for(const Design* cdes = this->_nClone; cdes!=0x0; cdes=cdes->_nClone)
        if(cdes == &des) return true;

    return false;
}

eddy::utilities::uint64_t
Design::CountClones(
    ) const
{
    EDDY_FUNC_DEBUGSCOPE
    eddy::utilities::uint64_t ret = 0;

    // count the nclones.
    for(const Design* clone=this->_nClone; clone!=0x0; clone=clone->_nClone)
        ++ret;

    // now the pclones
    for(const Design* clone=this->_pClone; clone!=0x0; clone=clone->_pClone)
        ++ret;

    return ret;
}

void
Design::SetVariableValue(
    size_t num,
    double val
    )
{
    EDDY_FUNC_DEBUGSCOPE
    EDDY_ASSERT(num < this->_target.GetNDV())

    JEGAIFLOG_CF_II_G_F(num >= this->_target.GetNDV(), this,
        ostream_entry(lfatal(), "Design: Attempt to set variable at index ")
            << num << " which is beyond the last index of "
            << (this->_target.GetNDV()-1)
        )
        
    DesignVariableInfoVector::size_type cnum =
        static_cast<DesignVariableInfoVector::size_type>(num);
    const DesignVariableInfoVector& dvinfos = 
        this->_target.GetDesignVariableInfos();
    this->_variables[num] = dvinfos.at(cnum)->GetDoubleRepOf(val);
}

double
Design::GetVariableValue(
    size_t num
    ) const
{
    EDDY_FUNC_DEBUGSCOPE
    EDDY_ASSERT(num < this->_target.GetNDV())
    
    JEGAIFLOG_CF_II_G_F(num >= this->_target.GetNDV(), this,
        ostream_entry(lfatal(), "Design: Attempt to get variable at index ")
            << num << " which is beyond the last index of "
            << (this->_target.GetNDV()-1)
        )

    DesignVariableInfoVector::size_type cnum =
        static_cast<DesignVariableInfoVector::size_type>(num);
    const DesignVariableInfoVector& dvinfos = 
        this->_target.GetDesignVariableInfos();
    return dvinfos.at(cnum)->GetValueOf(this->_variables[num]);
}

size_t
Design::GetNDV(
    ) const
{
    EDDY_FUNC_DEBUGSCOPE
    return this->_target.GetNDV();

} // Design::GetNDV

size_t
Design::GetNOF(
    ) const
{
    EDDY_FUNC_DEBUGSCOPE
    return this->_target.GetNOF();

} // Design::GetNOF

size_t
Design::GetNCN(
    ) const
{
    EDDY_FUNC_DEBUGSCOPE
    return this->_target.GetNCN();

} // Design::GetNCN






/*
================================================================================
Subclass Visible Methods
================================================================================
*/








/*
================================================================================
Subclass Overridable Methods
================================================================================
*/








/*
================================================================================
Private Methods
================================================================================
*/


void
Design::DoAllocation(
    )
{
    EDDY_FUNC_DEBUGSCOPE

    size_t ndv = this->_target.GetNDV();
    size_t nof = this->_target.GetNOF();
    size_t ncn = this->_target.GetNCN();

    if(ndv > 0) this->_variables = new double[ndv];
    if(nof > 0) this->_objectives = new double[nof];
    if(ncn > 0) this->_constraints = new double[ncn];
}





/*
================================================================================
Structors
================================================================================
*/
Design::Design(
    DesignTarget& target
    ) :
        _variables(0x0),
        _objectives(0x0),
        _constraints(0x0),
        _attributes(0),
        _target(target),
        _nClone(0x0),
        _pClone(0x0)
{
    EDDY_FUNC_DEBUGSCOPE
    this->DoAllocation();
}

Design::Design(
    const Design& copy
    ) :
        _variables(0x0),
        _objectives(0x0),
        _constraints(0x0),
        _attributes(0),
        _target(copy._target),
        _nClone(0x0),
        _pClone(0x0)
{
    EDDY_FUNC_DEBUGSCOPE
    this->DoAllocation();
    this->operator =(copy);
}

Design::~Design(
    )
{
    EDDY_FUNC_DEBUGSCOPE
    this->RemoveAsClone();
    delete [] this->_variables;
    delete [] this->_objectives;
    delete [] this->_constraints;

    this->_variables = 0x0;
    this->_objectives = 0x0;
    this->_constraints = 0x0;
}







/*
================================================================================
End Namespace
================================================================================
*/
    } // namespace Utilities
} // namespace JEGA
