/*  _______________________________________________________________________

    DAKOTA: Design Analysis Kit for Optimization and Terascale Applications
    Copyright (c) 2006, Sandia National Laboratories.
    This software is distributed under the GNU General Public License.
    For more information, see the README file in the top Dakota directory.
    _______________________________________________________________________ */

//- Class:       PRPCache (not quite a full-fledged class just yet)
//- Description: Common API for lookup of ParamResponsePairs in evaluation cache
//- Checked by:
//- Version: $Id$

#ifndef PRP_MULTI_INDEX_H
#define PRP_MULTI_INDEX_H

#ifdef HAVE_CONFIG_H
#include "dakota_config.h"
#endif
#include "system_defs.h"
#include "data_types.h"
#include "DakotaList.H"
#include "ParamResponsePair.H"

#ifdef HAVE_BOOST
#include "boost/multi_index_container.hpp"
#include "boost/multi_index/hashed_index.hpp"
#include "boost/multi_index/mem_fun.hpp"
#include "boost/multi_index/ordered_index.hpp"
//#include "boost/multi_index/random_access_index.hpp"
namespace bmi = boost::multi_index;
#endif // HAVE_BOOST

namespace Dakota {


// A common interface for lookup functions supporting different
// implementations of Dakota caches and queues (Boost.MultiIndex if
// available, otherwise Dakota::List)


// --------------------------------------------------------
// Comparison functions shared by PRPCache/PRPQueue/PRPList
// --------------------------------------------------------

/// search function for a particular ParamResponsePair within a PRPList based
/// on ActiveSet content (request vector and derivative variables vector)

/** a global function to compare the ActiveSet of a particular database_pr
    (presumed to be in the global history list) with a passed in ActiveSet
    (search_set). */
inline bool 
set_compare(const ParamResponsePair& database_pr, const ActiveSet& search_set)
{
  // Check each entry of ASV for presence of all requests in the stored data.
  // A match is now detected when the search_asv is a SUBSET of the stored_asv
  // (only exact matches were interpreted as duplicates previously).  This 
  // extension is widely applicable, but was first implemented to eliminate 
  // duplication in Model::estimate_derivatives() when the gradient evaluation 
  // contains inactive fns. whereas a previous line search evaluation at the 
  // same X had no inactive fns.
  const ActiveSet&  stored_set = database_pr.active_set();
  const ShortArray& stored_asv = stored_set.request_vector();
  const ShortArray& search_asv = search_set.request_vector();
  size_t i, asv_len = search_asv.length();
  if ( stored_asv.length() != asv_len )
    return false;
  bool deriv_flag = false;
  for (i=0; i<asv_len; i++) {
    short search_bits = search_asv[i];
    if (search_bits & 6)
      deriv_flag = true;
    // bit-wise AND used to check if each of the search bits is
    // present in the stored_asv value
    if ( (stored_asv[i] & search_bits) != search_bits )
      return false;
  }

  // check if each of the search derivative variables is present in stored_dvv
  if (deriv_flag) {
    const UIntArray& stored_dvv = stored_set.derivative_vector();
    const UIntArray& search_dvv = search_set.derivative_vector();
    size_t dvv_len = search_dvv.length();
    for (i=0; i<dvv_len; i++)
      if ( !stored_dvv.contains(search_dvv[i]) )
	return false;
  }

  return true;
}


#ifdef HAVE_BOOST
// ------------------------------------------------------
// Comparison and hashing functions for PRPCache/PRPQueue
// ------------------------------------------------------

/// search function for a particular ParamResponsePair within a PRPMultiIndex

/** a global function to compare the interface id and variables of a
    particular database_pr (presumed to be in the global history list) with
    a passed in key of interface id and variables provided by search_pr. */
inline bool id_vars_exact_compare(const ParamResponsePair& database_pr,
				  const ParamResponsePair& search_pr)
{
  // First check interface id strings.  If a different interface was used, then
  // we must assume that the results are not interchangeable (differing model
  // fidelity).
  if ( search_pr.interface_id() != database_pr.interface_id() )
    return false;

  // For Boost hashing, need exact binary equality instead of the
  // tolerance-based operator== for Variables
  //if ( search_vars != stored_vars )
  if ( !binary_equal_to( search_pr.prp_parameters(),
			 database_pr.prp_parameters() ) )
    return false;

  // For Boost hashing, a post-processing step is used to manage the ActiveSet
  // logic as shown below in id_vars_set_compare

  return true;
}


/// hash_value for ParamResponsePairs stored in a PRPMultiIndex
inline std::size_t hash_value(const ParamResponsePair& prp)
{
  // hash using interface ID string.
  std::size_t seed = 0;
  boost::hash_combine(seed, prp.interface_id());

  // Now, hash values of variables using Variables hash_value friend function
  boost::hash_combine(seed, prp.prp_parameters());

  return seed;
}


// --------------------------------------
// structs and typedefs for PRPMultiIndex
// --------------------------------------
// define structs used below in PRPMultiIndex typedef

/// wrapper to delegate to the ParamResponsePair hash_value function
struct partial_prp_hash {
  /// access operator
  std::size_t operator()(const ParamResponsePair& prp) const
  { return hash_value(prp); } // ONLY idInterface & Vars used for hash_value
};

/// predicate for comparing ONLY the idInterface and Vars attributes of PRPair
struct partial_prp_equality {
  /// access operator
  bool operator()(const ParamResponsePair& database_pr,
                  const ParamResponsePair& search_pr) const
  { return id_vars_exact_compare(database_pr, search_pr); }
};


// tags
struct ordered {};
struct hashed  {};
//struct random  {};


/// Boost Multi-Index Container for caching ParamResponsePairs
typedef bmi::multi_index_container<Dakota::ParamResponsePair, bmi::indexed_by<
  // sorted beginning with lowest evalId/interfaceId value
  bmi::ordered_unique<bmi::tag<ordered>,
		      bmi::const_mem_fun<Dakota::ParamResponsePair,
		      const IntStringPair&,
		      &Dakota::ParamResponsePair::eval_interface_ids> >,
  // hashed using partial_prp_hash and compared using partial_prp_equality
  bmi::hashed_non_unique<bmi::tag<hashed>,
			 bmi::identity<Dakota::ParamResponsePair>,
                         partial_prp_hash, partial_prp_equality> > >
PRPMultiIndexCache;

typedef PRPMultiIndexCache PRPCache;
typedef PRPCache::index_iterator<ordered>::type       PRPCacheOIter;
typedef PRPCache::index_const_iterator<ordered>::type PRPCacheOCIter;
typedef PRPCache::index_iterator<hashed>::type        PRPCacheHIter;
typedef PRPCache::index_const_iterator<hashed>::type  PRPCacheHCIter;
typedef PRPCacheOIter  PRPCacheIter;  // default cache iterator <0>
typedef PRPCacheOCIter PRPCacheCIter; // default cache const iterator <0>

// begin()/end() default to index 0.  These macros support other indices.

/// hashed definition of cache begin
inline PRPCacheHIter hashedCacheBegin(PRPCache& prp_cache)
{ return prp_cache.get<hashed>().begin(); }
/// hashed definition of cache end
inline PRPCacheHIter hashedCacheEnd(PRPCache& prp_cache)
{ return prp_cache.get<hashed>().end(); }


/// Boost Multi-Index Container for queueing ParamResponsePairs
typedef bmi::multi_index_container<Dakota::ParamResponsePair, bmi::indexed_by<
  // random access for index-based operator[] access
  //bmi::random_access<bmi::tag<random> >,
  // sorted (by evalInterfaceIds) for fast key-based lookups
  bmi::ordered_unique<bmi::tag<ordered>,
		      bmi::const_mem_fun<Dakota::ParamResponsePair, int,
		      &Dakota::ParamResponsePair::eval_id> >,
  // hashed using partial_prp_hash and compared using partial_prp_equality
  bmi::hashed_non_unique<bmi::tag<hashed>,
			 bmi::identity<Dakota::ParamResponsePair>,
                         partial_prp_hash, partial_prp_equality> > >
PRPMultiIndexQueue;

typedef PRPMultiIndexQueue PRPQueue;
//typedef PRPQueue::index_iterator<random>::type        PRPQueueRIter;
//typedef PRPQueue::index_const_iterator<random>::type  PRPQueueRCIter;
typedef PRPQueue::index_iterator<ordered>::type       PRPQueueOIter;
typedef PRPQueue::index_const_iterator<ordered>::type PRPQueueOCIter;
typedef PRPQueue::index_iterator<hashed>::type        PRPQueueHIter;
typedef PRPQueue::index_const_iterator<hashed>::type  PRPQueueHCIter;
typedef PRPQueueOIter  PRPQueueIter;  // default queue iterator <0>
typedef PRPQueueOCIter PRPQueueCIter; // default queue const iterator <0>

// begin()/end() default to index 0.  These macros support other indices.

/// hashed definition of queue begin
inline PRPQueueHIter hashedQueueBegin(PRPQueue& prp_queue)
{ return prp_queue.get<hashed>().begin(); }
/// hashed definition of queue end
inline PRPQueueHIter hashedQueueEnd(PRPQueue& prp_queue)
{ return prp_queue.get<hashed>().end(); }


// ------------------------------------
// lookup_by_val for PRPMultiIndexCache
// ------------------------------------

/// find a ParamResponsePair based on the interface id, variables, and
/// ActiveSet search data within search_pr.

/** Lookup occurs in two steps: (1) PRPMultiIndexCache lookup based on
    strict equality in interface id and variables, and (2) set_compare()
    post-processing based on ActiveSet subset logic. */
inline PRPCacheHIter
lookup_by_val(PRPMultiIndexCache& prp_cache, const ParamResponsePair& search_pr)
{
  PRPCacheHIter prp_hash_it0, prp_hash_it1;
  boost::tuples::tie(prp_hash_it0, prp_hash_it1)
    = prp_cache.get<hashed>().equal_range(search_pr);

  // equal_range returns a small sequence of possibilities resulting from
  // hashing with ONLY interfaceId and variables.  Post-processing is then
  // applied to this sequence using set_compare().
  while (prp_hash_it0 != prp_hash_it1) {
    if (set_compare(*prp_hash_it0, search_pr.active_set()))
      return prp_hash_it0;
    ++prp_hash_it0;
  }
  return prp_cache.get<hashed>().end();
}


inline bool 
lookup_by_val(PRPMultiIndexCache& prp_cache, const ParamResponsePair& search_pr,
	      ParamResponsePair& found_pr)
{
  PRPCacheHIter prp_hash_it = lookup_by_val(prp_cache, search_pr);
  if (prp_hash_it != prp_cache.get<hashed>().end()) {
    found_pr = *prp_hash_it;
    return true;
  }
  else
    return false;
}


/// find the evaluation id of a ParamResponsePair within a PRPMultiIndexCache
/// based on interface id, variables, and ActiveSet search data
inline PRPCacheHIter
lookup_by_val(PRPMultiIndexCache& prp_cache, const String& search_interface_id,
	      const Variables& search_vars,  const ActiveSet& search_set)
{
  Response search_resp(search_set);
  ParamResponsePair search_pr(search_vars, search_interface_id, search_resp);
  return lookup_by_val(prp_cache, search_pr);
}


/// find a ParamResponsePair within a PRPMultiIndexCache based on interface id,
/// variables, and ActiveSet search data
inline bool 
lookup_by_val(PRPMultiIndexCache& prp_cache, const String& search_interface_id,
	      const Variables& search_vars, const ActiveSet& search_set,
	      ParamResponsePair& found_pr)
{
  PRPCacheHIter prp_hash_it
    = lookup_by_val(prp_cache, search_interface_id, search_vars, search_set);
  if (prp_hash_it != prp_cache.get<hashed>().end()) {
    found_pr = *prp_hash_it;
    return true;
  }
  else
    return false;
}


/// find the evaluation id of a ParamResponsePair within a PRPMultiIndexCache
/// based on interface id, variables, and ActiveSet search data
inline bool
lookup_by_val(PRPMultiIndexCache& prp_cache, const String& search_interface_id,
	      const Variables& search_vars, const ActiveSet& search_set,
	      int& found_eval_id)
{
  PRPCacheHIter prp_hash_it
    = lookup_by_val(prp_cache, search_interface_id, search_vars, search_set);
  if (prp_hash_it != prp_cache.get<hashed>().end()) {
    found_eval_id = prp_hash_it->eval_id();
    return true;
  }
  else
    return false;
}


/// find the response of a ParamResponsePair within a PRPMultiIndexCache
/// based on interface id, variables, and ActiveSet search data
inline bool 
lookup_by_val(PRPMultiIndexCache& prp_cache, const String& search_interface_id,
	      const Variables& search_vars, const ActiveSet& search_set,
	      Response& found_resp)
{
  PRPCacheHIter prp_hash_it
    = lookup_by_val(prp_cache, search_interface_id, search_vars, search_set);
  if (prp_hash_it != prp_cache.get<hashed>().end()) {
    found_resp = prp_hash_it->prp_response();
    return true;
  }
  else
    return false;
}


// ------------------------------------
// lookup_by_ids for PRPMultiIndexCache
// ------------------------------------
// find a ParamResponsePair within a PRPMultiIndexCache based on search_ids
// (i.e. std::pair<eval_id,interface_id>) search data
inline PRPCacheOIter
lookup_by_ids(PRPMultiIndexCache& prp_cache, const IntStringPair& search_ids)
{ return prp_cache.get<ordered>().find(search_ids); }


// find a ParamResponsePair within a PRPMultiIndexCache based on
// eval_interface_ids
inline bool
lookup_by_ids(PRPMultiIndexCache& prp_cache,
              const IntStringPair& search_eval_interface_ids,
              ParamResponsePair& found_pr)
{
  PRPCacheOIter prp_iter = lookup_by_ids(prp_cache, search_eval_interface_ids);
  if (prp_iter != prp_cache.get<ordered>().end()) {
    found_pr = *prp_iter;
    return true;
  }
  else
    return false;
}


// find a ParamResponsePair within a PRPMultiIndexCache based on
// eval_interface_ids from the ParamResponsePair search data
inline bool 
lookup_by_ids(PRPMultiIndexCache& prp_cache, const ParamResponsePair& search_pr,
              ParamResponsePair& found_pr)
{ return lookup_by_ids(prp_cache, search_pr.eval_interface_ids(), found_pr); }


// ------------------------------------
// lookup_by_val for PRPMultiIndexQueue
// ------------------------------------

/// find a ParamResponsePair based on the interface id, variables, and
/// ActiveSet search data within search_pr.

/** Lookup occurs in two steps: (1) PRPMultiIndexQueue lookup based on
    strict equality in interface id and variables, and (2) set_compare()
    post-processing based on ActiveSet subset logic. */
inline PRPQueueHIter
lookup_by_val(PRPMultiIndexQueue& prp_queue, const ParamResponsePair& search_pr)
{
  PRPQueueHIter prp_hash_it0, prp_hash_it1;
  boost::tuples::tie(prp_hash_it0, prp_hash_it1)
    = prp_queue.get<hashed>().equal_range(search_pr);

  // equal_range returns a small sequence of possibilities resulting from
  // hashing with ONLY interfaceId and variables.  Post-processing is then
  // applied to this sequence using set_compare().
  while (prp_hash_it0 != prp_hash_it1) {
    if (set_compare(*prp_hash_it0, search_pr.active_set()))
      return prp_hash_it0;
    ++prp_hash_it0;
  }
  return prp_queue.get<hashed>().end();
}


inline bool 
lookup_by_val(PRPMultiIndexQueue& prp_queue, const ParamResponsePair& search_pr,
	      ParamResponsePair& found_pr)
{
  PRPQueueHIter prp_hash_it = lookup_by_val(prp_queue, search_pr);
  if (prp_hash_it != prp_queue.get<hashed>().end()) {
    found_pr = *prp_hash_it;
    return true;
  }
  else
    return false;
}


/// find the evaluation id of a ParamResponsePair within a PRPMultiIndexQueue
/// based on interface id, variables, and ActiveSet search data
inline PRPQueueHIter
lookup_by_val(PRPMultiIndexQueue& prp_queue, const String& search_interface_id,
	      const Variables& search_vars,  const ActiveSet& search_set)
{
  Response search_resp(search_set);
  ParamResponsePair search_pr(search_vars, search_interface_id, search_resp);
  return lookup_by_val(prp_queue, search_pr);
}


/// find a ParamResponsePair within a PRPMultiIndexQueue based on interface id,
/// variables, and ActiveSet search data
inline bool 
lookup_by_val(PRPMultiIndexQueue& prp_queue, const String& search_interface_id,
	      const Variables& search_vars, const ActiveSet& search_set,
	      ParamResponsePair& found_pr)
{
  PRPQueueHIter prp_hash_it
    = lookup_by_val(prp_queue, search_interface_id, search_vars, search_set);
  if (prp_hash_it != prp_queue.get<hashed>().end()) {
    found_pr = *prp_hash_it;
    return true;
  }
  else
    return false;
}


/// find the evaluation id of a ParamResponsePair within a PRPMultiIndexQueue
/// based on interface id, variables, and ActiveSet search data
inline bool
lookup_by_val(PRPMultiIndexQueue& prp_queue, const String& search_interface_id,
	      const Variables& search_vars, const ActiveSet& search_set,
	      int& found_eval_id)
{
  PRPQueueHIter prp_hash_it
    = lookup_by_val(prp_queue, search_interface_id, search_vars, search_set);
  if (prp_hash_it != prp_queue.get<hashed>().end()) {
    found_eval_id = prp_hash_it->eval_id();
    return true;
  }
  else
    return false;
}


/// find the response of a ParamResponsePair within a PRPMultiIndexQueue
/// based on interface id, variables, and ActiveSet search data
inline bool 
lookup_by_val(PRPMultiIndexQueue& prp_queue, const String& search_interface_id,
	      const Variables& search_vars, const ActiveSet& search_set,
	      Response& found_resp)
{
  PRPQueueHIter prp_hash_it
    = lookup_by_val(prp_queue, search_interface_id, search_vars, search_set);
  if (prp_hash_it != prp_queue.get<hashed>().end()) {
    found_resp = prp_hash_it->prp_response();
    return true;
  }
  else
    return false;
}


// ----------------------------------------
// lookup_by_eval_id for PRPMultiIndexQueue
// ----------------------------------------
// find a ParamResponsePair within a PRPMultiIndexQueue based on search_id
// (i.e. integer eval_id) search data
inline PRPQueueOIter
lookup_by_eval_id(PRPMultiIndexQueue& prp_queue, const int& search_id)
{ return prp_queue.get<ordered>().find(search_id); }


// find a ParamResponsePair within a PRPMultiIndexQueue based on eval_id
inline bool
lookup_by_eval_id(PRPMultiIndexQueue& prp_queue, const int& search_id,
		  ParamResponsePair& found_pr)
{
  PRPQueueOIter prp_iter = lookup_by_eval_id(prp_queue, search_id);
  if (prp_iter != prp_queue.get<ordered>().end()) {
    found_pr = *prp_iter;
    return true;
  }
  else
    return false;
}


// find a ParamResponsePair within a PRPMultiIndexQueue based on
// eval_id from the ParamResponsePair search data
inline bool 
lookup_by_eval_id(PRPMultiIndexQueue& prp_queue,
		  const ParamResponsePair& search_pr,
		  ParamResponsePair& found_pr)
{ return lookup_by_eval_id(prp_queue, search_pr.eval_id(), found_pr); }


#else // not HAVE_BOOST


// ---------------------------------
// structs and typedefs for PRPLists
// ---------------------------------
typedef List<ParamResponsePair> PRPList;
typedef PRPList::iterator       PRPLIter;
typedef PRPList::const_iterator PRPLCIter;

typedef PRPList   PRPCache;
typedef PRPList   PRPQueue;
typedef PRPLIter  PRPCacheIter;
typedef PRPLCIter PRPCacheCIter;
typedef PRPLIter  PRPQueueIter;
typedef PRPLCIter PRPQueueCIter;
typedef PRPLIter  PRPQueueHIter;

// PRPList definition of hashed queue begin
inline PRPLIter hashedQueueBegin(PRPList& prp_list)
{ return prp_list.begin(); }
// PRPList definition of hashed queue end
inline PRPLIter hashedQueueEnd(PRPList& prp_list)
{ return prp_list.end(); }

// ---------------------------------
// Comparison functions for PRPLists
// ---------------------------------

/// alternate form of ActiveSet-based search function where search_set
/// is passed by void* pointer (as required by Dakota::List::find() API)

/** a global function to compare the ActiveSet of a particular database_pr
    (presumed to be in the global history list) with a passed in ActiveSet
    (search_ptr). */
inline bool set_compare_by_ptr(const ParamResponsePair& database_pr,
			       const void* search_ptr)
{
  const ActiveSet* search_set = (const ActiveSet*)search_ptr;
  return set_compare(database_pr, *search_set);
}


/// search function for a particular ParamResponsePair within a PRPList
/// based on interface id, variables, and ActiveSet

/** a global function to compare the interface id, variables, and ActiveSet
    of a particular database_pr (presumed to be in the global history list)
    with a passed in interface id, variables, and ActiveSet. */
inline bool 
id_vars_set_compare(const ParamResponsePair& database_pr,
		    const String& search_interface_id,
		    const Variables& search_vars, const ActiveSet& search_set)
{
  // First check interface id strings.  If a different interface was used, then
  // we must assume that the results are not interchangeable (differing model
  // fidelity).
  if ( search_interface_id != database_pr.interface_id() )
    return false;

  // Next, check equality of variables using tolerance-based equality operator.
  if ( search_vars != database_pr.prp_parameters() )
    return false;

  // Check ActiveSet based on subset logic.
  if ( !set_compare( database_pr, search_set ) )
    return false;

  return true;
}


/// alternate form of id/set/vars-based search function where
/// search_pr is passed by const reference

/** a global function to compare the interface id, variables, and
    ActiveSet of a particular database_pr (presumed to be in the
    global history list) with a passed in interface id, variables, and
    ActiveSet provided by search_pr. */
inline bool id_vars_set_compare(const ParamResponsePair& database_pr,
				const ParamResponsePair& search_pr)
{
  return id_vars_set_compare(database_pr, search_pr.interface_id(),
			     search_pr.prp_parameters(),
			     search_pr.active_set());
}


/// alternate form of id/set/vars-based search function where search_pr
/// is passed by void* pointer (as required by Dakota::List::find() API)

/** a global function to compare the interface id, variables, and ActiveSet, and
    interface id of a particular database_pr (presumed to be in the
    global history list) with a passed in interface id, variables, and
    ActiveSet provided by search_ptr. */
inline bool id_vars_set_compare_by_ptr(const ParamResponsePair& database_pr,
				       const void* search_ptr)
{
  const ParamResponsePair* search_pr = (const ParamResponsePair*)search_ptr;
  return id_vars_set_compare(database_pr, search_pr->interface_id(),
			     search_pr->prp_parameters(),
			     search_pr->active_set());
}


// search function for a particular ParamResponsePair within a PRPList
// based on evaluation id, where the id is passed by const reference

/* a global function to compare the evalId of a particular
   ParamResponsePair (from a container) with a passed in evaluation id. */
//inline bool eval_id_compare(const ParamResponsePair& pair, const int& id)
//{ return ( pair.eval_id() == id ) ? true : false; }


/// search function for a particular ParamResponsePair within a
/// PRPList/PRPMultiIndex based on evaluation id, where the id is
/// passed by void*

/** a global function to compare the evalId of a particular
    ParamResponsePair (from a container) with a passed in evaluation id. */
inline bool 
eval_id_compare_by_ptr(const ParamResponsePair& pair, const void* id)
{ return ( pair.eval_id() == *(const int*)id ) ? true : false; }


// search function for a particular ParamResponsePair within a
// PRPList based on evaluation and interface ids, where the ids are
// passed as an IntStringPair

/* a global function to compare the evalInterfaceIds of a particular
   ParamResponsePair (from a container) with a passed in IntStringPair. */
//inline bool 
//eval_interface_ids_compare(const ParamResponsePair& pair,
//			     const IntStringPair& ids)
//{ return ( pair.eval_interface_ids() == ids ); }


/// search function for a particular ParamResponsePair within a
/// PRPList based on evaluation id, where the id is passed by void*

/** a global function to compare the evalInterfaceIds of a particular
    ParamResponsePair (from a container) with a passed in IntStringPair. */
inline bool 
eval_interface_ids_compare_by_ptr(const ParamResponsePair& pair,
				  const void* ids)
{ return ( pair.eval_interface_ids() == *(const IntStringPair*)ids ); }


/// sort function for ParamResponsePair

/** a global function used to sort a PRPList by evalId's. */
inline bool eval_interface_ids_sort_fn(const ParamResponsePair& pr1,
				       const ParamResponsePair& pr2)
{
  return ( pr1.interface_id() < pr2.interface_id() ||
           ( pr1.interface_id() == pr2.interface_id() &&
             pr1.eval_id() < pr2.eval_id() ) ) ? true : false;
}


// --------------------------
// lookup_by_val for PRPLists
// --------------------------
/// find the iterator of a ParamResponsePair within a PRPList based on
/// interface id, variables, and ActiveSet search data
inline PRPLIter 
lookup_by_val(PRPList& prp_list, const String& search_interface_id,
	      const Variables& search_vars, const ActiveSet& search_set)
{
  // Note: could use List::find(id_vars_set_compare_by_ptr, &search_data) if
  //       id/vars/set are collected within a tuple and passed as void*

  PRPLIter it;
  for (it = prp_list.begin(); it != prp_list.end(); it++)
    if (id_vars_set_compare(*it, search_interface_id, search_vars, search_set))
      return it;
  return prp_list.end();
}


/// find the evaluation id of a ParamResponsePair within a PRPList
/// based on interface id, variables, and ActiveSet search data
inline bool
lookup_by_val(PRPList& prp_list, const String& search_interface_id,
	      const Variables& search_vars, const ActiveSet& search_set,
	      int& found_eval_id)
{
  PRPLIter it
    = lookup_by_val(prp_list, search_interface_id, search_vars, search_set);
  if (it != prp_list.end()) {
    found_eval_id = it->eval_id();
    return true;
  }
  return false;
}


/// find the response of a ParamResponsePair within a PRPList
/// based on interface id, variables, and ActiveSet search data
inline bool 
lookup_by_val(PRPList& prp_list, const String& search_interface_id,
	      const Variables& search_vars, const ActiveSet& search_set,
	      Response& found_resp)
{
  PRPLIter it
    = lookup_by_val(prp_list, search_interface_id, search_vars, search_set);
  if (it != prp_list.end()) {
    found_resp = it->prp_response();
    return true;
  }
  return false;
}


/// find a ParamResponsePair within a PRPList based on interface id,
/// variables, and ActiveSet search data
inline bool 
lookup_by_val(PRPList& prp_list, const String& search_interface_id,
	      const Variables& search_vars, const ActiveSet& search_set,
	      ParamResponsePair& found_pr)
{
  PRPLIter it
    = lookup_by_val(prp_list, search_interface_id, search_vars, search_set);
  if (it != prp_list.end()) {
    found_pr = *it;
    return true;
  }
  return false;
}


// ------------------------------
// lookup_by_eval_id for PRPLists
// ------------------------------
/// find the iterator of a ParamResponsePair within a PRPList based on
/// evaluation id search data
inline PRPLIter lookup_by_eval_id(PRPList& prp_list, const int& search_eval_id)
{ return prp_list.find(eval_id_compare_by_ptr, &search_eval_id); }


/// find the response of a ParamResponsePair within a PRPList based on
/// evaluation id search data
inline bool 
lookup_by_eval_id(PRPList& prp_list, const int& search_eval_id,
		  Response& found_resp)
{
  PRPLIter it = lookup_by_eval_id(prp_list, search_eval_id);
  if (it != prp_list.end()) {
    found_resp = it->prp_response();
    return true;
  }
  return false;
}


/// find a ParamResponsePair within a PRPList based on evaluation id
/// search data
inline bool 
lookup_by_eval_id(PRPList& prp_list, const int& search_eval_id,
		  ParamResponsePair& found_pr)
{
  PRPLIter it = lookup_by_eval_id(prp_list, search_eval_id);
  if (it != prp_list.end()) {
    found_pr = *it;
    return true;
  }
  return false;
}


// --------------------------
// lookup_by_ids for PRPLists
// --------------------------
// find the iterator of a ParamResponsePair within a PRPList based on
// eval_interface_ids search data
inline PRPLIter 
lookup_by_ids(PRPList& prp_list, const IntStringPair& search_eval_interface_ids)
{
  return prp_list.find(eval_interface_ids_compare_by_ptr,
		       &search_eval_interface_ids);
}


// find a ParamResponsePair within a PRPList based on eval_interface_ids
// search data
inline bool 
lookup_by_ids(PRPList& prp_list, const IntStringPair& search_eval_interface_ids,
	      ParamResponsePair& found_pr)
{
  PRPLIter it = lookup_by_ids(prp_list, search_eval_interface_ids);
  if (it != prp_list.end()) {
    found_pr = *it;
    return true;
  }
  return false;
}


// --------------------------
// lookup_by_set for PRPLists
// --------------------------
/// find the iterator of a ParamResponsePair within a PRPList based on
/// ActiveSet search data
inline PRPLIter lookup_by_set(PRPList& prp_list, const ActiveSet& search_set)
{ return prp_list.find(set_compare_by_ptr, &search_set); }


/// find a ParamResponsePair within a PRPList based on ActiveSet search data
inline bool 
lookup_by_set(PRPList& prp_list, const ActiveSet& search_set,
	      ParamResponsePair& found_pr)
{
  PRPLIter it = lookup_by_set(prp_list, search_set);
  if (it != prp_list.end()) {
    found_pr = *it;
    return true;
  }
  return false;
}

#endif // HAVE_BOOST

} // namespace Dakota

#endif // PRP_MULTI_INDEX_H

