/*  _______________________________________________________________________

    DAKOTA: Design Analysis Kit for Optimization and Terascale Applications
    Copyright (c) 2006, Sandia National Laboratories.
    This software is distributed under the GNU General Public License.
    For more information, see the README file in the top Dakota directory.
    _______________________________________________________________________ */

//- Class:	 NonDIntervalEst
//- Description: Class for interval bound estimation for epistemic UQ
//- Owner:	 Laura Swiler
//- Checked by:
//- Version:

#ifndef NOND_INTERVALEST_H
#define NOND_INTERVALEST_H

#include "DakotaNonD.H"
#include "data_types.h"

namespace Dakota {

/// Class for using optimization approaches to calculate interval bounds 
/// for epistemic uncertainty quantification

/** The NonDIntervalEst class ultimately will allow various optimization 
    options to determine interval bounds for epistemic UQ.  The 
    interval bounds may be on the entire function in the case of pure 
    interval analysis (e.g. intervals on input = intervals on output), or 
    the intervals may be on statistics of an "inner loop" aleatory 
    analysis such as intervals on means, variances, or percentile levels. 
    The preliminary implementation will use a Gaussian process surrogate 
    to determine interval bounds. */

class NonDIntervalEst: public NonD
{
public:

  //
  //- Heading: Constructors and destructor
  //

  NonDIntervalEst(Model& model); ///< constructor
  ~NonDIntervalEst();            ///< destructor

  //
  //- Heading: Member functions
  //

  /// Performs an optimization to determine interval bounds for an   
  /// entire function or interval bounds on a particular statistical estimator
  void quantify_uncertainty(); // pure virtual, called by run_iterator

  /// print the interval estimates
  void print_results(ostream& s);

protected: 

  /// initialize finalStatistics for interval results sets
  void initialize_final_statistics();

private:

  //
  //- Heading: Convenience functions
  //

  /// construct the GP using EGO and optimize it twice (for min and max)
  void nongradient_global_estimation();

  /// determine current best solution from among sample data for expected
  /// improvement function
  void get_best_sample(bool find_max, bool eval_approx);

  /// static function used as the objective function in the
  /// Expected Improvement Function (EIF) for minimizing the GP
  static void EIF_objective_min(const Variables& sub_model_vars,
				const Variables& recast_vars,
				const Response& sub_model_response,
				Response& recast_response);
  /// static function used as the objective function in the
  /// Expected Improvement Function (EIF) for maximizing the GP
  static void EIF_objective_max(const Variables& sub_model_vars,
				const Variables& recast_vars,
				const Response& sub_model_response,
				Response& recast_response);

  //
  //- Heading: Data members
  //

  /// pointer to the active object instance used within the static evaluator
  /// functions in order to avoid the need for static data
  static NonDIntervalEst* nondIntervalEstInstance;

  /// counter for the response function set
  size_t respFnCntr;

  /// LHS iterator for constructing initial GP for all response functions
  Iterator daceIterator;
  /// NCSU DIRECT optimizer for maximizing expected improvement
  Iterator gpOptimizer;

  /// GP model of response, one approximation per response function
  Model fHatModel;
  /// recast model which assimilates mean and variance to solve the
  /// max(EIF) sub-problem
  Model eifModel;

  /// approximate response corresponding to minimum/maximum truth response
  Real approxFnStar;
  /// minimum/maximum truth response function value
  Real truthFnStar;
};

} // namespace Dakota

#endif
