/*  _______________________________________________________________________

    DAKOTA: Design Analysis Kit for Optimization and Terascale Applications
    Copyright (c) 2006, Sandia National Laboratories.
    This software is distributed under the GNU General Public License.
    For more information, see the README file in the top Dakota directory.
    _______________________________________________________________________ */

//- Class:        MergedConstraints
//- Description:  Class implementation
//- Owner:        Mike Eldred

#include "MergedConstraints.H"
#include "ProblemDescDB.H"
#include "data_io.h"
#include "data_util.h"

static const char rcsId[]="@(#) $Id";


namespace Dakota {

/** In this class, a merged data approach is used in which continuous
    and discrete arrays are combined into a single continuous array
    (integrality is relaxed; the converse of truncating reals is not
    currently supported but could be in the future if needed).
    Iterators/strategies which use this class include:
    BranchBndOptimizer.  Extract fundamental lower and upper bounds
    and merge continuous and discrete domains to create
    mergedDesignLowerBnds, mergedDesignUpperBnds,
    mergedStateLowerBnds, and mergedStateUpperBnds. */
MergedConstraints::
MergedConstraints(const ProblemDescDB& problem_db,
		  const pair<short,short>& view,
		  const Sizet2DArray& vars_comps):
  Constraints(BaseConstructor(), problem_db, view, vars_comps)
{
  // Create merged arrays
  RealDenseVector mdv_l_bnds, mdv_u_bnds, msv_l_bnds, msv_u_bnds;
  aggregate_merged(
    problem_db.get_rdv("variables.continuous_design.lower_bounds"),
    problem_db.get_idv("variables.discrete_design.lower_bounds"), mdv_l_bnds);
  aggregate_merged(
    problem_db.get_rdv("variables.continuous_design.upper_bounds"),
    problem_db.get_idv("variables.discrete_design.upper_bounds"), mdv_u_bnds);
  aggregate_merged(
    problem_db.get_rdv("variables.continuous_state.lower_bounds"),
    problem_db.get_idv("variables.discrete_state.lower_bounds"), msv_l_bnds);
  aggregate_merged(
    problem_db.get_rdv("variables.continuous_state.upper_bounds"),
    problem_db.get_idv("variables.discrete_state.upper_bounds"), msv_u_bnds);

  // Aggregate merged arrays into all arrays
  aggregate_data(mdv_l_bnds,
    problem_db.get_rdv("variables.uncertain.lower_bounds"), msv_l_bnds,
    allContinuousLowerBnds);
  aggregate_data(mdv_u_bnds,
    problem_db.get_rdv("variables.uncertain.upper_bounds"), msv_u_bnds,
    allContinuousUpperBnds);

  // Construct active/inactive views of all arrays
  build_views();

  // Manage linear constraints.
  manage_linear_constraints(problem_db);

#ifdef REFCOUNT_DEBUG
  Cout << "Letter instantiated: variablesView active = " << variablesView.first
       << " inactive = " << variablesView.second << endl;
#endif
}


void MergedConstraints::copy_rep(const Constraints* con_rep)
{ build_views(); }


void MergedConstraints::reshape_rep()
{
  size_t num_acv = (*variablesComponents)[0][0] + (*variablesComponents)[1][0] +
    (*variablesComponents)[2][0] + (*variablesComponents)[3][0];

  allContinuousLowerBnds.resize(num_acv);
  allContinuousUpperBnds.resize(num_acv);

  build_views();
}


void MergedConstraints::build_active_views()
{
  size_t num_mdv = (*variablesComponents)[0][0],
    num_auv = (*variablesComponents)[1][0],
    num_euv = (*variablesComponents)[2][0],
    num_msv = (*variablesComponents)[3][0];

  // Initialize active views
  size_t cv_start = 0, num_cv = 0;
  switch (variablesView.first) {
  case MERGED_DISTINCT_DESIGN:
    num_cv = num_mdv;                                                   break;
  case MERGED_DISTINCT_ALEATORY_UNCERTAIN:
    cv_start = num_mdv;                     num_cv = num_auv;           break;
  case MERGED_DISTINCT_EPISTEMIC_UNCERTAIN:
    cv_start = num_mdv+num_auv;             num_cv = num_euv;           break;
  case MERGED_DISTINCT_UNCERTAIN:
    cv_start = num_mdv;                     num_cv = num_auv + num_euv; break;
  case MERGED_DISTINCT_STATE:
    cv_start = num_mdv + num_auv + num_euv; num_cv = num_msv;           break;
  }
  if (num_cv) {
    continuousLowerBnds = RealDenseVector(Teuchos::View,
      &allContinuousLowerBnds[cv_start], num_cv);
    continuousUpperBnds = RealDenseVector(Teuchos::View,
      &allContinuousUpperBnds[cv_start], num_cv);
  }
}


void MergedConstraints::build_inactive_views()
{
  size_t num_mdv = (*variablesComponents)[0][0],
    num_auv = (*variablesComponents)[1][0],
    num_euv = (*variablesComponents)[2][0],
    num_msv = (*variablesComponents)[3][0];

  // Initialize inactive views
  size_t icv_start = 0, num_icv = 0;
  switch (variablesView.second) {
  case MERGED_DISTINCT_DESIGN:
    num_icv = num_mdv;                                                    break;
  case MERGED_DISTINCT_ALEATORY_UNCERTAIN:
    icv_start = num_mdv;                     num_icv = num_auv;           break;
  case MERGED_DISTINCT_EPISTEMIC_UNCERTAIN:
    icv_start = num_mdv+num_auv;             num_icv = num_euv;           break;
  case MERGED_DISTINCT_UNCERTAIN:
    icv_start = num_mdv;                     num_icv = num_auv + num_euv; break;
  case MERGED_DISTINCT_STATE:
    icv_start = num_mdv + num_auv + num_euv; num_icv = num_msv;           break;
  }
  if (num_icv) {
    inactiveContinuousLowerBnds = RealDenseVector(Teuchos::View,
      &allContinuousLowerBnds[icv_start], num_icv);
    inactiveContinuousUpperBnds = RealDenseVector(Teuchos::View,
      &allContinuousUpperBnds[icv_start], num_icv);
  }
}


void MergedConstraints::read(istream& s)
{
  read_data(s, allContinuousLowerBnds);
  read_data(s, allContinuousUpperBnds);
}


void MergedConstraints::write(ostream& s) const
{
  write_data(s, allContinuousLowerBnds);
  write_data(s, allContinuousUpperBnds);
}

} // namespace Dakota
