/*  _______________________________________________________________________

    DAKOTA: Design Analysis Kit for Optimization and Terascale Applications
    Copyright (c) 2006, Sandia National Laboratories.
    This software is distributed under the GNU General Public License.
    For more information, see the README file in the top Dakota directory.
    _______________________________________________________________________ */

//- Class:        Dakota::Matrix<T>
//- Description:  A class template to create dynamically allocated 2D arrays
//-               of objects of any type.  The matrix is zero-based,
//-               rows: 0 to (numRows-1) and cols: 0 to (numColumns-1).
//-               The class supports overloading of the subscript operator
//-               allowing it to emulate a normal built-in 2D array type.
//-               Based on the 2D matrix implementation from C++ Report, 9/95.
//- Owner:        Bill Bohnhoff
//- Checked by:
//- Version: $Id: DakotaMatrix.H 3989 2006-09-27 13:59:30Z slbrow $

#ifndef DAKOTA_MATRIX_H
#define DAKOTA_MATRIX_H

#include "DakotaBaseVector.H"
#include "MPIPackBuffer.H"


namespace Dakota {

/// Template class for the Dakota numerical matrix

/** A matrix class template to provide 2D arrays of objects. The
    matrix is zero-based, rows: 0 to (numRows-1) and cols: 0 to
    (numColumns-1).  The class supports overloading of the subscript
    operator allowing it to emulate a normal built-in 2D array
    type. Matrix relies on the BaseVector template class to manage any
    differences between underlying DAKOTA_BASE_VECTOR implementations
    (RW, STL, etc.). */

template <class T>
class Matrix: public BaseVector< BaseVector<T> >
{
public:

  //
  //- Heading: Constructor and Destructor
  //

  /// Constructor, takes number of rows, and number of columns as arguments
  Matrix(size_t num_rows=0, size_t num_cols=0);

  /// Destructor
  ~Matrix();

  //
  //- Heading: Operator overloaded members
  //

  /// Sets all elements in the matrix to ival
  Matrix<T>& operator=(const T& ival);

  //
  //- Heading: Inquire functions
  //

  /// Returns the number of rows for the matrix
  size_t num_rows() const;
 
  /// Returns the number of columns for the matrix
  size_t num_columns() const;

  //
  //- Heading: Member Methods
  //

  /// Resizes the matrix to num_rows by num_cols
  void reshape_2d(size_t num_rows, size_t num_cols);

  /// Reads a portion of the Matrix from an input stream
  void read(istream& s, size_t nr, size_t nc);
  /// Reads the complete Matrix from an input stream
  void read(istream& s);
  /// Reads a portion of the ith Matrix row vector from an input stream
  void read_row_vector(istream& s, size_t i, size_t nc);
  /// Reads the ith Matrix row vector from an input stream
  void read_row_vector(istream& s, size_t i);

  /// Writes a portion of the Matrix to an output stream
  void write(ostream& s, size_t nr, size_t nc, bool brackets, bool row_rtn,
	     bool final_rtn) const;
  /// Writes the complete Matrix to an output stream
  void write(ostream& s, bool brackets, bool row_rtn, bool final_rtn) const;
  /// Writes a portion of the ith Matrix row vector to an output stream
  void write_row_vector(ostream& s, size_t i, size_t nc, bool brackets,
			bool break_line, bool final_rtn) const;
  /// Writes the ith Matrix row vector to an output stream
  void write_row_vector(ostream& s, size_t i, bool brackets, bool break_line,
			bool final_rtn) const;

  /// Reads a portion of the Matrix from a binary input stream
  void read(BiStream& s, size_t nr, size_t nc);
  /// Reads the complete Matrix from a binary input stream
  void read(BiStream& s);
  /// Reads a portion of the ith Matrix row vector from a binary input stream
  void read_row_vector(BiStream& s, size_t i, size_t nc);
  /// Reads the ith Matrix row vector from a binary input stream
  void read_row_vector(BiStream& s, size_t i);

  /// Writes a portion of the Matrix to a binary output stream
  void write(BoStream& s, size_t nr, size_t nc) const;
  /// Writes the complete Matrix to a binary output stream
  void write(BoStream& s) const;
  /// Writes a portion of the ith Matrix row vector to a binary output stream
  void write_row_vector(BoStream& s, size_t i, size_t nc) const;
  /// Writes the ith Matrix row vector to a binary output stream
  void write_row_vector(BoStream& s, size_t i) const;

  /// Reads a Matrix from an MPIUnpackBuffer after an MPI receive
  void read(MPIUnpackBuffer& s);
  /// Reads an annotated Matrix from an MPIUnpackBuffer after an MPI receive
  void read_annotated(MPIUnpackBuffer& s);
  /// Reads the ith Matrix row vector from an MPIUnpackBuffer after an MPI recv
  void read_row_vector(MPIUnpackBuffer& s, size_t i);

  /// Writes a Matrix to a MPIPackBuffer prior to an MPI send
  void write(MPIPackBuffer& s) const;
  /// Writes an annotated Matrix to a MPIPackBuffer prior to an MPI send
  void write_annotated(MPIPackBuffer& s) const;
  /// Writes the ith Matrix row vector to a MPIPackBuffer prior to an MPI send
  void write_row_vector(MPIPackBuffer& s, size_t i) const;
};


template <class T>
inline Matrix<T>::Matrix(size_t num_rows, size_t num_cols):
BaseVector<BaseVector<T> >(num_rows)
{
  // Each row is initialized with zero length.
  // Reshape each row to the appropriate number of columns.
  BaseVector<T>* array_ = BaseVector<BaseVector<T> >::array();
  for (size_t i=0; i<num_rows; i++)
    array_[i].reshape(num_cols);
}


template <class T>
inline Matrix<T>::~Matrix()
{ }


template <class T>
inline void Matrix<T>::reshape_2d(size_t num_rows, size_t num_cols)
{
  BaseVector<BaseVector<T> >::reshape(num_rows);
  // reshape the reshaped array
  BaseVector<T>* array_ = BaseVector<BaseVector<T> >::array();
  for (size_t i=0; i<num_rows; i++) 
    array_[i].reshape(num_cols);
}


/** calls base class operator=(ival) */
template <class T>
inline Matrix<T>& Matrix<T>::operator=(const T& val)
{
  size_t num_rows = BaseVector<BaseVector<T> >::size();
  BaseVector<T>* array_ = BaseVector<BaseVector<T> >::array();
  for (register size_t i=0; i<num_rows; i++)
    array_[i] = val; // invokes BaseVector<T>::operator=
  return *this;
}


template <class T>
inline size_t Matrix<T>::num_rows() const
{ return BaseVector<BaseVector<T> >::size(); }


template <class T>
inline size_t Matrix<T>::num_columns() const
{
  return (BaseVector<BaseVector<T> >::empty()) ? 0 :
    BaseVector<BaseVector<T> >::array()->size();
}


template <class T>
void Matrix<T>::read(istream& s, size_t nr, size_t nc)
{
  if (nr > num_rows() || nc > num_columns()) {
    Cerr << "Error: indexing in Matrix<T>::read(istream, nr, nc) out of bounds."
	 << endl;
    abort_handler(-1);
  }
  BaseVector<T>* array_ = BaseVector<BaseVector<T> >::array();
  for (size_t i=0; i<nr; i++)
    for (size_t j=0; j<nc; j++)
      s >> array_[i][j];
}


template <class T>
inline void Matrix<T>::read(istream& s)
{ read(s, num_rows(), num_columns()); }


template <class T>
void Matrix<T>::
read_row_vector(istream& s, size_t i, size_t nc)
{
  if (i >= num_rows() || nc > num_columns()) {
    Cerr << "Error: indexing in Matrix<T>::read_row_vector(istream, i, nc) out "
	 << "of bounds." << endl;
    abort_handler(-1);
  }
  BaseVector<T>* array_ = BaseVector<BaseVector<T> >::array();
  for (size_t j=0; j<nc; j++)
    s >> array_[i][j];
}


template <class T>
inline void Matrix<T>::read_row_vector(istream& s, size_t i)
{ read_row_vector(s, i, num_columns()); }


template <class T>
void Matrix<T>::
write(ostream& s, size_t nr, size_t nc, bool brackets, bool row_rtn,
      bool final_rtn) const
{
  if (nr > num_rows() || nc > num_columns()) {
    Cerr << "Error: indexing in Matrix<T>::write(ostream, nr, nc) out of "
	 << "bounds." << endl;
    abort_handler(-1);
  }
  BaseVector<T>* array_ = BaseVector<BaseVector<T> >::array();
  s.setf(ios::scientific); // formatting optimized for T = Real
  s << setprecision(write_precision);
  if (brackets)
    s << "[[ ";
  for (size_t i=0; i<nr; i++) {
    for (size_t j=0; j<nc; j++)
      s << setw(write_precision+7) << array_[i][j] << ' ';
    // NOTE: newlines on every 4th component (as in the row vector case) 
    // could lead to ambiguity in the matrix case. 
    if (row_rtn && i!=nr-1)
      s << "\n   ";
  }
  if (brackets)
    s << "]] ";
  if (final_rtn)
    s << '\n';
}


template <class T>
inline void Matrix<T>::
write(ostream& s, bool brackets, bool row_rtn, bool final_rtn) const
{ write(s, num_rows(), num_columns(), brackets, row_rtn, final_rtn); }


template <class T>
void Matrix<T>::
write_row_vector(ostream& s, size_t i, size_t nc, bool brackets, 
		 bool break_line, bool final_rtn) const
{
  if (i >= num_rows() || nc > num_columns()) {
    Cerr << "Error: indexing in Matrix<T>::write_row_vector(ostream, i, nc) "
	 << "out of bounds." << endl;
    abort_handler(-1);
  }
  BaseVector<T>* array_ = BaseVector<BaseVector<T> >::array();
  s.setf(ios::scientific); // formatting optimized for T = Real
  s << setprecision(write_precision);
  if (brackets)
    s << " [ ";
  for (size_t j=0; j<nc; j++) {
    s << setw(write_precision+7) << array_[i][j] << ' ';
    if (break_line && (j+1)%4 == 0)
      s << "\n   "; // Output 4 gradient components per line
  }
  if (brackets)
    s << "] ";
  if (final_rtn)
    s << '\n';
}


template <class T>
inline void Matrix<T>::
write_row_vector(ostream& s, size_t i, bool brackets, bool break_line,
		 bool final_rtn) const
{ write_row_vector(s, i, num_columns(), brackets, break_line, final_rtn); }


/// global istream extraction operator for Matrix
template <class T>
inline istream& operator>>(istream& s, Matrix<T>& data)
{ data.read(s); return s; }


/// global ostream insertion operator for Matrix
template <class T>
inline ostream& operator<<(ostream& s, const Matrix<T>& data)
{ data.write(s, true, true, true); return s; }


template <class T>
void Matrix<T>::read(BiStream& s, size_t nr, size_t nc)
{
  if (nr > num_rows() || nc > num_columns()) {
    Cerr << "Error: indexing in Matrix<T>::read(BiStream, nr, nc) out of "
	 << "bounds." << endl;
    abort_handler(-1);
  }
  BaseVector<T>* array_ = BaseVector<BaseVector<T> >::array();
  for (size_t i=0; i<nr; i++)
    for (size_t j=0; j<nc; j++)
      s >> array_[i][j];
}


template <class T>
inline void Matrix<T>::read(BiStream& s)
{ read(s, num_rows(), num_columns()); }


template <class T>
void Matrix<T>::write(BoStream& s, size_t nr, size_t nc) const
{
  if (nr > num_rows() || nc > num_columns()) {
    Cerr << "Error: indexing in Matrix<T>::write(BoStream, nr, nc) out of "
	 << "bounds." << endl;
    abort_handler(-1);
  }
  BaseVector<T>* array_ = BaseVector<BaseVector<T> >::array();
  for (size_t i=0; i<nr; i++)
    for (size_t j=0; j<nc; j++)
      s << array_[i][j];
}


template <class T>
inline void Matrix<T>::write(BoStream& s) const
{ write(s, num_rows(), num_columns()); }


template <class T>
void Matrix<T>::read_row_vector(BiStream& s, size_t i, size_t nc)
{
  if (i >= num_rows() || nc > num_columns()) {
    Cerr << "Error: indexing in Matrix<T>::read_row_vector(BiStream, i, nc) "
	 << "out of bounds." << endl;
    abort_handler(-1);
  }
  BaseVector<T>* array_ = BaseVector<BaseVector<T> >::array();
  for (size_t j=0; j<nc; j++)
    s >> array_[i][j];
}


template <class T>
inline void Matrix<T>::read_row_vector(BiStream& s, size_t i)
{ read_row_vector(s, i, num_columns()); }


template <class T>
void Matrix<T>::write_row_vector(BoStream& s, size_t i, size_t nc) const
{
  if (i >= num_rows() || nc > num_columns()) {
    Cerr << "Error: indexing in Matrix<T>::write_row_vector(BoStream, i, nc) "
	 << "out of bounds." << endl;
    abort_handler(-1);
  }
  BaseVector<T>* array_ = BaseVector<BaseVector<T> >::array();
  for (size_t j=0; j<nc; j++)
    s << array_[i][j];
}


template <class T>
inline void Matrix<T>::write_row_vector(BoStream& s, size_t i) const
{ write_row_vector(s, i, num_columns()); }


template <class T>
void Matrix<T>::read(MPIUnpackBuffer& s)
{
  size_t n = num_rows(), m = num_columns();
  BaseVector<T>* array_ = BaseVector<BaseVector<T> >::array();
  for (size_t i=0; i<n; i++)
    for (size_t j=0; j<m; j++)
      s >> array_[i][j];
}


template <class T>
void Matrix<T>::write(MPIPackBuffer& s) const
{
  size_t n = num_rows(), m = num_columns();
  BaseVector<T>* array_ = BaseVector<BaseVector<T> >::array();
  for (size_t i=0; i<n; i++)
    for (size_t j=0; j<m; j++)
      s << array_[i][j];
}


template <class T>
void Matrix<T>::read_row_vector(MPIUnpackBuffer& s, size_t i)
{
  if (i >= num_rows()) {
    Cerr << "Error: indexing in Matrix<T>::read_row_vector(MPIUnpackBuffer, i) "
	 << "out of bounds." << endl;
    abort_handler(-1);
  }
  size_t m = num_columns();
  BaseVector<T>* array_ = BaseVector<BaseVector<T> >::array();
  for (size_t j=0; j<m; j++)
    s >> array_[i][j];
}


template <class T>
void Matrix<T>::write_row_vector(MPIPackBuffer& s, size_t i) const
{
  if (i >= num_rows()) {
    Cerr << "Error: indexing in Matrix<T>::write_row_vector(MPIPackBuffer, i) "
	 << "out of bounds." << endl;
    abort_handler(-1);
  }
  size_t m = num_columns();
  BaseVector<T>* array_ = BaseVector<BaseVector<T> >::array();
  for (size_t j=0; j<m; j++)
    s << array_[i][j];
}


template <class T>
void Matrix<T>::read_annotated(MPIUnpackBuffer& s)
{
  size_t n, m;
  s >> n >> m;
  if (n != num_rows() || m != num_columns())
    reshape_2d(n, m);
  BaseVector<T>* array_ = BaseVector<BaseVector<T> >::array();
  for (size_t i=0; i<n; i++)
    for (size_t j=0; j<m; j++)
      s >> array_[i][j];
}


template <class T>
void Matrix<T>::write_annotated(MPIPackBuffer& s) const
{
  size_t n = num_rows(), m = num_columns();
  s << n << m;
  BaseVector<T>* array_ = BaseVector<BaseVector<T> >::array();
  for (size_t i=0; i<n; i++)
    for (size_t j=0; j<m; j++)
      s << array_[i][j];
}


/// global MPIUnpackBuffer extraction operator for Matrix
template <class T>
inline MPIUnpackBuffer& operator>>(MPIUnpackBuffer& s, Matrix<T>& data)
{ data.read_annotated(s); return s; }


/// global MPIPackBuffer insertion operator for Matrix
template <class T>
inline MPIPackBuffer& operator<<(MPIPackBuffer& s, const Matrix<T>& data)
{ data.write_annotated(s); return s; }

} // namespace Dakota

#endif
