/*  _______________________________________________________________________

    DAKOTA: Design Analysis Kit for Optimization and Terascale Applications
    Copyright (c) 2006, Sandia National Laboratories.
    This software is distributed under the GNU General Public License.
    For more information, see the README file in the top Dakota directory.
    _______________________________________________________________________ */

//- Class:        OrthogPolyApproximation
//- Description:  Implementation code for OrthogPolyApproximation class
//-               
//- Owner:        Mike Eldred

#include "BasisPolyApproximation.H"
//#include "BasisPolynomial.H"
//#include "NonDQuadrature.H"
#include "NonDSparseGrid.H"
#include "ProblemDescDB.H"


namespace Dakota {

BasisPolyApproximation::
BasisPolyApproximation(const ProblemDescDB& problem_db, const size_t& num_acv):
  Approximation(BaseConstructor(), problem_db, num_acv),
  expCoeffsSolnApproach(SAMPLING), expansionCoeffFlag(true),
  expansionGradFlag(false), integrationRep(NULL)
{ }


/** Return the number of terms in a tensor-product expansion.  For
    isotropic and anisotropic expansion orders, calculation of the
    number of expansion terms is straightforward: Prod(p_i + 1). */
size_t BasisPolyApproximation::
tensor_product_terms(const UShortArray& order, bool exp_order_offset)
{
  size_t i, n = order.length(), num_terms = 1;
  if (exp_order_offset)
    for (i=0; i<n; ++i)
      num_terms *= order[i] + 1; // PCE: expansion order p
  else
    for (i=0; i<n; ++i)
      num_terms *= order[i];     // SC: quadrature order m (default)

  return num_terms;
}


void BasisPolyApproximation::
tensor_product_multi_index(const UShortArray& order,
			   UShort2DArray& multi_index, bool exp_order_offset)
{
  size_t i, n = order.length(), mi_len = multi_index.length();
  if (!mi_len) { // could be more robust; avoid tensor_product_terms() if sized
    mi_len = tensor_product_terms(order, exp_order_offset);
    multi_index.reshape(mi_len);
  }
  UShortArray indices(n, 0);
  for (i=0; i<mi_len; ++i) {
    multi_index[i] = indices;
    // increment the n-dimensional index set
    increment_indices(indices, order, !exp_order_offset);
  }
}


/** Return the number of terms in a total-order expansion.  For
    isotropic expansion order, the number of expansion terms is simply
    (n+p)!/(n!p!). For anisotropic expansion order, no simple
    expression is currently available and the number of expansion
    terms is computed by brute force using the multiIndex recursion. */
size_t BasisPolyApproximation::
total_order_terms(const UShortArray& upper_bound, int lower_bound_offset)
{
  size_t i, n = upper_bound.length(), num_terms;
  if (!n) {
    Cerr << "Error: empty upper_bound in BasisPolyApproximation::"
	 << "total_order_terms()." << endl;
    abort_handler(-1);
  }

  bool mi_lower_bound = (lower_bound_offset >= 0); // default is -1

  bool isotropic = true;
  unsigned short order = upper_bound[0];
  for (i=1; i<n; ++i)
    if (upper_bound[i] != order)
      { isotropic = false; break; }

  if (isotropic) {
    //  (n+p)!/(n!p!)                         if no lower bound (PCE)
    // ((n+p)!/p! - (n+omit_p)!/(omit_p)!)/n! if lower bound (Smolyak)
    num_terms = (size_t)BasisPolynomial::factorial_ratio(order+n, order);
    if (mi_lower_bound) {
      int omit_order = order - lower_bound_offset - 1;
      if (omit_order >= 0)
	num_terms -=
	  (size_t)BasisPolynomial::factorial_ratio(omit_order+n, omit_order);
    }
    num_terms /= (size_t)BasisPolynomial::factorial(n);
  }
  else { // anisotropic: use multiIndex recursion to compute
    if (mi_lower_bound) {
      // Smolyak combinatorial form is invalid for anisotropic levels
      Cerr << "Error: anisotropic orders not currently supported with "
	   << "multi-index lower bound\n       in BasisPolyApproximation::"
	   << "total_order_terms()." << endl;
      abort_handler(-1);
    }
    unsigned short max_order = order, order_nd;
    for (i=1; i<n; ++i)
      max_order = max(max_order, upper_bound[i]);
    num_terms = 1;      // order 0
    if (max_order >= 1) // order 1
      for (i=0; i<n; ++i)
	if (upper_bound[i] >= 1) // only upper bound may be anisotropic
	  num_terms++;
    for (order_nd=2; order_nd<=max_order; ++order_nd) { // order 2 through max
      UShortArray terms(order_nd, 1); // # of terms = current order
      bool order_complete = false;
      while (!order_complete) {
	size_t last_index = order_nd - 1, prev_index = order_nd - 2;
	for (terms[last_index]=1; terms[last_index]<=terms[prev_index]; 
	     ++terms[last_index]) {
	  bool include = true;
	  for (i=0; i<n; ++i) {
	    if (std::count(terms.begin(), terms.end(), i+1) > upper_bound[i]) {
	      // only upper bound may be anisotropic
	      include = false;
	      break;
	    }
	  }
	  if (include)
	    num_terms++;
	}
	// increment term hierarchy
	increment_terms(terms, last_index, prev_index, n, order_complete);
      }
    }
  }

  return num_terms;
}


void BasisPolyApproximation::
total_order_multi_index(const UShortArray& upper_bound,
			UShort2DArray& multi_index, int lower_bound_offset)
{
  // populate multi_index: implementation follows ordering of Eq. 4.1 in
  // [Xiu and Karniadakis, 2002].
  // To handle anisotropy, we currently perform a complete total-order
  // recursion based on max_order and reject multi-indices with components
  // greater than those in upper_bound.
  size_t i, cntr = 0, n = upper_bound.length(), mi_len = multi_index.length();
  if (!n) {
    Cerr << "Error: empty upper_bound in BasisPolyApproximation::"
	 << "total_order_multi_index()." << endl;
    abort_handler(-1);
  }
  if (!mi_len) { // could be more robust, but avoid total_order_terms() if sized
    mi_len = total_order_terms(upper_bound, lower_bound_offset);
    multi_index.reshape(mi_len);
  }

  bool mi_lower_bound = (lower_bound_offset >= 0); // default is -1

  bool isotropic = true;
  unsigned short order = upper_bound[0];
  for (i=1; i<n; ++i)
    if (upper_bound[i] != order)
      { isotropic = false; break; }

  unsigned short max_order = order, min_order = 0, order_nd;
  if (isotropic) {
    if (mi_lower_bound)
      min_order = (lower_bound_offset >= order)
	? 0 : order - lower_bound_offset; // e.g., Smolyak l.b. = w-N+1
  }
  else {
    if (mi_lower_bound) {
      // Smolyak combinatorial form is invalid for anisotropic levels
      Cerr << "Error: anisotropic orders not currently supported with "
	   << "multi-index lower bound\n       in BasisPolyApproximation::"
	   << "total_order_multi_index()." << endl;
      abort_handler(-1);
    }
    for (i=1; i<n; ++i)
      max_order = max(max_order, upper_bound[i]);
  }

  // special logic required for order_nd < 2 due to prev_index defn below
  if (min_order == 0) { // && max_order >= 0
    multi_index[cntr].reshape(n);
    multi_index[cntr++] = 0; // order 0
  }
  if (min_order <= 1 && max_order >= 1) { // order 1
    for (i=0; i<n && cntr<mi_len; ++i) {
      if (upper_bound[i] >= 1) { // only upper bound may be anisotropic
	multi_index[cntr].reshape(n);
	multi_index[cntr]    = 0; // initialize
	multi_index[cntr][i] = 1; // ith entry is nonzero
	++cntr;
      }
    }
  }
  for (order_nd=max(min_order,(unsigned short)2);
       order_nd<=max_order; ++order_nd) {
    UShortArray mi(n), terms(order_nd, 1);//# of terms = current order
    bool order_complete = false;
    while (!order_complete) {
      // this is the inner-most loop w/i the nested looping managed by terms
      size_t last_index = order_nd - 1, prev_index = order_nd - 2;
      for (terms[last_index]=1;
	   terms[last_index]<=terms[prev_index] && cntr<mi_len;
	   ++terms[last_index]) {
	// store the orders of the univariate polynomials to be used for
	// constructing the current multivariate basis function
	bool include = true;
	for (i=0; i<n; ++i) {
	  mi[i] = std::count(terms.begin(), terms.end(), i+1);
	  if (mi[i] > upper_bound[i]) { // only upper bound may be anisotropic
	    include = false;
	    break;
	  }
	}
	if (include)
	  multi_index[cntr++] = mi;
#ifdef DEBUG
	Cout << "terms:\n" << terms << endl;
#endif // DEBUG
      }
      if (cntr == mi_len)
	order_complete = true;
      else // increment term hierarchy
	increment_terms(terms, last_index, prev_index, n, order_complete);
    }
  }

#ifdef DEBUG
  for (i=0; i<mi_len; ++i)
    Cout << "multiIndex[" << i << "]:\n" << multi_index[i] << endl;
#endif // DEBUG
}

} // namespace Dakota
